"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetSingletonRole = void 0;
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
/**
 * Role which will be reused across asset jobs
 *
 * Has some '*' resources to save IAM policy space, and will not
 * actually add policies that look like policies that were already added.
 */
class AssetSingletonRole extends iam.Role {
    constructor(scope, id, props) {
        super(scope, id, props);
        this._rejectDuplicates = false;
        // Logging permissions
        this.addToPolicy(new iam.PolicyStatement({
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '/aws/codebuild/*',
                })],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        this.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
                'codebuild:BatchPutCodeCoverages',
            ],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'report-group',
                    resourceName: '*',
                })],
        }));
        // CodeBuild start/stop
        this.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        this._rejectDuplicates = true;
    }
    addToPrincipalPolicy(statement) {
        const json = statement.toStatementJson();
        const acts = JSON.stringify(json.Action);
        // These have already been added with wildcard resources on creation
        const alreadyAdded = [
            '["logs:CreateLogGroup","logs:CreateLogStream","logs:PutLogEvents"]',
            '["codebuild:CreateReportGroup","codebuild:CreateReport","codebuild:UpdateReport","codebuild:BatchPutTestCases","codebuild:BatchPutCodeCoverages"]',
            '["codebuild:BatchGetBuilds","codebuild:StartBuild","codebuild:StopBuild"]',
        ];
        if (this._rejectDuplicates && alreadyAdded.includes(acts)) {
            // Pretend we did it
            return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
        }
        // These are added in duplicate (specifically these come from
        // Project#bindToCodePipeline) -- the original singleton asset role didn't
        // have these, and they're not necessary either, so in order to not cause
        // unnecessary diffs, recognize and drop them there as well.
        if (acts === '["kms:Decrypt","kms:Encrypt","kms:ReEncrypt*","kms:GenerateDataKey*"]') {
            // Pretend we did it
            return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
        }
        return super.addToPrincipalPolicy(statement);
    }
}
exports.AssetSingletonRole = AssetSingletonRole;
//# sourceMappingURL=data:application/json;base64,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