"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.preferredCliVersion = void 0;
const fs = require("fs");
const path = require("path");
/**
 * Return the preferred CLI version for the current CDK Library version
 *
 * This is necessary to prevent cxapi version incompatibility between the two
 * CDK major versions. Since changes currently go into v1 before they go into
 * v2, a cxapi change can be released in v1 while the v2 CLI doesn't support it
 * yet.
 *
 * In those cases, simply installing the "latest" CLI (2) is not good enough
 * because it won't be able to read the Cloud Assembly of the v1 app.
 *
 * Find this version by finding the containing `package.json` and reading
 * `preferredCdkCliVersion` from it.
 */
function preferredCliVersion() {
    const pjLocation = findUp('package.json', __dirname);
    if (!pjLocation) {
        return undefined;
    }
    const pj = JSON.parse(fs.readFileSync(pjLocation, { encoding: 'utf-8' }));
    return pj.preferredCdkCliVersion ? `${pj.preferredCdkCliVersion}` : undefined;
}
exports.preferredCliVersion = preferredCliVersion;
function findUp(name, directory) {
    const absoluteDirectory = path.resolve(directory);
    const file = path.join(directory, name);
    if (fs.existsSync(file)) {
        return file;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory == root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
//# sourceMappingURL=data:application/json;base64,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