from __future__ import annotations

import datetime
from pathlib import Path
from types import NoneType
from typing import TYPE_CHECKING, Any

import pytest

from bear_dereth.exceptions import ObjectTypeError
from bear_dereth.typing_tools import (
    TypeHint,
    a_or_b,
    check_for_conflicts,
    coerce_to_type,
    format_default_value,
    infer_type,
    is_array_like,
    is_json_like,
    is_mapping,
    is_object,
    mapping_to_type,
    num_type_params,
    str_to_bool,
    str_to_type,
    type_param,
    type_to_str,
    validate_type,
)

if TYPE_CHECKING:
    from collections.abc import Callable


class Single[T]: ...


class Dual[T, U]: ...


class SingleInt(Single[int]): ...


class DualStrFloat(Dual[str, float]): ...


class SingleNone(Single[None]): ...


class NonGeneric: ...


def test_num_type_params_counts_generic_arguments() -> None:
    assert num_type_params(SingleInt) == 1
    assert num_type_params(DualStrFloat) == 2


def test_num_type_params_raises_for_non_generic() -> None:
    with pytest.raises(TypeError):
        num_type_params(NonGeneric)


def test_type_param_retrieves_requested_argument() -> None:
    assert type_param(SingleInt) is int
    assert type_param(DualStrFloat, 1) is float


def test_type_param_errors_for_invalid_indices_and_none_values() -> None:
    with pytest.raises(IndexError):
        type_param(SingleInt, 5)
    with pytest.raises(TypeError):
        type_param(SingleNone)
    with pytest.raises(TypeError):
        type_param(NonGeneric)


def test_mapping_to_type_returns_converted_values_and_defaults() -> None:
    data: dict[str, str] = {"count": "3"}
    assert mapping_to_type(data, "count", int) == 3
    assert mapping_to_type(data, "missing", int, default="5") == 5


def test_mapping_to_type_raises_for_missing_key_and_bad_coercion() -> None:
    with pytest.raises(KeyError):
        mapping_to_type({}, "value", int)
    with pytest.raises(ValueError, match="Cannot coerce value"):
        mapping_to_type({"value": "abc"}, "value", int)


def test_validate_type_accepts_matches_and_raises_for_mismatches() -> None:
    validate_type(10, int)
    with pytest.raises(TypeError):
        validate_type("abc", int)
    with pytest.raises(ObjectTypeError):
        validate_type("abc", int, exception=ObjectTypeError)


def test_type_hint_returns_runtime_stub_class() -> None:
    hinted = TypeHint(list)
    assert isinstance(hinted, type)
    assert hinted is not list
    assert issubclass(hinted, object)
    assert hinted() is not None


def test_json_and_array_like_guards() -> None:
    assert is_json_like({})
    assert is_json_like([])
    assert not is_json_like(set())

    assert is_array_like([1, 2])
    assert is_array_like((1, 2))
    assert is_array_like({1, 2})
    assert not is_array_like({"a": 1})


class FauxMapping:
    def __init__(self) -> None:
        """A simple mapping-like class."""
        self._store: dict[str, int] = {}

    def __getitem__(self, key: str) -> int:
        return self._store[key]

    def __setitem__(self, key: str, value: int) -> None:
        self._store[key] = value


class PlainObject:
    def __init__(self) -> None:
        """A simple object with attributes."""
        self.value = 42


def test_mapping_and_object_detection_helpers() -> None:
    assert is_mapping({"a": 1})
    assert is_mapping(FauxMapping())
    assert not is_mapping(PlainObject())

    assert is_object(PlainObject())
    assert not is_object({"a": 1})
    assert not is_object(10)


def test_a_or_b_dispatches_to_mapping_or_object_handlers() -> None:
    calls: list[str] = []

    def handle_mapping(doc: object) -> None:
        calls.append("mapping")

    def handle_object(doc: object) -> None:
        calls.append("object")

    handler: Callable[..., None] = a_or_b(handle_mapping, handle_object)
    handler({"a": 1})
    handler(PlainObject())
    handler(5)

    assert calls == ["mapping", "object"]


def test_str_to_bool_handles_truthy_strings() -> None:
    assert str_to_bool("True")
    assert str_to_bool(" yes ")
    assert not str_to_bool("false")


def test_coerce_to_type_success_and_failure_cases() -> None:
    assert coerce_to_type("5", int) == 5
    with pytest.raises(ValueError, match="Cannot coerce"):
        coerce_to_type("bad", int)


def test_infer_type_identifies_known_types(tmp_path: Path) -> None:
    assert infer_type("[1, 2]") == "list"
    assert infer_type("(1, 2)") == "tuple"
    assert infer_type("{'a': 1}") == "dict"
    assert infer_type(datetime) == "Any"
    assert infer_type("{1, 2}") == "set"
    assert infer_type("b'bytes'") == "bytes"
    assert infer_type("None") == "None"
    assert infer_type("true") == "bool"
    assert infer_type("3") == "int"
    assert infer_type("3.5") == "float"

    file_path = tmp_path / "demo.txt"
    file_path.write_text("x")
    assert infer_type(file_path) == "Path"

    assert infer_type("hello") == "str"
    assert infer_type(object()) == "Any"


def test_str_to_type_supports_known_entries_and_defaults() -> None:
    assert str_to_type("int") is int
    assert str_to_type("FLOAT") is float
    assert str_to_type("unknown", default=list) is list


def test_type_to_str_converts_supported_types_and_handles_arbitrary() -> None:
    assert type_to_str(int) == "int"
    assert type_to_str(Path) == "path"
    assert type_to_str(datetime) == "datetime"
    with pytest.raises(TypeError):
        type_to_str(complex)

    class Custom: ...

    assert type_to_str(Custom, arb_types_allowed=True) == "Any"


def test_str_to_type() -> None:
    type_map: dict[str, Any] = {
        "EpochTimestamp": int,
        "datetime": str,
        "str": str,
        "int": int,
        "float": float,
        "bool": bool,
        "list": list,
        "dict": dict,
        "tuple": tuple,
        "path": Path,
        "bytes": bytes,
        "set": set,
        "frozenset": frozenset,
        "none": NoneType,
        "nonetype": NoneType,
        "any": Any,
        "object": str,
    }

    for key, val in type_map.items():
        assert str_to_type(key, str) == val


def test_check_for_conflicts_uses_fallbacks_and_modifiers() -> None:
    assert check_for_conflicts("class") == "class_"
    assert check_for_conflicts("len") == "len_"
    assert check_for_conflicts("async", modifier=lambda name: f"{name}X") == "asyncX"


def test_format_default_value_formats_common_types() -> None:
    assert format_default_value("value") == '"value"'
    assert format_default_value(value=True) == "True"
    assert format_default_value(3) == "3"
    assert format_default_value(3.14) == "3.14"
    assert format_default_value([1, 2, 3]) == "[1, 2, 3]"
