# pytest-cream

**The cream of test execution** - smooth pytest workflows with intelligent orchestration.

pytest-cream is a Python package that revolutionizes test execution by combining intelligent orchestration, duration-based splitting, and parallel execution for optimal performance.

## Why pytest-cream?

### 🚀 Intelligent Test Orchestration
Unlike traditional test runners, pytest-cream analyzes test durations and automatically:
- **Splits tests** into long-running (sequential) and short-running (parallel) buckets
- **Optimizes execution** by running short tests in parallel while long tests run sequentially
- **Adapts to your suite** with customizable thresholds and worker counts

### ⚡ Development-Optimized Workflows
Speed up your development cycle:
- **`--short-only`**: Skip slow tests during development for instant feedback
- **`--limit N`**: Run just N tests for quick validation
- **`--exclude-tests`**: Exclude expensive integration or E2E tests
- **Smart defaults**: Auto-detect workspace and Python environments

### 🎯 Performance Tuning at Your Fingertips
Fine-tune execution for your environment:
- **`--threshold`**: Adjust the boundary between long/short tests (default: 0.5s)
- **`--workers`**: Scale parallelization to match your CPU cores
- **Duration profiling**: Collect and analyze test performance data
- **Selective execution**: Run exactly what you need, when you need it

## Features
- **Intelligent orchestration** with duration-based test splitting
- **Selective execution** with `--short-only`, `--limit`, and `--exclude-tests`
- **Parallel execution** with configurable worker counts
- **Automatic workspace detection** (no more complex path management)
- Fetch tests from remote repositories or local directories
- Advanced dependency management with custom installation commands
- **Uses uv by default** for faster and more reliable installations
- Support for uv, pip, poetry, and other package managers
- Python version control for project compatibility
- Environment isolation for complex projects (Rust extensions, etc.)

## Installation

```bash
pip install pytest-cream
```

## Usage

### Quick Start (Recommended)
The easiest way to get started is with the `init` command, which sets up everything automatically:

```bash
# Step 1: Initialize workspace (fetches tests and installs dependencies)
# Uses current directory by default (with safety check if not empty)
pytest-cream init --repo owner/project

# Step 2: Profile tests to collect duration statistics
pytest-cream profile

# Step 3: Run tests with intelligent orchestration
pytest-cream run --threshold 0.5 --workers 4
```

**Safety Note:** If the current directory is not empty, pytest-cream will ask for confirmation before proceeding. To avoid this prompt, use `--ws` to specify a dedicated workspace:

```bash
pytest-cream init --repo owner/project --ws ~/pytest-cream-tests
```

**Key Features:**
- 🚀 **Intelligent Orchestration**: Long tests run sequentially, short tests run in parallel
- ⚡ **Performance Tuning**: Adjust `--threshold` to optimize test distribution
- 🎯 **Selective Execution**: Use `--short-only` to skip slow tests during development
- 🔧 **Flexible Control**: Limit test count, exclude patterns, customize workers
- 📁 **Zero Configuration**: Uses current directory as workspace by default
- 🛡️ **Safety First**: Warns when current directory is not empty

**That's it!** The workspace path is automatically saved to `.pytest-cream.json` after `init`, so `profile` and `run` commands work without any additional flags.

### Advanced Orchestration Examples

```bash
# Run only fast tests during development (skip long tests)
pytest-cream run --short-only

# Run with custom threshold and parallelization
pytest-cream run --threshold 1.0 --workers 8

# Limit test execution for quick checks
pytest-cream run --limit 20

# Exclude specific test files or patterns
pytest-cream run --exclude-tests "test_integration/,test_slow.py"

# Combine multiple options
pytest-cream run --short-only --limit 50 --workers 8 --threshold 0.3

# Fine-tune for CI pipelines
pytest-cream run --threshold 2.0 --workers 16 --exclude-tests "test_e2e/"
```

### Real-World Scenarios

```bash
# Scenario 1: Quick feedback during development
pytest-cream profile
pytest-cream run --short-only --limit 30
# ✓ Runs only fast tests, max 30 tests, instant feedback

# Scenario 2: Pre-commit validation
pytest-cream run --threshold 0.5 --exclude-tests "test_integration/"
# ✓ Skips integration tests, optimized parallelization

# Scenario 3: CI pipeline - full test suite
pytest-cream run --threshold 1.0 --workers 16
# ✓ Optimal distribution for CI environment

# Scenario 4: Nightly builds - thorough testing
pytest-cream run --threshold 2.0
# ✓ Runs all tests with proper long/short splitting
```

### Step-by-Step Workflow (Manual Control)
For more control, you can still specify paths explicitly:

```bash
# Step 1: Fetch tests from repository
pytest-cream fetch --repo owner/project --branch main --ws ~/my-workspace

# Step 2: Install dependencies manually
cd ~/my-workspace/owner_project_main
pip install -e .  # or your preferred installation method

# Step 3: Profile tests
pytest-cream profile --ws ~/my-workspace/owner_project_main --python python3.11

# Step 4: Run tests
pytest-cream run --ws ~/my-workspace/owner_project_main --python python3.11

# (Optional) Step 5: Filter tests by duration threshold for analysis
pytest-cream filter --ws ~/my-workspace/owner_project_main --threshold 0.5
```

### Basic Usage Examples
```bash
# Simple initialization (uses uv by default for faster installs)
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project --install-mode editable \
  --ws ~/workspace
pytest-cream profile
pytest-cream run --threshold 0.5

# Fast iteration during development - only run quick tests
pytest-cream init --repo owner/project --ws ~/workspace
pytest-cream profile
pytest-cream run --short-only --workers 8

# Use pip instead of uv (if needed for compatibility)
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project \
  --install-mode editable \
  --use-pip \
  --ws ~/workspace
pytest-cream profile
pytest-cream run --threshold 1.0 --workers 4

# Use custom installation commands for complex projects
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project \
  --install-cmd "uv sync --extra cpu" \
  --ws ~/workspace
pytest-cream profile
pytest-cream run --threshold 0.5 --exclude-tests "test_e2e/"
```

### Advanced Usage Examples
```bash
# Multiple installation commands with fallback
pytest-cream init --repo owner/complex-project --branch main \
  --install-repo owner/complex-project \
  --install-cmd "uv sync --extra gpu; pip install torch" \
  --install-fallback \
  --ws ~/workspace

# Profile and run with intelligent orchestration
pytest-cream profile
pytest-cream run --threshold 1.5 --workers 12 \
  --exclude-tests "test_slow.py,test_integration/" \
  --limit 100

# Override auto-detected values when needed
pytest-cream profile --ws ~/custom_workspace --python /custom/python
pytest-cream run --ws ~/custom_workspace --python /custom/python \
  --short-only --workers 16
```

## Command Line Options

### Core Commands

#### `init` - Initialize Workspace
Fetch tests from a repository and optionally install dependencies. This is the first step in the workflow.

```bash
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project --install-mode editable \
  --ws ~/workspace
```

**Note:** After running `init`, you need to run `profile` and then `run` to actually execute tests.

#### `fetch` - Fetch Tests
Clone a repository to a workspace for testing.

```bash
pytest-cream fetch --repo owner/project --branch main --ws ~/workspace
```

**Required Parameters:**
- `--repo`: Repository to fetch (format: 'owner/repo' or GitHub URL)
- `--ws` / `--workspace`: Directory where the repository will be cloned

**Optional Parameters:**
- `--branch`: Git branch to checkout (default: main)

**Output:** Creates a directory in workspace named `owner_project_branch/` containing the cloned repository.

#### `profile` - Profile Tests
Run all tests once to collect duration statistics. This is essential before using the `run` command.

```bash
# Simple (uses saved workspace from init)
pytest-cream profile

# Or with explicit workspace
pytest-cream profile --ws ~/workspace/owner_project_main --python python3.11
```

**Optional Parameters:**
- `--ws` / `--workspace`: Workspace directory containing the tests (auto-detected from last init if not provided)
- `--python`: Python executable to use (auto-detected from workspace .venv if not provided)
- `--tests-dir`: Subdirectory containing tests (default: tests)
- `--output`: Output filename for duration log (default: test_durations.log)

**Output:** Creates `test_durations.log` in the workspace directory.

#### `filter` - Filter Tests by Duration
Parse duration logs and categorize tests as long/short.

```bash
pytest-cream filter --ws ~/workspace/owner_project_main --threshold 0.5
```

**Required Parameters:**
- `--ws` / `--workspace`: Workspace directory containing duration logs
- `--threshold`: Duration threshold in seconds

**Optional Parameters:**
- `--input`: Input duration log filename (default: test_durations.log)
- `--output`: Output filename for filtered results (default: filtered_tests.txt)

**Output:** Creates `filtered_tests.txt` in the workspace directory.

#### `run` - Run Tests with Orchestration
Execute tests with intelligent orchestration based on duration analysis. Long tests run sequentially, short tests run in parallel.

```bash
# Simple (uses saved workspace from init)
pytest-cream run

# With intelligent orchestration options (recommended)
pytest-cream run --threshold 0.5 --workers 8

# Development mode - only fast tests
pytest-cream run --short-only --limit 50

# Or with explicit workspace and full control
pytest-cream run --ws ~/workspace/owner_project_main \
  --python python3.11 \
  --threshold 1.0 \
  --workers 16 \
  --exclude-tests "test_integration/,test_slow.py"
```

**Optional Parameters:**
- `--ws` / `--workspace`: Workspace directory containing the tests and duration logs (auto-detected from last init if not provided)
- `--python`: Python executable to use (auto-detected from workspace .venv if not provided)
- `--durations`: Duration log file (default: test_durations.log)
- `--threshold`: Duration threshold in seconds for splitting long/short tests (default: 0.5). Tests >= threshold run sequentially, tests < threshold run in parallel
- `--workers`: Number of parallel workers for short tests (default: 4). Higher values = more parallelization
- `--short-only`: Run only short tests (skip long tests). Perfect for quick development feedback
- `--limit`: Maximum number of tests to run. Useful for sampling or quick validation
- `--exclude-tests`: Comma-separated patterns of test files/directories to exclude (e.g., "test_e2e/,test_slow.py")

**Performance Tips:**
- **Development**: Use `--short-only` to get instant feedback
- **CI**: Increase `--workers` to match available CPU cores
- **Fine-tuning**: Adjust `--threshold` based on your test suite profile
- **Selective**: Use `--exclude-tests` to skip expensive integration tests

**Output:** Runs tests with intelligent orchestration and displays results.

### `init` Options

For the workspace initialization command, these options are available:

**Basic Options:**
- `--repo`: Repository to fetch (owner/repo or URL) - **Required**
- `--branch`: Branch to checkout (default: main)
- `--ws` / `--workspace`: Directory to use as workspace - **Required**

**Installation Options:**
- `--install-repo`: Repository to install dependencies from (same as `--repo` typically)
- `--install-branch`: Branch to install from (default: same as `--branch`)
- `--install-mode`: How to install (`pip`, `editable`, `wheel`) - default: pip
- `--use-pip`: Use pip instead of uv (by default, uv is used for faster installations)
- `--install-cmd`: Custom install command (overrides `--install-mode`)
- `--install-fallback`: Fallback to standard installation if custom command fails
- `--python`: Python executable for installation (e.g., `python3.11`)

## Intelligent Orchestration Deep Dive

### How It Works

1. **Profiling Phase** (`pytest-cream profile`):
   - Runs all tests once with `--durations=0`
   - Collects execution time for each test
   - Saves to `test_durations.log`

2. **Orchestration Phase** (`pytest-cream run`):
   - Reads duration data
   - Splits tests based on `--threshold` (default: 0.5s)
   - Long tests (>= threshold): Run sequentially (1 worker)
   - Short tests (< threshold): Run in parallel (N workers)

### Key Parameters Explained

#### `--threshold` (Duration Splitting)
Controls the boundary between "long" and "short" tests.

```bash
# Conservative (more parallel tests)
pytest-cream run --threshold 0.3
# Tests < 0.3s run in parallel, >= 0.3s run sequentially

# Balanced (default)
pytest-cream run --threshold 0.5

# Aggressive (fewer parallel tests, safer)
pytest-cream run --threshold 2.0
# Only tests < 2s run in parallel
```

**When to adjust:**
- Low threshold (0.1-0.3): Fast, simple tests with no shared resources
- Medium threshold (0.5-1.0): Mixed test suite (default)
- High threshold (2.0+): Tests with complex setup, shared resources, or flaky parallel behavior

#### `--workers` (Parallelization Level)
Number of parallel processes for short tests.

```bash
# Light parallelization
pytest-cream run --workers 2

# Match CPU cores (recommended for CI)
pytest-cream run --workers $(nproc)

# Heavy parallelization
pytest-cream run --workers 16
```

**When to adjust:**
- Local development: 4-8 workers
- CI with 8 cores: 8 workers
- CI with 32 cores: 16-32 workers
- Memory-intensive tests: Reduce workers

#### `--short-only` (Development Mode)
Skip all long-running tests for instant feedback.

```bash
# During development
pytest-cream run --short-only

# Combined with limit
pytest-cream run --short-only --limit 30
```

**Use cases:**
- TDD workflow: Instant feedback loop
- Quick validation: Check if changes break fast tests
- Pre-commit: Run only unit tests, skip integration

#### `--limit` (Test Sampling)
Run only the first N tests (after long/short splitting).

```bash
# Quick sanity check
pytest-cream run --limit 10

# Sample 20% of suite
pytest-cream run --limit 50  # if you have ~250 tests
```

**Use cases:**
- Initial development: Test basic functionality
- Debugging: Isolate issues with small test set
- Performance testing: Profile with subset

#### `--exclude-tests` (Selective Execution)
Skip specific test files or directories.

```bash
# Exclude single file
pytest-cream run --exclude-tests "test_slow.py"

# Exclude directory
pytest-cream run --exclude-tests "test_integration/"

# Multiple exclusions
pytest-cream run --exclude-tests "test_e2e/,test_slow.py,test_manual/"
```

**Use cases:**
- Skip integration tests locally
- Exclude manual tests
- Avoid expensive E2E tests during development

### Performance Comparison

Example test suite with 100 tests:
- 10 long tests (avg 5s each) = 50s total
- 90 short tests (avg 0.2s each) = 18s total

**Traditional pytest (sequential):**
```
Total time: 68s
```

**pytest-cream (orchestrated with 8 workers):**
```
Long tests: 50s (sequential)
Short tests: 18s / 8 workers = 2.25s (parallel)
Total time: ~52s (24% faster)
```

**pytest-cream with `--short-only`:**
```
Short tests only: 2.25s (parallel with 8 workers)
Total time: ~2.5s (96% faster for development)
```

### Real-World Optimization Strategies

#### Strategy 1: Fast Development Cycle
```bash
# First run: Profile everything
pytest-cream profile

# During development: Only fast tests
pytest-cream run --short-only --workers 8

# Pre-commit: Fast tests with sampling
pytest-cream run --short-only --limit 50
```

#### Strategy 2: CI Pipeline
```bash
# Profile once (cache the result)
pytest-cream profile

# Pull request: All tests, exclude E2E
pytest-cream run --threshold 1.0 --workers 16 \
  --exclude-tests "test_e2e/"

# Main branch: Everything
pytest-cream run --threshold 1.0 --workers 16
```

#### Strategy 3: Large Test Suite
```bash
# Profile with higher threshold for complex tests
pytest-cream profile

# Distribute intelligently
pytest-cream run --threshold 2.0 --workers 32 \
  --exclude-tests "test_manual/"
```

## Workspace Organization

All commands use a shared workspace for easy file management:

```
~/workspace/
└── owner_project_main/           # Created by fetch command
    ├── src/                       # Your project files
    ├── tests/                     # Test files
    ├── test_durations.log         # Created by run command
    └── filtered_tests.txt         # Created by filter command
```

This organization makes it easy to:
- Find all test-related files in one place
- Chain commands together using the same workspace
- Review logs and results after test execution

## Installation Modes

When using `--install-repo` in init, you have several options. **By default, uv is used for faster and more reliable installations** (automatically detects virtual environments and uses appropriate flags).

### Standard Modes:
- `pip`: Direct install from git+https URL (fastest)
- `editable`: Clone repository and install in editable mode (for development)
- `wheel`: Build wheel and install (most reproducible)

All standard modes use uv by default. To use pip instead, add `--use-pip`.

**Note**: uv automatically detects if you're in a virtual environment. Outside a venv, it uses `--system` flag; inside a venv, it installs normally.

### Using pip Instead of uv:
If you encounter compatibility issues with uv, use `--use-pip` to fallback to pip:

```bash
# Standard install with pip (slower but maximum compatibility)
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project \
  --install-mode pip \
  --use-pip \
  --ws ~/workspace

# Editable install with pip
pytest-cream init --repo owner/project --branch main \
  --install-repo owner/project \
  --install-mode editable \
  --use-pip \
  --ws ~/workspace
```

### Custom Commands:
Use `--install-cmd` for complex dependency management that standard modes can't handle:

```bash
# uv with extras
--install-cmd "uv sync --extra cpu"

# Poetry with multiple extras
--install-cmd "poetry install --extras dev,test"

# pip with specific requirements file
--install-cmd "pip install -r requirements-dev.txt"

# Conda environment setup
--install-cmd "conda env update -f environment.yml"
```

**Note**: `--install-cmd` overrides `--install-mode` when provided. The command runs in the cloned repository directory.

### Usage Priority:
1. If `--install-cmd` is provided → Use custom command
2. If `--use-pip` is provided → Use pip instead of uv
3. Otherwise → Use uv with standard `--install-mode` (pip/editable/wheel)

**Note**: uv is used by default for faster installations. Install uv with: `pip install uv` or `curl -LsSf https://astral.sh/uv/install.sh | sh`

## Troubleshooting

### Python Version Issues
Some projects require specific Python versions, especially those with Rust extensions or compiled components:

```bash
# Use Python 3.10 for projects with Rust extensions
pytest-cream init \
  --repo owner/rust-project \
  --python python3.10 \
  --ws ~/workspace

# Use Python 3.11 for newer projects
pytest-cream init \
  --repo owner/modern-project \
  --python python3.11 \
  --ws ~/workspace
```

**Common Python version issues:**
- Rust extensions often require Python 3.10 or specific versions
- Some packages may not be compatible with Python 3.13+
- Use `--python` to specify the exact Python executable to use

### Excluding Problematic Tests
When some tests fail due to missing dependencies or environment issues, exclude them using the `run` command:

```bash
# Exclude specific test files during test execution
pytest-cream run --ws ~/workspace/owner_project_main \
  --exclude-tests "test_integration.py,test_slow.py" \
  --python python3.11

# Exclude test directories
pytest-cream run --ws ~/workspace/owner_project_main \
  --exclude-tests "tests/integration,tests/gpu" \
  --python python3.11
```

### Pip Issues
If you encounter "No module named pip" errors or issues with uv, try using the `--use-pip` flag to use pip instead of uv:

```bash
pytest-cream init --repo owner/repo --install-repo owner/repo \
  --use-pip --install-mode editable --python python3.11 --ws ~/workspace
```

By default, pytest-cream uses [uv](https://github.com/astral-sh/uv) for faster and more reliable installations.

### Custom Command Issues
If your `--install-cmd` fails:

1. **Check the command works manually**:
   ```bash
   git clone https://github.com/owner/repo
   cd repo
   uv sync --extra cpu  # or your command
   ```

2. **Common fixes**:
   - For uv: Ensure `pyproject.toml` has the required extras defined
   - For Poetry: Make sure `pyproject.toml` exists and has proper dependencies
   - For build errors: Check if system dependencies (like Rust, C compilers) are needed

3. **Debugging**: Check the workspace directory for logs and cloned repository to investigate build issues

## Complete Workflow Examples

### Example 1: Step-by-Step Testing
Execute each phase separately for maximum control:

```bash
# Set up variables for easier reuse
WORKSPACE=~/my-test-workspace
REPO=owner/my-project
BRANCH=main

# Step 1: Initialize - fetch the repository and install dependencies
pytest-cream init --repo $REPO --branch $BRANCH \
  --install-repo $REPO --install-mode editable \
  --ws $WORKSPACE

# Step 2: Profile - collect test duration statistics
pytest-cream profile --ws $WORKSPACE/owner_my-project_$BRANCH --python python3.11

# Step 3: Run - execute tests with intelligent orchestration
pytest-cream run --ws $WORKSPACE/owner_my-project_$BRANCH \
  --python python3.11 --threshold 0.5

# Step 4: (Optional) Filter - analyze test durations
pytest-cream filter --ws $WORKSPACE/owner_my-project_$BRANCH --threshold 0.5

# Step 5: Review results
echo "Tests are in: $WORKSPACE/owner_my-project_$BRANCH"
cat $WORKSPACE/owner_my-project_$BRANCH/filtered_tests.txt
```

### Example 2: Quick Testing
For rapid testing with automated setup:

```bash
# Initialize workspace
pytest-cream init \
  --repo owner/project \
  --branch develop \
  --install-repo owner/project \
  --install-mode editable \
  --ws ~/quick-workspace

# Profile tests
pytest-cream profile \
  --ws ~/quick-workspace/owner_project_develop \
  --python python3.11

# Run with custom settings
pytest-cream run \
  --ws ~/quick-workspace/owner_project_develop \
  --python python3.11 \
  --threshold 0.3 \
  --workers 8
```

### Example 3: Testing with Custom Dependencies
When your project needs special installation steps:

```bash
# Initialize with custom install command
pytest-cream init \
  --repo owner/ml-project \
  --branch main \
  --install-repo owner/ml-project \
  --install-cmd "uv sync --extra gpu,training" \
  --install-fallback \
  --ws ~/ml-workspace

# Profile tests
pytest-cream profile \
  --ws ~/ml-workspace/owner_ml-project_main \
  --python python3.10

# Run tests with exclusions
pytest-cream run \
  --ws ~/ml-workspace/owner_ml-project_main \
  --python python3.10 \
  --exclude-tests "test_slow_training.py,tests/integration/"
```

### Example 4: Development Testing
Test changes in a development branch with editable install:

```bash
# Initialize with editable install
pytest-cream init \
  --repo owner/my-library \
  --branch feature-branch \
  --install-repo owner/my-library \
  --install-branch feature-branch \
  --install-mode editable \
  --ws ~/dev-workspace

# Profile and run
pytest-cream profile --ws ~/dev-workspace/owner_my-library_feature-branch --python python3.11
pytest-cream run --ws ~/dev-workspace/owner_my-library_feature-branch --python python3.11
```

### Example 5: CI/CD Pipeline
Suitable for automated testing environments:

```bash
#!/bin/bash
set -e  # Exit on error

# Initialize workspace
pytest-cream init \
  --repo $CI_REPO \
  --branch $CI_BRANCH \
  --install-repo $CI_REPO \
  --install-mode pip \
  --ws /tmp/ci-workspace-$CI_JOB_ID

WORKSPACE="/tmp/ci-workspace-$CI_JOB_ID/$(echo $CI_REPO | tr '/' '_')_$CI_BRANCH"

# Profile tests
pytest-cream profile --ws $WORKSPACE --python python3.11

# Run tests and capture results
pytest-cream run \
  --ws $WORKSPACE \
  --python python3.11 \
  --exclude-tests "tests/manual/,test_interactive.py"

# Check if tests passed
if [ $? -eq 0 ]; then
  echo "✅ All tests passed!"
  exit 0
else
  echo "❌ Tests failed!"
  exit 1
fi
```

## Tips and Best Practices

1. **Always specify `--python`**: Avoid "python command not found" errors by explicitly setting the Python executable (e.g., `--python python3.11`)

2. **Use meaningful workspace names**: Organize workspaces by project or purpose (e.g., `~/workspaces/myproject-dev`, `~/workspaces/myproject-prod`)

3. **Keep workspaces outside project directories**: Store workspaces in a separate location (e.g., `~/pytest-cream-workspaces/`) to avoid pytest collection issues. **Never create workspaces inside your pytest-cream installation directory.**

4. **Reuse workspaces for iteration**: When testing the same project multiple times, you can reuse the workspace or let pytest-cream create timestamped directories

5. **Check workspace contents**: After running commands, inspect the workspace directory to review logs, test results, and any generated files

6. **Exclude flaky tests**: Use `--exclude-tests` with the `run` command to skip tests that are environment-dependent or have external dependencies

7. **Use `init` for setup**: The `init` command handles fetching and installing, then use `profile` and `run` for test execution

8. **Clean up old workspaces**: Periodically remove old workspace directories to free up disk space

9. **Profile before running**: Always run `profile` before `run` to collect test duration statistics for intelligent orchestration

## Documentation

For comprehensive orchestration guidance, see:
- **[Orchestration Guide](ORCHESTRATION_GUIDE.md)** - Complete reference with patterns, troubleshooting, and optimization techniques
- **[Usage Examples](USAGE.md)** - Basic workflows and commands
- **[Real-World Example](nanochat_readme.md)** - Using pytest-cream with the nanochat project

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## License
MIT License