import argparse
import os
import sys
import tempfile
from pytest_cream.fetch import fetch_tests
from pytest_cream.run import run_tests
from pytest_cream.filter import filter_tests
from pytest_cream import workflow
from pytest_cream.config import (
    set_last_workspace, 
    resolve_workspace, 
    resolve_python_executable,
    get_last_workspace
)


def main():
    parser = argparse.ArgumentParser(
        description="pytest-cream: The cream of test execution - smooth pytest workflows with intelligent orchestration",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Basic usage - initialize (fetch + install)
  pytest-cream init --repo owner/project
  pytest-cream profile
  pytest-cream run

  # With specific branch
  pytest-cream init --repo owner/project --branch develop
  pytest-cream profile
  pytest-cream run

  # Custom workspace location
  pytest-cream init --repo owner/project --ws ~/custom-workspace
  pytest-cream profile
  pytest-cream run

For more examples and troubleshooting, see the documentation.
        """
    )
    subparsers = parser.add_subparsers(dest="command")

    # Fetch command
    fetch_parser = subparsers.add_parser(
        "fetch", 
        help="Fetch tests from a repository",
        description="Clone a repository branch to a workspace directory for testing."
    )
    fetch_parser.add_argument(
        "--repo",
        required=True,
        help="Repository to fetch from in format 'owner/repo' or full GitHub URL"
    )
    fetch_parser.add_argument(
        "--branch", 
        default="main",
        help="Git branch to checkout and fetch tests from. Default: main"
    )
    fetch_parser.add_argument(
        "--ws",
        "--workspace",
        dest="workspace",
        required=True,
        help="Directory path where the repository will be cloned. Use the same workspace across fetch/run/filter commands."
    )

    # Profile command (was 'run')
    profile_parser = subparsers.add_parser(
        "profile", 
        help="Profile tests and collect duration statistics",
        description="Execute tests from a workspace directory and generate duration logs for analysis."
    )
    profile_parser.add_argument(
        "--ws",
        "--workspace",
        dest="workspace",
        required=False,
        help="Workspace directory containing the fetched tests. If not provided, uses the last initialized workspace."
    )
    profile_parser.add_argument(
        "--output", 
        default="test_durations.log", 
        help="Output file name for test durations (saved in workspace). Default: test_durations.log"
    )
    profile_parser.add_argument(
        "--tests-dir", 
        default="tests", 
        help="Subdirectory within workspace containing test files. Default: tests"
    )
    profile_parser.add_argument(
        "--python",
        dest="python_executable",
        default=None,
        help="Python executable to use (e.g., 'python3.11'). If not specified, auto-detects from workspace .venv."
    )
    profile_parser.add_argument(
        "--use-pip",
        action="store_true",
        default=False,
        help="Use pip instead of uv for running tests. Default: use uv"
    )

    # Filter command
    filter_parser = subparsers.add_parser(
        "filter", 
        help="Filter tests by duration threshold",
        description="Parse test duration logs and filter tests into long/short categories based on execution time."
    )
    filter_parser.add_argument(
        "--ws",
        "--workspace",
        dest="workspace",
        required=True,
        help="Workspace directory containing the test durations log. Use the same workspace from fetch/run commands."
    )
    filter_parser.add_argument(
        "--input", 
        default="test_durations.log",
        help="Input file name with test durations (relative to workspace). Default: test_durations.log"
    )
    filter_parser.add_argument(
        "--output", 
        default="filtered_tests.txt",
        help="Output file name for filtered test results (saved in workspace). Default: filtered_tests.txt"
    )
    filter_parser.add_argument(
        "--threshold", 
        type=float, 
        required=True, 
        help="Duration threshold in seconds. Tests >= threshold are 'long', tests < threshold are 'short'."
    )

    # Run command (orchestration - was 'execute')
    run_parser = subparsers.add_parser(
        "run",
        help="Run tests with intelligent orchestration based on duration analysis",
        description="Execute tests with intelligent orchestration: long tests sequentially, short tests in parallel."
    )
    run_parser.add_argument(
        "--ws",
        "--workspace",
        dest="workspace",
        required=False,
        help="Workspace directory containing the tests and duration logs. If not provided, uses the last initialized workspace."
    )
    run_parser.add_argument(
        "--durations",
        default="test_durations.log",
        help="Duration log file (relative to workspace). Default: test_durations.log"
    )
    run_parser.add_argument(
        "--threshold",
        type=float,
        default=0.5,
        help="Duration threshold in seconds for splitting long/short tests. Default: 0.5"
    )
    run_parser.add_argument(
        "--workers",
        type=int,
        default=4,
        help="Number of parallel workers for short tests. Default: 4"
    )
    run_parser.add_argument(
        "--python",
        dest="python_executable",
        default=None,
        help="Python executable to use (e.g., 'python3.11'). If not specified, auto-detects from workspace .venv."
    )
    run_parser.add_argument(
        "--limit",
        type=int,
        default=None,
        help="Maximum number of tests to run."
    )
    run_parser.add_argument(
        "--exclude-tests",
        dest="exclude_tests",
        default=None,
        help="Comma-separated list of test patterns to exclude."
    )
    run_parser.add_argument(
        "--short-only",
        dest="short_only",
        action="store_true",
        help="Run only short tests (skip long tests)."
    )
    run_parser.add_argument(
        "--use-pip",
        action="store_true",
        default=False,
        help="Use pip instead of uv for running tests. Default: use uv"
    )

    # Init command (setup: fetch + install)
    init_parser = subparsers.add_parser(
        "init",
        help="Initialize: fetch tests and install dependencies from a repository",
        description="Setup workflow: fetch tests from repository and install dependencies. Does not profile or run tests.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Basic usage (recommended - uses current directory)
  pytest-cream init --repo owner/project
  pytest-cream profile
  pytest-cream run

  # With specific branch
  pytest-cream init --repo owner/project --branch develop
  pytest-cream profile
  pytest-cream run

  # Custom workspace location
  pytest-cream init --repo owner/project --ws ~/workspace
  pytest-cream profile
  pytest-cream run

  # Use pip instead of uv (for compatibility)
  pytest-cream init --repo owner/project --use-pip
  pytest-cream profile
  pytest-cream run

  # Project with custom dependencies using uv
  pytest-cream init --repo owner/project \\
    --install-cmd "uv sync --extra cpu"
  pytest-cream profile
  pytest-cream run

  # Development workflow with editable install
  pytest-cream init --repo owner/project \\
    --install-mode editable
  pytest-cream profile
  pytest-cream run

  # Manual control with explicit paths (optional)
  pytest-cream init --repo owner/project --ws ~/workspace
  pytest-cream profile --ws ~/workspace/owner_project_main --python python3.11
  pytest-cream run --ws ~/workspace/owner_project_main --python python3.11
        """
    )
    init_parser.add_argument(
        "--repo",
        default="owner/project",
        help="Repository to fetch in format 'owner/repo' or full GitHub URL.",
    )
    init_parser.add_argument(
        "--branch", 
        default="main", 
        help="Git branch to checkout. Default: main"
    )
    init_parser.add_argument(
        "--ws",
        "--workspace",
        dest="workspace",
        default=None,
        help="Directory path where the repository will be cloned. Default: current directory (with safety check if not empty)",
    )
    init_parser.add_argument(
        "--install-repo",
        dest="install_repo",
        default=None,
        help="Repository to install dependencies from (format: 'owner/repo'). Default: same as --repo. Use this to install from a different repository than the one being tested.",
    )
    init_parser.add_argument(
        "--install-branch",
        dest="install_branch",
        default=None,
        help="Git branch to use when installing from --install-repo. If not specified, uses the same branch as --branch.",
    )
    init_parser.add_argument(
        "--install-mode",
        dest="install_mode",
        default="pip",
        choices=["pip", "editable", "wheel"],
        help="Installation method: 'pip' (direct install), 'editable' (pip install -e, for development), 'wheel' (build wheel first). Default: pip.",
    )
    init_parser.add_argument(
        "--use-pip",
        dest="use_pip",
        action="store_true",
        help="Use pip instead of uv package manager. By default, uv is used for faster and more reliable installations.",
    )
    init_parser.add_argument(
        "--install-cmd",
        dest="install_cmd",
        default=None,
        help="Custom shell command for dependency installation (e.g., 'uv sync --extra cpu', 'poetry install'). Overrides --install-mode.",
    )
    init_parser.add_argument(
        "--install-fallback",
        dest="install_fallback",
        action="store_true",
        help="If --install-cmd fails, fallback to standard pip installation instead of aborting.",
    )
    init_parser.add_argument(
        "--python",
        dest="python_executable",
        default=None,
        help="Python executable to use (e.g., 'python3.11'). Useful for install verification.",
    )

    args = parser.parse_args()

    if args.command == "fetch":
        # Validate and create workspace
        os.makedirs(args.workspace, exist_ok=True)
        output_dir = os.path.join(args.workspace, f"{args.repo.replace('/', '_')}_{args.branch}")
        extracted_path = fetch_tests(branch=args.branch, repo=args.repo, output_dir=output_dir)
        print(f"Tests fetched to: {extracted_path}")
        
    elif args.command == "profile":
        # Resolve workspace
        workspace = resolve_workspace(args.workspace)
        if not workspace:
            if args.workspace:
                print(f"Error: Workspace '{args.workspace}' does not exist.", file=sys.stderr)
            else:
                print("Error: No workspace specified and no previous workspace found.", file=sys.stderr)
                print("Either provide --ws or run 'pytest-cream init' first.", file=sys.stderr)
            sys.exit(1)
        
        print(f"Using workspace: {workspace}")
        
        # Resolve Python executable
        python_exec = resolve_python_executable(workspace, args.python_executable)
        if not python_exec and not args.python_executable:
            print("Warning: No Python executable specified and none found in workspace .venv")
            print("Using system Python (this may cause import errors)")
        
        # Resolve paths relative to workspace
        tests_path = os.path.join(workspace, args.tests_dir)
        output_path = os.path.join(workspace, args.output)
        
        run_tests(
            output_file=output_path, 
            tests_dir=tests_path,
            python_executable=python_exec,
            use_uv=not args.use_pip
        )
        print(f"Test durations saved to: {output_path}")
        
    elif args.command == "filter":
        # Validate workspace exists
        if not os.path.exists(args.workspace):
            print(f"Error: Workspace '{args.workspace}' does not exist. Run fetch/run commands first.", file=sys.stderr)
            sys.exit(1)
        
        # Resolve paths relative to workspace
        input_path = os.path.join(args.workspace, args.input)
        output_path = os.path.join(args.workspace, args.output)
        
        if not os.path.exists(input_path):
            print(f"Error: Input file '{input_path}' does not exist. Run the run command first.", file=sys.stderr)
            sys.exit(1)
        
        filter_tests(
            input_file=input_path, 
            output_file=output_path, 
            threshold=args.threshold
        )
        print(f"Filtered tests saved to: {output_path}")
        
    elif args.command == "run":
        # Resolve workspace
        workspace = resolve_workspace(args.workspace)
        if not workspace:
            if args.workspace:
                print(f"Error: Workspace '{args.workspace}' does not exist.", file=sys.stderr)
            else:
                print("Error: No workspace specified and no previous workspace found.", file=sys.stderr)
                print("Either provide --ws or run 'pytest-cream init' first.", file=sys.stderr)
            sys.exit(1)
        
        print(f"Using workspace: {workspace}")
        
        # Resolve Python executable
        python_exec = resolve_python_executable(workspace, args.python_executable)
        if not python_exec and not args.python_executable:
            print("Warning: No Python executable specified and none found in workspace .venv")
        
        # Resolve durations file path
        durations_path = os.path.join(workspace, args.durations)
        if not os.path.exists(durations_path):
            print(f"Error: Durations file '{durations_path}' does not exist. Run profile command first.", file=sys.stderr)
            sys.exit(1)
        
        # Parse exclude patterns if provided
        exclude_tests = []
        if args.exclude_tests:
            exclude_tests = [pattern.strip() for pattern in args.exclude_tests.split(",") if pattern.strip()]
        
        # Execute tests with orchestration
        from pytest_cream.workflow import _parse_durations, _split_tests, _filter_excluded_tests, _run_tests_orchestrated
        
        parsed = _parse_durations(durations_path)
        long_tests, short_tests = _split_tests(parsed, args.threshold)
        
        # Filter excluded tests
        if exclude_tests:
            long_tests = _filter_excluded_tests(long_tests, exclude_tests)
            short_tests = _filter_excluded_tests(short_tests, exclude_tests)
        
        # Apply test limit if specified
        if args.limit is not None and args.limit > 0:
            total_tests = len(long_tests) + len(short_tests)
            if total_tests > args.limit:
                print(f"Limiting tests from {total_tests} to {args.limit}")
                if len(long_tests) >= args.limit:
                    long_tests = long_tests[:args.limit]
                    short_tests = []
                else:
                    remaining = args.limit - len(long_tests)
                    short_tests = short_tests[:remaining]
        
        # Apply short-only filter if specified
        if args.short_only:
            print(f"Running only short tests (skipping {len(long_tests)} long tests)")
            long_tests = []
        
        print(f"Found {len(long_tests)} long tests and {len(short_tests)} short tests (threshold={args.threshold})")
        
        # Run tests with orchestration
        _run_tests_orchestrated(
            workspace=workspace,
            long_tests=long_tests,
            short_tests=short_tests,
            workers=args.workers,
            python_executable=python_exec,
            exclude_tests=exclude_tests,
            use_uv=not args.use_pip
        )
        
    elif args.command == "init":
        # Default to current directory if workspace not provided
        workspace = args.workspace if args.workspace else os.getcwd()
        
        # Safety check: warn if current directory is not empty and no explicit workspace provided
        if not args.workspace and workspace == os.getcwd():
            # Check if directory has files (excluding hidden files and common safe files)
            files = [f for f in os.listdir(workspace) if not f.startswith('.') and f not in ['README.md', 'LICENSE']]
            
            if len(files) > 0:
                print(f"⚠️  Warning: Current directory is not empty ({len(files)} files/folders)")
                print(f"   Location: {workspace}")
                print()
                print("This will create test workspace files in the current directory.")
                print("Consider using --ws option to specify a different location.")
                print()
                response = input("Continue with current directory? [y/N]: ").strip().lower()
                
                if response not in ['y', 'yes']:
                    print()
                    print("Cancelled. To use a different workspace, run:")
                    print(f"  pytest-cream init --repo {args.repo} --ws ~/pytest-cream-tests")
                    sys.exit(0)
                print()
        
        print(f"📁 Using workspace: {workspace}")
        
        # Validate workspace upfront and fail fast if it's not usable
        try:
            # Try to create the workspace directory if missing
            os.makedirs(workspace, exist_ok=True)
            # Try creating and removing a temporary file to verify writability
            fd, tmpfile = tempfile.mkstemp(dir=workspace)
            os.close(fd)
            os.remove(tmpfile)
        except Exception as e:
            print(
                f"Error: cannot use workspace '{workspace}': {e}",
                file=sys.stderr,
            )
            sys.exit(2)

        # Default install_repo to the same repo if not specified
        install_repo = args.install_repo if args.install_repo else args.repo
        install_branch = args.install_branch if args.install_branch else args.branch

        workflow.init_only(
            repo_url=args.repo,
            branch=args.branch,
            workspace=workspace,
            install_repo=install_repo,
            install_branch=install_branch,
            install_mode=args.install_mode,
            install_cmd=args.install_cmd,
            install_fallback=args.install_fallback,
            install_uv=not args.use_pip,  # Default to uv unless --use-pip is specified
        )
        
        # Save workspace configuration
        set_last_workspace(workspace, args.repo, args.branch)
        print(f"\n✓ Workspace saved to .pytest-cream.json")
        print(f"  pytest-cream profile")
        print(f"  pytest-cream run")
        
    else:
        parser.print_help()
