"""CLI for MTB Video Sync."""

import time
from pathlib import Path
from typing import Optional

import numpy as np
import typer
from rich.console import Console
from rich.panel import Panel

from mtbsync.features.descriptors import compute_orb_descriptors, save_reference_index
from mtbsync.features.keyframes import resize_max
from mtbsync.io.video import extract_keyframes, video_fps
from mtbsync.match.retrieval import (
    load_reference_index,
    retrieve_coarse_pairs,
    save_pairs_csv,
)
from mtbsync.match.local import refine_pairs_locally, save_refined_pairs
from mtbsync.telemetry import TelemetryRecorder

app = typer.Typer(
    name="mtbsync",
    help="MTB Video Sync - Align GoPro runs and transfer markers",
    add_completion=False,
    rich_markup_mode=None,
)

console = Console()


@app.command()
def index(
    reference: Path = typer.Option(..., "--reference", help="Reference MP4 file"),
    fps: float = typer.Option(3.0, "--fps", help="Keyframe extraction rate (fps)"),
    out: Path = typer.Option(..., "--out", help="Output index file (.npz)"),
    n_features: int = typer.Option(1500, "--n-features", help="Max ORB features per frame"),
    no_clahe: bool = typer.Option(False, "--no-clahe", help="Disable CLAHE preprocessing"),
) -> None:
    """Extract and index keyframes from reference video."""
    use_clahe = not no_clahe

    console.print(
        Panel(
            f"[cyan]Index command[/cyan]\n"
            f"Reference: {reference}\n"
            f"FPS: {fps}\n"
            f"Features: {n_features}\n"
            f"CLAHE: {'enabled' if use_clahe else 'disabled'}\n"
            f"Output: {out}",
            title="mtbsync index",
        )
    )

    try:
        # Extract keyframes
        console.print(f"Extracting keyframes at {fps} fps...")
        keyframes = extract_keyframes(str(reference), fps)

        if not keyframes:
            console.print("[red]Error: No keyframes extracted[/red]")
            raise typer.Exit(1)

        # Resize frames
        console.print(f"Resizing {len(keyframes)} frames...")
        resized_keyframes = []
        for timestamp, frame in keyframes:
            resized_frame = resize_max(frame, max_dim=960)
            resized_keyframes.append((timestamp, resized_frame))

        # Prepare data
        timestamps = np.array([t for t, _ in resized_keyframes], dtype=np.float64)
        frames = [f for _, f in resized_keyframes]

        # Compute ORB descriptors
        console.print(f"Computing ORB descriptors (n_features={n_features})...")
        kpts_list, desc_list = compute_orb_descriptors(
            frames, n_features=n_features, use_clahe=use_clahe
        )

        # Check if we got any keypoints at all
        total_kpts = sum(len(kpts) for kpts in kpts_list)
        if total_kpts == 0:
            console.print(
                "[red]Error: No keypoints detected in any frame. "
                "Video may be too blurry or featureless.[/red]"
            )
            raise typer.Exit(1)

        # Calculate statistics
        kpt_counts = [len(kpts) for kpts in kpts_list]
        mean_kpts = np.mean(kpt_counts)
        median_kpts = np.median(kpt_counts)
        zero_kpt_frames = sum(1 for count in kpt_counts if count == 0)
        pct_zero = (zero_kpt_frames / len(kpt_counts)) * 100 if kpt_counts else 0

        # Create output directory if needed
        out.parent.mkdir(parents=True, exist_ok=True)

        # Prepare metadata
        meta = {
            "source_video": str(reference),
            "fps": float(fps),
            "image_max_dim": 960,
            "orb_n_features": n_features,
            "use_clahe": use_clahe,
            "num_frames": len(timestamps),
            "total_keypoints": int(total_kpts),
        }

        # Save reference index
        console.print(f"Saving index to {out}...")
        save_reference_index(
            str(out),
            t_ref=timestamps,
            kpts_list=kpts_list,
            desc_list=desc_list,
            meta=meta,
        )

        # Get file size
        file_size_mb = out.stat().st_size / (1024 * 1024)

        # Print summary
        first_ts = timestamps[0]
        last_ts = timestamps[-1]
        console.print(
            Panel(
                f"[green]Success![/green]\n"
                f"Keyframes: {len(keyframes)}\n"
                f"Mean keypoints/frame: {mean_kpts:.1f}\n"
                f"Median keypoints/frame: {median_kpts:.1f}\n"
                f"Frames with 0 keypoints: {zero_kpt_frames} ({pct_zero:.1f}%)\n"
                f"First timestamp: {first_ts:.3f}s\n"
                f"Last timestamp: {last_ts:.3f}s\n"
                f"Output: {out}\n"
                f"File size: {file_size_mb:.2f} MB",
                title="Index Summary",
            )
        )

    except Exception as e:
        console.print(f"[red]Error: {e}[/red]")
        raise typer.Exit(1)


@app.command()
def sync(
    reference: Path = typer.Option(..., "--reference", help="Reference MP4 file"),
    new: Path = typer.Option(..., "--new", help="New MP4 file to sync"),
    index: Path = typer.Option(..., "--index", help="Reference index file (.npz)"),
    out: Path = typer.Option(..., "--out", help="Output pairs CSV"),
    ref_gpx: Optional[Path] = typer.Option(None, "--ref-gpx", help="Reference GPX file"),
    new_gpx: Optional[Path] = typer.Option(None, "--new-gpx", help="New GPX file"),
    fps: float = typer.Option(3.0, "--fps", help="Keyframe extraction rate (fps)"),
    n_features: int = typer.Option(1500, "--n-features", help="Max ORB features per frame"),
    no_clahe: bool = typer.Option(False, "--no-clahe", help="Disable CLAHE preprocessing"),
    top_k: int = typer.Option(3, "--top-k", help="Top K reference matches per frame"),
    no_refine: bool = typer.Option(False, "--no-refine", help="Skip local refinement"),
    refine_window: int = typer.Option(1, "--refine-window", help="Refinement window in keyframes"),
    lowe: float = typer.Option(0.75, "--lowe", help="Lowe ratio test threshold"),
    ransac_reproj: float = typer.Option(3.0, "--ransac-reproj", help="RANSAC reprojection threshold (px)"),
    min_inliers: int = typer.Option(20, "--min-inliers", help="Minimum RANSAC inliers required"),
    # --- Time-Warp parameters ---
    no_warp: bool = typer.Option(False, "--no-warp", help="Disable time-warp fitting and gating"),
    warp_window_sec: float = typer.Option(
        1.0,
        "--warp-window-sec",
        help="Acceptance window for time-warp gating (seconds)",
    ),
    warp_inlier_thresh: float = typer.Option(
        0.08,
        "--warp-inlier-thresh",
        help="RANSAC inlier threshold for time-warp fitting (seconds)",
    ),
    warp_ransac_iters: int = typer.Option(
        500,
        "--warp-ransac-iters",
        help="Number of RANSAC iterations for time-warp fitting",
    ),
    warp_max_ppm: int = typer.Option(
        1000,
        "--warp-max-ppm",
        help="Maximum allowed drift (ppm) for accepted warp model",
    ),
    warp_min_inlier_frac: float = typer.Option(
        0.25,
        "--warp-min-inlier-frac",
        help="Minimum inlier fraction required to accept warp model",
    ),
    warp_rng_seed: Optional[int] = typer.Option(
        0,
        "--warp-rng-seed",
        help="Random seed for deterministic time-warp fitting (None to disable)",
    ),
    # --- Performance parameters ---
    threads: int = typer.Option(1, "--threads", help="Number of threads for coarse retrieval"),
    fast: bool = typer.Option(False, "--fast", help="Enable preset for large jobs (auto-tunes params)"),
    no_gpu: bool = typer.Option(False, "--no-gpu", help="Disable GPU telemetry probe"),
    # -----------------------------
) -> None:
    """Build time pairs between reference and new video."""
    use_clahe = not no_clahe

    # Apply --fast preset
    if fast:
        warp_window_sec = 1.0
        warp_inlier_thresh = 0.10
        warp_ransac_iters = 250
        threads = max(threads, 4)
        console.print("[yellow]--fast mode: warp_window=1.0s, warp_inlier_thresh=0.10, warp_ransac_iters=250, threads>=4[/yellow]")

    console.print(
        Panel(
            f"[cyan]Sync command[/cyan]\n"
            f"Reference: {reference}\n"
            f"New: {new}\n"
            f"Index: {index}\n"
            f"FPS: {fps}\n"
            f"Features: {n_features}\n"
            f"CLAHE: {'enabled' if use_clahe else 'disabled'}\n"
            f"Top-K: {top_k}\n"
            f"Refinement: {'disabled' if no_refine else 'enabled'}\n"
            f"Output: {out}\n"
            f"Ref GPX: {ref_gpx or 'None'}\n"
            f"New GPX: {new_gpx or 'None'}",
            title="mtbsync sync",
        )
    )

    try:
        start_time = time.time()

        # Ensure output directory exists before any artefact writes
        try:
            out_path = Path(out)
            out_path.parent.mkdir(parents=True, exist_ok=True)
        except Exception:
            # Let Typer/stack trace surface a clear error, but don't fail silently
            pass

        # Load reference index
        console.print("Loading reference index...")
        ref_timestamps, ref_kpts_list, ref_desc_list = load_reference_index(str(index))
        console.print(f"Loaded {len(ref_timestamps)} reference frames")

        # Check if GPS-based alignment is requested
        if ref_gpx or new_gpx:
            console.print("[yellow]GPS-based alignment not yet implemented[/yellow]")
            console.print("Falling back to visual retrieval...")

        # Extract keyframes from new video
        console.print(f"Extracting keyframes from new video at {fps} fps...")
        new_keyframes = extract_keyframes(str(new), fps)

        if not new_keyframes:
            console.print("[red]Error: No keyframes extracted from new video[/red]")
            raise typer.Exit(1)

        # Resize frames
        console.print(f"Resizing {len(new_keyframes)} frames...")
        resized_frames = []
        for timestamp, frame in new_keyframes:
            resized_frame = resize_max(frame, max_dim=960)
            resized_frames.append((timestamp, resized_frame))

        # Extract timestamps and frames
        new_timestamps = np.array([t for t, _ in resized_frames], dtype=np.float64)
        frames = [f for _, f in resized_frames]

        # Compute descriptors for new frames
        console.print(f"Computing ORB descriptors (n_features={n_features})...")
        new_kpts_list, new_desc_list = compute_orb_descriptors(
            frames, n_features=n_features, use_clahe=use_clahe
        )

        # Check if we got any keypoints
        total_new_kpts = sum(len(kpts) for kpts in new_kpts_list)
        if total_new_kpts == 0:
            console.print(
                "[red]Error: No keypoints detected in new video frames. "
                "Video may be too blurry or featureless.[/red]"
            )
            raise typer.Exit(1)

        # Perform coarse retrieval matching
        console.print(f"Matching frames (top-{top_k} per frame, threads={threads})...")
        pairs_raw_df, timings = retrieve_coarse_pairs(
            new_timestamps,
            new_desc_list,
            ref_timestamps,
            ref_desc_list,
            top_k=top_k,
            lowe_ratio=lowe,
            warp_enable=not no_warp,
            warp_window_sec=warp_window_sec,
            warp_inlier_thresh=warp_inlier_thresh,
            warp_ransac_iters=warp_ransac_iters,
            warp_max_ppm=warp_max_ppm,
            warp_min_inlier_frac=warp_min_inlier_frac,
            warp_rng_seed=warp_rng_seed,
            threads=threads,
        )

        # Print timing information
        console.print(
            f"[cyan]Timing:[/cyan] retrieval={timings.get('retrieval_sec', 0):.2f}s, "
            f"warp={timings.get('warp_sec', 0):.2f}s, "
            f"markers={timings.get('markers_sec', 0):.2f}s, "
            f"total={timings.get('total_sec', 0):.2f}s"
        )

        # Create output directory if needed
        out.parent.mkdir(parents=True, exist_ok=True)

        # Save intermediate pairs_raw.csv
        pairs_raw_path = out.parent / "pairs_raw.csv"
        console.print(f"Saving coarse pairs to {pairs_raw_path}...")
        save_pairs_csv(pairs_raw_df, str(pairs_raw_path))

        coarse_total = len(pairs_raw_df)
        coarse_unique_frames = pairs_raw_df["t_new"].nunique()

        # Perform local refinement unless --no-refine
        if no_refine:
            console.print("[yellow]Skipping local refinement (--no-refine)[/yellow]")
            # Just copy coarse pairs to output
            save_pairs_csv(pairs_raw_df, str(out))

            elapsed_time = time.time() - start_time
            console.print(
                Panel(
                    f"[green]Success![/green]\n"
                    f"New frames: {len(new_keyframes)}\n"
                    f"Coarse pairs: {coarse_total}\n"
                    f"Frames matched: {coarse_unique_frames}\n"
                    f"Runtime: {elapsed_time:.2f}s\n"
                    f"Output: {out}",
                    title="Sync Summary (No Refinement)",
                )
            )
        else:
            console.print(
                f"Refining pairs (window=±{refine_window}, "
                f"ransac_thresh={ransac_reproj}px, min_inliers={min_inliers})..."
            )

            try:
                refined_df = refine_pairs_locally(
                    pairs_raw_df,
                    new_timestamps,
                    new_kpts_list,
                    new_desc_list,
                    ref_timestamps,
                    ref_kpts_list,
                    ref_desc_list,
                    fps=fps,
                    refine_window=refine_window,
                    lowe_ratio=lowe,
                    ransac_thresh=ransac_reproj,
                    min_inliers=min_inliers,
                )
            except RuntimeError as e:
                console.print(f"[red]Refinement failed: {e}[/red]")
                raise typer.Exit(1)

            # Save refined pairs
            console.print(f"Saving refined pairs to {out}...")
            save_refined_pairs(refined_df, str(out))

            # Calculate statistics
            elapsed_time = time.time() - start_time
            refined_total = len(refined_df)
            refined_unique_frames = refined_df["t_new"].nunique()

            # Refinement stats
            mean_inliers = refined_df["inliers"].mean()
            median_inliers = refined_df["inliers"].median()
            median_error = refined_df["reproj_error"].median()

            # Count model types
            model_counts = refined_df["model"].value_counts().to_dict()
            n_homography = model_counts.get("H", 0)
            n_affine = model_counts.get("A", 0)

            # Dropped frames
            dropped_frames = coarse_unique_frames - refined_unique_frames
            pct_dropped = (dropped_frames / coarse_unique_frames * 100) if coarse_unique_frames > 0 else 0

            # Print summary
            console.print(
                Panel(
                    f"[green]Success![/green]\n"
                    f"New frames: {len(new_keyframes)}\n"
                    f"Coarse pairs: {coarse_total} ({coarse_unique_frames} frames)\n"
                    f"Refined pairs: {refined_total} ({refined_unique_frames} frames)\n"
                    f"Dropped frames: {dropped_frames} ({pct_dropped:.1f}%)\n"
                    f"Mean inliers: {mean_inliers:.1f}\n"
                    f"Median inliers: {median_inliers:.0f}\n"
                    f"Median reproj error: {median_error:.2f}px\n"
                    f"Models: {n_homography} homography, {n_affine} affine\n"
                    f"Runtime: {elapsed_time:.2f}s\n"
                    f"Output: {out}",
                    title="Sync Summary (With Refinement)",
                )
            )

        # Write performance telemetry
        try:
            frames = timings.get("frames_processed", 0)
            retr_sec = timings.get("retrieval_sec", 0.0)
            fps_calc = (frames / retr_sec) if retr_sec > 0 else None

            # Extended metrics (best-effort)
            cpu_mem = TelemetryRecorder.collect_cpu_pct_rss()
            gpu = TelemetryRecorder.collect_gpu_metrics(enable_gpu=(not no_gpu))

            perf_payload = {
                "ok": True,
                "cmd": "sync",
                "out": str(out),
                "timings": timings,
                "fps": fps_calc,
                "cpu_pct": cpu_mem.get("cpu_pct"),
                "rss_mb": cpu_mem.get("rss_mb"),
                "gpu_util": gpu.get("gpu_util"),
                "gpu_mem_mb": gpu.get("gpu_mem_mb"),
            }
            TelemetryRecorder.write_perf_json(out.parent, perf_payload)
        except Exception:
            # Best effort - don't crash on telemetry failure
            pass

    except Exception as e:
        console.print(f"[red]Error: {e}[/red]")
        raise typer.Exit(1)


@app.command()
def warp(
    pairs: Path = typer.Option(..., "--pairs", help="Pairs CSV file"),
    out: Path = typer.Option(..., "--out", help="Output warp file (.npz)"),
) -> None:
    """Fit time-warp mapping from pairs."""
    console.print(
        Panel(
            f"[cyan]Warp command[/cyan]\n" f"Pairs: {pairs}\n" f"Output: {out}",
            title="mtbsync warp",
        )
    )
    console.print("[yellow]Not yet implemented[/yellow]")


@app.command()
def transfer(
    reference_markers: Path = typer.Option(
        ..., "--reference-markers", help="Reference markers CSV"
    ),
    warp: Path = typer.Option(..., "--warp", help="Warp file (.npz)"),
    out: Path = typer.Option(..., "--out", help="Output markers CSV"),
    review_threshold: float = typer.Option(
        0.6, "--review-threshold", help="Confidence threshold for review flag"
    ),
) -> None:
    """Transfer markers from reference to new video."""
    console.print(
        Panel(
            f"[cyan]Transfer command[/cyan]\n"
            f"Reference markers: {reference_markers}\n"
            f"Warp: {warp}\n"
            f"Output: {out}\n"
            f"Review threshold: {review_threshold}",
            title="mtbsync transfer",
        )
    )
    console.print("[yellow]Not yet implemented[/yellow]")


@app.command()
def export(
    format: str = typer.Argument(..., help="Export format (edl)"),
    markers: Path = typer.Option(..., "--markers", help="Markers CSV file"),
    out: Path = typer.Option(..., "--out", help="Output file"),
    reel: str = typer.Option("001", "--reel", help="Reel name for EDL"),
    fps: float = typer.Option(29.97, "--fps", help="Frame rate for EDL"),
) -> None:
    """Export markers to various formats."""
    console.print(
        Panel(
            f"[cyan]Export command[/cyan]\n"
            f"Format: {format}\n"
            f"Markers: {markers}\n"
            f"Output: {out}\n"
            f"Reel: {reel}\n"
            f"FPS: {fps}",
            title="mtbsync export",
        )
    )
    console.print("[yellow]Not yet implemented[/yellow]")


@app.command()
def preview(
    reference: Path = typer.Option(..., "--reference", help="Reference MP4 file"),
    new: Path = typer.Option(..., "--new", help="New MP4 file"),
    markers: Path = typer.Option(..., "--markers", help="Markers CSV file"),
) -> None:
    """Launch Streamlit preview UI."""
    console.print(
        Panel(
            f"[cyan]Preview command[/cyan]\n"
            f"Reference: {reference}\n"
            f"New: {new}\n"
            f"Markers: {markers}",
            title="mtbsync preview",
        )
    )
    console.print("[yellow]Not yet implemented[/yellow]")


@app.command("transfer-markers")
def transfer_markers_cmd(
    ref_markers: str = typer.Option(..., "--ref-markers", help="Reference marker CSV file (must contain marker_id,t_ref)."),
    timewarp_json: str = typer.Option(..., "--timewarp-json", help="Path to timewarp.json generated by sync."),
    out: str = typer.Option("new_markers.csv", "--out", help="Output CSV for mapped markers."),
    label: Optional[str] = typer.Option(None, "--label", help="Optional label/track name to add to each marker."),
    vis_json: bool = typer.Option(False, "--vis-json", help="Also write a JSON version for visualisation."),
    plot_overlay: bool = typer.Option(
        False,
        "--plot-overlay",
        help="Generate PNG overlay comparing reference and transferred markers.",
    ),
):
    """
    Transfer markers from a reference video timeline to the new video using
    affine parameters in timewarp.json.
    """
    from mtbsync.match.marker_transfer import transfer_markers, load_timewarp_json
    import json

    result_path = transfer_markers(ref_markers, timewarp_json, out)

    if vis_json:
        tw = load_timewarp_json(timewarp_json)
        json_out = str(result_path).replace(".csv", ".json")
        import csv
        rows = list(csv.DictReader(open(result_path)))
        payload = {
            "params": tw["meta"].get("params", {}),
            "markers": rows,
            "label": label or "default"
        }
        with open(json_out, "w", encoding="utf-8") as f:
            json.dump(payload, f, indent=2)
        print(f"[Marker-Transfer] JSON visualisation written to {json_out}")

    # --- Optional visual overlay ---
    if plot_overlay:
        try:
            from mtbsync.match.marker_transfer import plot_marker_overlay
            ref_path = ref_markers
            new_path = out
            out_png = str(out).replace(".csv", "_overlay.png")
            plot_marker_overlay(ref_path, new_path, out_png)
        except Exception as e:
            print(f"[Marker-Transfer] Overlay generation failed: {e}")

    print(f"[Marker-Transfer] Complete. Output: {result_path}")


@app.command("viewer")
def viewer_cmd(
    ref_markers: Optional[str] = typer.Option(None, "--ref-markers", help="Reference markers CSV (marker_id,t_ref,...)"),
    timewarp_json: Optional[str] = typer.Option(None, "--timewarp-json", help="timewarp.json produced by sync"),
    new_markers: Optional[str] = typer.Option(None, "--new-markers", help="Transferred markers CSV (optional)"),
    out_png: Optional[str] = typer.Option(None, "--out-png", help="Headless: save overlay PNG to this path and exit"),
    headless: bool = typer.Option(False, "--headless", help="Render overlay PNG without opening a window"),
):
    """Open a synchronisation viewer (Tk) or render overlay in headless mode."""
    from mtbsync.viewer import launch_viewer, render_overlay_headless

    if headless:
        if not ref_markers or not timewarp_json or not out_png:
            raise typer.BadParameter("Headless mode requires --ref-markers, --timewarp-json, and --out-png.")
        render_overlay_headless(Path(ref_markers), Path(timewarp_json), Path(out_png), Path(new_markers) if new_markers else None)
        print(f"[Viewer] Headless overlay saved to {out_png}")
        return
    # GUI mode
    launch_viewer(ref_markers, timewarp_json, new_markers)


@app.command("batch")
def batch_cmd(
    input_dir: str = typer.Argument(..., help="Directory containing *_ref.mp4 / *_new.mp4 pairs."),
    out_dir: str = typer.Option("batch_out", "--out-dir", help="Output root directory."),
    ref_suffix: str = typer.Option("_ref.mp4", "--ref-suffix", help="Suffix for reference videos."),
    new_suffix: str = typer.Option("_new.mp4", "--new-suffix", help="Suffix for new videos."),
    dry_run: bool = typer.Option(False, "--dry-run", help="List work without running pipeline."),
    no_gpu: bool = typer.Option(False, "--no-gpu", help="Disable GPU telemetry probe"),
    threads: int = typer.Option(1, "--threads", help="Number of threads for coarse retrieval."),
    fast: bool = typer.Option(False, "--fast", help="Enable preset for large jobs (auto-tunes params)."),
):
    """Process multiple video pairs and produce alignment artefacts per pair."""
    from mtbsync.batch import run_batch
    results = run_batch(input_dir, out_dir, ref_suffix=ref_suffix, new_suffix=new_suffix,
                       dry_run=dry_run, no_gpu=no_gpu, threads=threads, fast=fast)
    ok = sum(1 for r in results if r.ok)
    print(f"[Batch] {ok}/{len(results)} ok")


@app.command("import-markers")
def import_markers_cmd(
    in_file: str = typer.Argument(..., help="Input file (EDL or XML)."),
    fmt: str = typer.Option(None, "--format", case_sensitive=False, help="Format: edl|xml (auto-detect if omitted)"),
    out: str = typer.Option(None, "--out", help="Output CSV path; inferred if omitted."),
    fps: int = typer.Option(30, "--fps", help="Frame rate for timecode/frame conversion."),
):
    """
    Import markers from EDL or XML formats to CSV.
    """
    from pathlib import Path
    from mtbsync.importers import import_edl, import_xml, save_markers_csv

    p_in = Path(in_file)
    if not p_in.exists():
        raise FileNotFoundError(f"Input file not found: {in_file}")

    # Auto-detect format from extension if not specified
    if fmt is None:
        ext = p_in.suffix.lower()
        if ext == ".edl":
            fmt = "edl"
        elif ext in [".xml", ".fcpxml"]:
            fmt = "xml"
        else:
            raise ValueError(f"Cannot auto-detect format from extension: {ext}. Use --format edl|xml")

    # Import markers
    fmt = fmt.lower()
    if fmt == "edl":
        markers = import_edl(p_in, fps=fps)
    elif fmt == "xml":
        markers = import_xml(p_in, fps=fps)
    else:
        raise ValueError(f"Unsupported format: {fmt}. Use edl|xml")

    # Infer output path if not specified
    if out is None:
        out_path = p_in.with_suffix(".csv")
    else:
        out_path = Path(out)

    # Save to CSV
    save_markers_csv(markers, out_path)
    print(f"[Import] Read {len(markers)} markers from {fmt.upper()} → {out_path}")


@app.command("export-markers")
def export_markers_cmd(
    in_csv: str = typer.Argument(..., help="Input markers CSV (from transfer-markers)."),
    fmt: str = typer.Option("csv", "--format", case_sensitive=False, help="Export format: csv|json|edl|xml|fcpxml|premiere"),
    preset: str = typer.Option(None, "--preset", case_sensitive=False, help="Built-in preset (fcpxml|premiere|resolve-edl) or custom from config (overrides --format)"),
    out: str = typer.Option(None, "--out", help="Output file path; inferred if omitted."),
    fps: int = typer.Option(30, "--fps", help="Frame rate (EDL/XML timecode)."),
    reel: str = typer.Option("AX", "--reel", help="EDL reel name (2-8 chars)."),
    label: str = typer.Option(None, "--label", help="Optional label for JSON payload."),
):
    """
    Export markers to CSV, JSON, EDL (CMX-3600), XML, FCPXML, or Premiere XML.

    Built-in presets:
      - fcpxml: Final Cut Pro X (modern FCPXML 1.8+)
      - premiere: Adobe Premiere Pro XML
      - resolve-edl: DaVinci Resolve EDL (CMX-3600)

    Custom presets via ~/.mtbsync/config.toml:
      [presets.my-preset]
      format = "fcpxml"
      fps = 24
      reel = "A1"
    """
    from pathlib import Path
    from mtbsync.export import export_markers as _export
    import tomllib

    # Apply preset if provided
    if preset:
        preset_lower = preset.lower()
        # Check built-in presets first
        if preset_lower == "fcpxml":
            fmt = "fcpxml"
        elif preset_lower == "premiere":
            fmt = "premiere"
        elif preset_lower == "resolve-edl":
            fmt = "edl"
        else:
            # Try loading from custom config
            config_path = Path.home() / ".mtbsync" / "config.toml"
            if config_path.exists():
                try:
                    with open(config_path, "rb") as f:
                        config = tomllib.load(f)
                    presets = config.get("presets", {})
                    if preset in presets:
                        preset_config = presets[preset]
                        # Apply preset values
                        fmt = preset_config.get("format", fmt)
                        if "fps" in preset_config:
                            fps = preset_config["fps"]
                        if "reel" in preset_config:
                            reel = preset_config["reel"]
                        if "label" in preset_config:
                            label = preset_config["label"]
                    else:
                        typer.echo(f"Warning: Preset '{preset}' not found in {config_path}", err=True)
                except Exception as e:
                    typer.echo(f"Warning: Could not load presets from {config_path}: {e}", err=True)
            else:
                raise ValueError(f"Unknown preset: {preset}. Use fcpxml|premiere|resolve-edl or define in {config_path}")

    # Validate EDL-specific parameters
    if fmt.lower() == "edl":
        if not (2 <= len(reel) <= 8):
            raise ValueError(f"EDL reel name must be 2-8 characters (got: {reel!r})")
        if fps not in [23, 24, 25, 29, 30, 50, 60]:
            typer.echo(f"Warning: Non-standard fps={fps} for EDL. Common values: 24, 25, 30, 60", err=True)

    p_in = Path(in_csv)
    if out is None:
        # infer extension from format
        ext = fmt.lower()
        if ext == "json": out_path = p_in.with_suffix(".json")
        elif ext == "edl": out_path = p_in.with_suffix(".edl")
        elif ext == "xml": out_path = p_in.with_suffix(".xml")
        elif ext == "fcpxml": out_path = p_in.with_suffix(".fcpxml")
        elif ext == "premiere": out_path = p_in.with_suffix(".xml")
        else: out_path = p_in.with_suffix(".csv")
    else:
        out_path = Path(out)
    path = _export(p_in, out_path, fmt=fmt, fps=fps, reel=reel, label=label)
    print(f"[Export] Wrote {fmt.upper()} → {path}")


@app.command("dashboard")
def dashboard_cmd(
    root: str = typer.Option(".", "--root", help="Root directory to serve (artefacts, CSVs, JSONs)."),
    host: str = typer.Option("127.0.0.1", "--host", help="Bind host."),
    port: int = typer.Option(8000, "--port", help="Bind port."),
    open_browser: bool = typer.Option(True, "--open-browser/--no-open-browser", help="Open default browser."),
    allow_write: bool = typer.Option(False, "--allow-write", help="Permit server-side JSON export."),
):
    """
    Run a zero-dependency local dashboard to inspect artefacts.
    """
    from mtbsync.dashboard import run_dashboard
    run_dashboard(root=root, host=host, port=port, open_browser=open_browser, allow_write=allow_write)


def main() -> None:
    """Entry point for CLI."""
    app()


if __name__ == "__main__":
    main()
