from __future__ import annotations

import csv
import json
import os
import sys
import time
import threading
import webbrowser
from datetime import datetime, timezone
from http.server import SimpleHTTPRequestHandler, ThreadingHTTPServer
from pathlib import Path
from typing import Dict, Any, Optional
from urllib.parse import urlparse, parse_qs

INDEX_HTML = r"""<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<title>MTB Sync Dashboard</title>
<meta name="viewport" content="width=device-width, initial-scale=1" />
<style>
  body { font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif; margin: 20px; }
  h1 { margin: 0 0 16px 0; }
  .cards { display: grid; gap: 12px; grid-template-columns: repeat(auto-fit,minmax(280px,1fr)); }
  .card { border: 1px solid #ddd; border-radius: 8px; padding: 12px; }
  table { border-collapse: collapse; width: 100%; }
  th, td { padding: 6px 8px; border-bottom: 1px solid #eee; text-align: left; font-size: 14px; }
  .muted { color: #555; font-size: 13px; }
  code { background: #f6f8fa; padding: 2px 4px; border-radius: 4px; }
  .sparkline { display: inline-block; margin-left: 8px; vertical-align: middle; }
  .controls { margin: 12px 0; }
  select, button { padding: 6px 10px; margin-right: 8px; font-size: 14px; }
  button { background: #0969da; color: white; border: none; border-radius: 6px; cursor: pointer; }
  button:hover { background: #0550ae; }
  button:disabled { background: #6c757d; cursor: not-allowed; }
  .gauge { height: 8px; background: #eee; position: relative; border-radius: 4px; overflow: hidden; }
  .gauge > span { display:block; height:100%; background:#8bc34a; }
  .gauge.warn > span { background:#ffc107; }
  .gauge.danger > span { background:#f44336; }
</style>
</head>
<body>
  <h1>MTB Sync — Dashboard</h1>
  <p class="muted">Browse alignment artefacts in <code id="rootSpan"></code></p>

  <div class="cards">
    <div class="card">
      <h3>Time-Warp</h3>
      <pre id="timewarpBox" class="muted">Loading...</pre>
    </div>
    <div class="card">
      <h3>Batch Summary</h3>
      <div id="batchBox" class="muted">Loading...</div>
    </div>
    <div class="card">
      <h3>Markers (first 10)
        <div class="controls">
          <select id="markerSelect" style="display:none;"></select>
          <button id="exportBtn" style="display:none;" onclick="exportJSON()">Export JSON</button>
        </div>
      </h3>
      <div id="markersBox" class="muted">Loading...</div>
    </div>
    <div class="card">
      <h3>Telemetry</h3>
      <div id="perfBox" class="muted">Loading...</div>
      <div id="perfLive" class="muted" style="margin-top:8px;"></div>
    </div>
  </div>

  <h2 style="margin-top:20px;">Files</h2>
  <div id="filesBox" class="muted">Loading...</div>

<script>
const el = (sel) => document.querySelector(sel);
let currentMarkerFile = null;

async function loadJSON(path) {
  try {
    const r = await fetch(path);
    if (!r.ok) throw new Error(r.statusText);
    return await r.json();
  } catch (e) { return { error: String(e) }; }
}

function renderTable(headers, rows) {
  if (!rows || !rows.length) return "<p class='muted'>No rows.</p>";
  let thead = "<thead><tr>"+headers.map(h=>`<th>${h}</th>`).join("")+"</tr></thead>";
  let tbody = "<tbody>"+rows.map(r=>"<tr>"+headers.map(h=>`<td>${(r[h]??"")}</td>`).join("")+"</tr>").join("")+"</tbody>";
  return "<table>"+thead+tbody+"</table>";
}

function renderSparkline(values, width=60, height=20) {
  if (!values || !values.length) return "";
  const max = Math.max(...values, 0.01);
  const points = values.map((v,i) => {
    const x = (i / (values.length-1 || 1)) * width;
    const y = height - (v / max) * height;
    return `${x},${y}`;
  }).join(" ");
  return `<svg class="sparkline" width="${width}" height="${height}" viewBox="0 0 ${width} ${height}">
    <polyline fill="none" stroke="#0969da" stroke-width="1.5" points="${points}"/>
  </svg>`;
}

async function loadMarkers(file) {
  const data = file ? await loadJSON(`/api/markers?file=${encodeURIComponent(file)}`) : await loadJSON("/api/markers");
  if (data && data.rows) {
    const headers = data.headers || Object.keys(data.rows[0] || {});
    const sample = data.rows.slice(0, 10);
    el("#markersBox").innerHTML = renderTable(headers, sample);
    currentMarkerFile = file || data.file || null;
  } else {
    el("#markersBox").textContent = JSON.stringify(data, null, 2);
  }
}

async function exportJSON() {
  if (!currentMarkerFile) {
    alert("No marker file selected");
    return;
  }
  try {
    const resp = await fetch("/api/export-json", {
      method: "POST",
      headers: {"Content-Type": "application/json"},
      body: JSON.stringify({in_csv: currentMarkerFile, out_json: currentMarkerFile.replace(".csv", ".json")})
    });
    const result = await resp.json();
    if (resp.ok) {
      alert(`Exported to ${result.path || "JSON file"}`);
    } else {
      alert(`Error: ${result.error || "Export failed"}`);
    }
  } catch (e) {
    alert(`Export error: ${e}`);
  }
}

async function init() {
  el("#rootSpan").textContent = window.location.pathname.replace(/\/$/,"") || "/";
  const [tw, batch, markerList, files, perf] = await Promise.all([
    loadJSON("/api/timewarp"),
    loadJSON("/api/batch"),
    loadJSON("/api/marker-list"),
    loadJSON("/api/files"),
    loadJSON("/api/perf")
  ]);

  el("#timewarpBox").textContent = JSON.stringify(tw, null, 2);

  // Batch summary with sparklines
  if (batch && batch.rows) {
    const headers = batch.headers || Object.keys(batch.rows[0] || {});
    let html = renderTable(headers, batch.rows);

    // Add sparklines for timing columns
    if (batch.rows.length > 1) {
      const retrieval = batch.rows.map(r => parseFloat(r.retrieval_sec) || 0).filter(v => v > 0);
      const warp = batch.rows.map(r => parseFloat(r.warp_sec) || 0).filter(v => v > 0);
      if (retrieval.length) html += "<p class='muted'>Retrieval " + renderSparkline(retrieval) + "</p>";
      if (warp.length) html += "<p class='muted'>Warp " + renderSparkline(warp) + "</p>";
    }
    el("#batchBox").innerHTML = html;
  } else {
    el("#batchBox").textContent = JSON.stringify(batch, null, 2);
  }

  // Marker selector
  if (markerList && markerList.files && markerList.files.length) {
    const sel = el("#markerSelect");
    sel.innerHTML = markerList.files.map(f => `<option value="${f}">${f}</option>`).join("");
    sel.style.display = "inline-block";
    sel.onchange = () => loadMarkers(sel.value);
    el("#exportBtn").style.display = "inline-block";
    await loadMarkers(markerList.files[0]);
  } else {
    await loadMarkers(null);
  }

  // Files
  if (files && files.items) {
    const list = files.items.map(x => `<div><a href="${x.href}" download>${x.name}</a> <span class="muted">(${x.size} B)</span></div>`).join("");
    el("#filesBox").innerHTML = list || "<p class='muted'>No files</p>";
  } else {
    el("#filesBox").textContent = JSON.stringify(files, null, 2);
  }

  // Telemetry
  if (perf && perf.items) {
    const rows = perf.items.map((x,i) => ({
      idx: i+1,
      cmd: x.cmd || "",
      fps: (x.fps ?? (x.timings?.frames_processed && x.timings?.retrieval_sec ? (x.timings.frames_processed / x.timings.retrieval_sec).toFixed(2) : "")),
      wall: (x.timings?.total_sec ?? "").toString(),
      cpu: (x.cpu_pct ?? ""),
      gpu: (x.gpu_util ?? ""),
      vram_mb: (x.gpu_mem_mb ?? ""),
      rss_mb: x.rss_mb?.toFixed ? x.rss_mb.toFixed(1) : x.rss_mb,
      file: x._path || ""
    }));
    const headers = ["idx","cmd","fps","wall","cpu","gpu","vram_mb","rss_mb","file"];
    el("#perfBox").innerHTML = renderTable(headers, rows.slice(-10));
  } else {
    el("#perfBox").textContent = JSON.stringify(perf, null, 2);
  }

  // Live updates via SSE
  try {
    const es = new EventSource("/api/perf/stream");
    el("#perfLive").textContent = "Live: connected";
    es.addEventListener("perf", (ev) => {
      const x = JSON.parse(ev.data);
      // Append a tiny row to the existing table
      const cpu = (x.cpu_pct ?? "");
      const gpu = (x.gpu_util ?? "");
      const vram = (x.gpu_mem_mb ?? "");
      const r = { idx: "•", cmd: x.cmd||"", fps: x.fps ?? "", wall: x.wall ?? "", cpu: cpu, gpu: gpu, vram_mb: vram, rss_mb: x.rss_mb?.toFixed ? x.rss_mb.toFixed(1) : x.rss_mb, file: x.file||"" };
      const existing = el("#perfBox").querySelector("table tbody");
      if (existing) {
        const tr = document.createElement("tr");
        ["idx","cmd","fps","wall","cpu","gpu","vram_mb","rss_mb","file"].forEach(k=>{
          const td = document.createElement("td"); td.textContent = (r[k] ?? "").toString(); tr.appendChild(td);
        });
        existing.appendChild(tr);
      }
    });
    es.addEventListener("error", () => { el("#perfLive").textContent = "Live: disconnected"; });
  } catch (e) {
    el("#perfLive").textContent = "Live: unsupported";
  }

  // Render CPU/GPU gauges for the latest row if present
  function gaugeHTML(val) {
    if (val === null || val === undefined || val === "") return "";
    const v = Math.max(0, Math.min(100, Number(val)));
    const cls = v < 60 ? "gauge" : (v < 85 ? "gauge warn" : "gauge danger");
    return `<div class="${cls}"><span style="width:${v}%;"></span></div>`;
  }
  // Replace numeric cpu/gpu cells with gauges on initial render
  setTimeout(() => {
    const rows = document.querySelectorAll("#perfBox table tbody tr");
    rows.forEach((tr) => {
      const cpuCell = tr.children[4]; const gpuCell = tr.children[5];
      if (cpuCell && cpuCell.textContent && !cpuCell.querySelector(".gauge")) cpuCell.innerHTML = gaugeHTML(cpuCell.textContent);
      if (gpuCell && gpuCell.textContent && !gpuCell.querySelector(".gauge")) gpuCell.innerHTML = gaugeHTML(gpuCell.textContent);
    });
  }, 50);
}
init();
</script>
</body>
</html>
"""

def _read_batch_summary(root: Path) -> Dict[str, Any]:
    path = root / "batch_summary.csv"
    if not path.exists():
        return {"headers": [], "rows": []}
    with open(path, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        rows = list(reader)
    headers = reader.fieldnames or (rows[0].keys() if rows else [])
    return {"headers": list(headers), "rows": rows}

def _list_marker_files(root: Path) -> Dict[str, Any]:
    """List all new_markers*.csv files."""
    candidates = [p for p in root.glob("**/new_markers*.csv") if p.is_file()]
    files = [str(p.relative_to(root)) for p in sorted(candidates, key=lambda p: len(p.parts))]
    return {"files": files}

def _read_markers_any(root: Path, file: Optional[str] = None) -> Dict[str, Any]:
    """Read markers from specific file or auto-select first available."""
    if file:
        # Resolve and ensure the requested path stays within root
        try:
            req = Path(file)
            if req.is_absolute():
                return {"error": "Absolute paths not allowed"}
            path = (root / req).resolve()
            path.relative_to(root.resolve())
        except Exception:
            return {"error": "Invalid path"}
        if not path.exists() or not path.is_file():
            return {"error": f"File not found: {file}"}
    else:
        # Prefer new_markers.csv in root; otherwise list the first *new_markers*.csv found.
        candidates = [p for p in root.glob("**/new_markers*.csv") if p.is_file()]
        if not candidates:
            return {"headers": [], "rows": []}
        path = min(candidates, key=lambda p: len(p.parts))

    try:
        with open(path, newline="", encoding="utf-8") as f:
            reader = csv.DictReader(f)
            rows = list(reader)
        headers = reader.fieldnames or (rows[0].keys() if rows else [])
        return {"headers": list(headers), "rows": rows, "file": str(path.relative_to(root))}
    except Exception as e:
        return {"error": str(e)}

def _read_timewarp_any(root: Path) -> Dict[str, Any]:
    # Prefer timewarp.json in root; else first match underneath.
    candidates = [root / "timewarp.json"] + [p for p in root.glob("**/timewarp.json")]
    for path in candidates:
        if path.exists():
            try:
                return json.loads(path.read_text(encoding="utf-8"))
            except Exception as e:
                return {"error": str(e)}
    return {"ok": False, "error": "timewarp.json not found"}

def _read_perf_any(root: Path, max_items: int = 200) -> Dict[str, Any]:
    """Aggregate perf.json files under root (capped to last N by mtime for performance)."""
    items = []
    for p in root.glob("**/perf.json"):
        if p.is_file():
            try:
                obj = json.loads(p.read_text(encoding="utf-8"))
                obj["_path"] = str(p.relative_to(root))
                obj["_mtime"] = p.stat().st_mtime
                items.append(obj)
            except Exception:
                pass
    # Sort by mtime (newest first) and cap to prevent large responses
    items.sort(key=lambda x: x.get("_mtime", 0), reverse=True)
    items = items[:max_items]
    return {"count": len(items), "items": items}

def _iter_perf_events(root: Path, interval: float = 0.5):
    """
    Poll perf.json files under root and yield SSE lines when a file is new/changed.
    Emits an 'init' event immediately with current count.
    """
    seen: Dict[str, float] = {}
    root = root.resolve()
    # Initial state
    yield f"event: init\ndata: {json.dumps({'time': datetime.now(timezone.utc).isoformat()})}\n\n"
    while True:
        changed = []
        for p in root.glob("**/perf.json"):
            try:
                st = p.stat()
                key = str(p.resolve())
                m = st.st_mtime
                if key not in seen or m > seen[key]:
                    # read a compact summary for the event payload
                    obj = json.loads(p.read_text(encoding="utf-8"))
                    payload = {
                        "file": str(p.relative_to(root)),
                        "cmd": obj.get("cmd"),
                        "fps": obj.get("fps"),
                        "wall": (obj.get("run", {}) or {}).get("wall_sec") or obj.get("timings", {}).get("total_sec"),
                        "rss_mb": obj.get("rss_mb") if obj.get("rss_mb") is not None else (((obj.get("run", {}) or {}).get("rss_bytes") or 0)/1e6),
                        "cpu_pct": obj.get("cpu_pct"),
                        "gpu_util": obj.get("gpu_util"),
                        "gpu_mem_mb": obj.get("gpu_mem_mb"),
                    }
                    changed.append(payload)
                    seen[key] = m
            except Exception:
                continue
        for item in changed:
            yield f"event: perf\ndata: {json.dumps(item)}\n\n"
        time.sleep(interval)

def _list_files(root: Path) -> Dict[str, Any]:
    items = []
    for p in root.glob("**/*"):
        if p.is_file():
            rel = p.relative_to(root)
            items.append({"name": str(rel), "href": f"/{rel.as_posix()}", "size": p.stat().st_size})
    items.sort(key=lambda x: x["name"])
    return {"items": items}

class _Handler(SimpleHTTPRequestHandler):
    allow_write = False  # Class variable set by factory

    def __init__(self, *args, directory: str | None = None, **kwargs):
        super().__init__(*args, directory=directory, **kwargs)

    def do_GET(self):
        root = Path(self.directory or ".").resolve()
        parsed = urlparse(self.path)
        path = parsed.path
        query = parse_qs(parsed.query)

        if path == "/" or path == "/index.html":
            self._send_text(200, INDEX_HTML, "text/html; charset=utf-8")
            return
        if path == "/api/timewarp":
            self._send_json(_read_timewarp_any(root))
            return
        if path == "/api/batch":
            self._send_json(_read_batch_summary(root))
            return
        if path == "/api/marker-list":
            self._send_json(_list_marker_files(root))
            return
        if path == "/api/markers":
            file_param = query.get("file", [None])[0]
            self._send_json(_read_markers_any(root, file=file_param))
            return
        if path == "/api/files":
            self._send_json(_list_files(root))
            return
        if path == "/api/perf":
            self._send_json(_read_perf_any(root))
            return
        if path == "/api/perf/stream":
            self.send_response(200)
            self.send_header("Content-Type", "text/event-stream")
            self.send_header("Cache-Control", "no-cache")
            self.send_header("Connection", "keep-alive")
            self.end_headers()
            try:
                for chunk in _iter_perf_events(root):
                    self.wfile.write(chunk.encode("utf-8"))
                    self.wfile.flush()
            except BrokenPipeError:
                pass
            return

        # Fallback to static files served from root
        return super().do_GET()

    def do_POST(self):
        root = Path(self.directory or ".").resolve()
        parsed = urlparse(self.path)
        path = parsed.path

        if path == "/api/export-json":
            if not self.allow_write:
                self._send_json({"error": "Server export disabled (use --allow-write)"}, code=403)
                return

            try:
                content_length = int(self.headers.get("Content-Length", 0))
                body = self.rfile.read(content_length).decode("utf-8")
                data = json.loads(body)
                in_csv = data.get("in_csv")
                out_json = data.get("out_json")

                if not in_csv or not out_json:
                    self._send_json({"error": "Missing in_csv or out_json"}, code=400)
                    return

                # Resolve and enforce both paths are inside root
                try:
                    in_req = Path(in_csv)
                    out_req = Path(out_json)
                    if in_req.is_absolute() or out_req.is_absolute():
                        self._send_json({"error": "Absolute paths not allowed"}, code=400)
                        return
                    csv_path = (root / in_req).resolve()
                    json_path = (root / out_req).resolve()
                    csv_path.relative_to(root)
                    json_path.relative_to(root)
                except Exception:
                    self._send_json({"error": "Invalid path"}, code=400)
                    return

                if not csv_path.exists():
                    self._send_json({"error": f"CSV file not found: {in_csv}"}, code=404)
                    return

                with open(csv_path, newline="", encoding="utf-8") as f:
                    reader = csv.DictReader(f)
                    rows = list(reader)

                json_path.parent.mkdir(parents=True, exist_ok=True)
                with open(json_path, "w", encoding="utf-8") as f:
                    json.dump({"markers": rows}, f, indent=2)

                self._send_json({"ok": True, "path": str(json_path.relative_to(root))})
            except Exception as e:
                self._send_json({"error": str(e)}, code=500)
            return

        self._send_json({"error": "Not found"}, code=404)

    def _send_json(self, obj: Dict[str, Any], code: int = 200):
        data = json.dumps(obj).encode("utf-8")
        self.send_response(code)
        self.send_header("Content-Type", "application/json; charset=utf-8")
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

    def _send_text(self, code: int, text: str, ctype: str = "text/plain; charset=utf-8"):
        data = text.encode("utf-8")
        self.send_response(code)
        self.send_header("Content-Type", ctype)
        self.send_header("Content-Length", str(len(data)))
        self.end_headers()
        self.wfile.write(data)

def run_dashboard(root: str | Path = ".", host: str = "127.0.0.1", port: int = 8000, open_browser: bool = True, allow_write: bool = False) -> None:
    """
    Serve a simple dashboard rooted at `root`. Ctrl+C to stop.

    Args:
        root: Directory to serve
        host: Bind host
        port: Bind port
        open_browser: Open browser automatically
        allow_write: Allow server-side JSON export via POST /api/export-json
    """
    root = Path(root).resolve()
    if not root.exists():
        raise FileNotFoundError(f"Root not found: {root}")

    # Create handler class with allow_write flag
    class ConfiguredHandler(_Handler):
        pass
    ConfiguredHandler.allow_write = allow_write

    handler = lambda *args, **kwargs: ConfiguredHandler(*args, directory=str(root), **kwargs)  # noqa: E731
    # Use a threaded server so long-lived SSE connections don't block other endpoints.
    httpd = ThreadingHTTPServer((host, port), handler)
    # Ensure worker threads don't block interpreter shutdown on server_close().
    httpd.daemon_threads = True

    url = f"http://{host}:{port}/"
    write_msg = " (write enabled)" if allow_write else ""
    print(f"[Dashboard] Serving {root} at {url}{write_msg}")
    if open_browser:
        threading.Thread(target=lambda: (time.sleep(0.5), webbrowser.open(url)), daemon=True).start()

    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        pass
    finally:
        httpd.server_close()
        print("[Dashboard] Stopped.")
