from __future__ import annotations

import csv
import json
from pathlib import Path
from typing import List, Dict, Optional

# Our canonical input schema is the output of marker_transfer.transfer_markers:
# Columns: marker_id, t_ref, t_new_est, (+ arbitrary metadata columns)

def _read_markers(in_csv: str | Path) -> List[Dict[str, str]]:
    rows: List[Dict[str, str]] = []
    with open(in_csv, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        for r in reader:
            rows.append(r)
    return rows

def export_csv(in_csv: str | Path, out: str | Path) -> Path:
    """Re-emit CSV (identity) to a chosen path. Useful for batch naming."""
    out = Path(out)
    rows = _read_markers(in_csv)
    fieldnames = list(rows[0].keys()) if rows else ["marker_id", "t_ref", "t_new_est"]
    with open(out, "w", newline="", encoding="utf-8") as f:
        w = csv.DictWriter(f, fieldnames=fieldnames)
        w.writeheader()
        w.writerows(rows)
    return out

def export_json(in_csv: str | Path, out: str | Path, label: Optional[str] = None) -> Path:
    out = Path(out)
    rows = _read_markers(in_csv)
    payload = {"label": label or "markers", "markers": rows}
    out.write_text(json.dumps(payload, indent=2), encoding="utf-8")
    return out

def _seconds_to_tc(seconds: float, fps: int) -> str:
    # Format: HH:MM:SS:FF (dropframe not handled; keep simple non-drop)
    if seconds < 0: seconds = 0.0
    total_frames = int(round(seconds * fps))
    ff = total_frames % fps
    total_seconds = total_frames // fps
    ss = total_seconds % 60
    total_minutes = total_seconds // 60
    mm = total_minutes % 60
    hh = total_minutes // 60
    return f"{hh:02d}:{mm:02d}:{ss:02d}:{ff:02d}"

def export_edl(in_csv: str | Path, out: str | Path, fps: int = 30, reel: str = "AX") -> Path:
    """
    Minimal CMX-3600 EDL with each marker as a CUT at the marker time.
    For simplicity, we create zero-length events with identical in/out (display-only).
    Many apps will at least list the events for navigation.
    """
    rows = _read_markers(in_csv)
    out = Path(out)
    lines: List[str] = []
    lines.append("TITLE: MARKERS")
    lines.append(f"FCM: NON-DROP FRAME")  # simple, non-drop
    event_id = 1
    for r in rows:
        name = str(r.get("marker_id", event_id))
        # Use t_new_est if available, else t_ref
        t = float(r.get("t_new_est", r.get("t_ref", "0")) or 0)
        tc = _seconds_to_tc(t, fps)
        # CMX-3600 line: 001  AX       V     C        00:00:00:00 00:00:00:00 00:00:00:00 00:00:00:00
        lines.append(f"{event_id:003d}  {reel:>2}       V     C        {tc} {tc} {tc} {tc}")
        # Comment line with marker label if present
        label = r.get("label") or r.get("name") or name
        lines.append(f"* FROM CLIP NAME: {label}")
        event_id += 1
    out.write_text("\n".join(lines) + "\n", encoding="utf-8")
    return out

def export_xml(in_csv: str | Path, out: str | Path, fps: int = 30) -> Path:
    """
    Very simple Final Cut–style XML (FCPXML-like) containing markers as <marker> entries.
    This is not a full timeline; many NLEs can still parse markers from such XML.
    """
    rows = _read_markers(in_csv)
    out = Path(out)
    # Build a minimal XML by hand (no external deps).
    # Timebase: fps, duration in seconds (optional).
    xml = []
    xml.append('<?xml version="1.0" encoding="UTF-8"?>')
    xml.append('<xmeml version="5">')
    xml.append('  <sequence>')
    xml.append('    <name>Markers</name>')
    xml.append(f'    <rate><timebase>{fps}</timebase><ntsc>FALSE</ntsc></rate>')
    xml.append('    <marker-collection>')
    for r in rows:
        name = r.get("marker_id", "M")
        t = r.get("t_new_est", r.get("t_ref", "0"))
        try:
            t = float(t)
        except Exception:
            t = 0.0
        tc = _seconds_to_tc(t, fps)
        comment = r.get("label", "") or r.get("name", "")
        xml.append('      <marker>')
        xml.append(f'        <name>{_xml_escape(str(name))}</name>')
        xml.append(f'        <comment>{_xml_escape(comment)}</comment>')
        xml.append(f'        <in>{tc}</in>')
        xml.append(f'        <out>{tc}</out>')
        xml.append('      </marker>')
    xml.append('    </marker-collection>')
    xml.append('  </sequence>')
    xml.append('</xmeml>')
    out.write_text("\n".join(xml) + "\n", encoding="utf-8")
    return out

def _xml_escape(s: str) -> str:
    return (s.replace("&", "&amp;")
             .replace("<", "&lt;")
             .replace(">", "&gt;")
             .replace('"', "&quot;")
             .replace("'", "&apos;"))

def export_fcpxml(in_csv: str | Path, out: str | Path, fps: int = 30) -> Path:
    """
    FCPXML export with sequence rate derived from fps. Uses rational frameDuration (e.g. 100/2400s for 24 fps).
    """
    rows = _read_markers(in_csv)
    out = Path(out)

    # Map common fps to FCP format names; fall back to a generic container if unknown.
    fmt_map = {24: "FFVideoFormat1080p24", 25: "FFVideoFormat1080p25", 30: "FFVideoFormat1080p30", 60: "FFVideoFormat1080p60"}
    fmt_name = fmt_map.get(int(round(fps)), "FFVideoFormatRateUndefined")
    # Use rational with small denominator to keep files readable: 100 / (fps*100) seconds.
    denom = max(1, int(round(fps * 100)))
    frame_duration = f"100/{denom}s"

    xml = []
    xml.append('<?xml version="1.0" encoding="UTF-8"?>')
    xml.append('<!DOCTYPE fcpxml>')
    xml.append('<fcpxml version="1.8">')
    xml.append('  <resources>')
    xml.append(f'    <format id="r1" name="{fmt_name}" frameDuration="{frame_duration}" width="1920" height="1080"/>')
    xml.append('  </resources>')
    xml.append('  <library>')
    xml.append('    <event name="Markers">')
    xml.append('      <project name="Markers">')
    xml.append('        <sequence format="r1" duration="0s">')
    xml.append('          <spine>')

    for i, r in enumerate(rows):
        name = r.get("marker_id", f"M{i+1}")
        t = r.get("t_new_est", r.get("t_ref", "0"))
        try:
            t_sec = float(t)
        except Exception:
            t_sec = 0.0

        # Convert to rational time (frames/fps)
        frames = int(round(t_sec * fps))
        rational_time = f"{frames}/{fps}s"

        comment = r.get("label", "") or r.get("name", "")

        # FCPXML marker element
        xml.append(f'            <marker start="{rational_time}" duration="1/{fps}s" value="{_xml_escape(str(name))}">')
        if comment:
            xml.append(f'              <note>{_xml_escape(comment)}</note>')
        xml.append('            </marker>')

    xml.append('          </spine>')
    xml.append('        </sequence>')
    xml.append('      </project>')
    xml.append('    </event>')
    xml.append('  </library>')
    xml.append('</fcpxml>')

    out.write_text("\n".join(xml) + "\n", encoding="utf-8")
    return out

def export_premiere_xml(in_csv: str | Path, out: str | Path, fps: int = 30) -> Path:
    """
    Export markers as Adobe Premiere Pro XML format.
    Based on FCP 7 XML schema but with Premiere-specific extensions.
    """
    rows = _read_markers(in_csv)
    out = Path(out)

    # Premiere XML uses frame numbers for timing
    xml = []
    xml.append('<?xml version="1.0" encoding="UTF-8"?>')
    xml.append('<!DOCTYPE xmeml>')
    xml.append('<xmeml version="4">')
    xml.append('  <sequence id="sequence-1">')
    xml.append('    <name>Markers</name>')
    xml.append('    <rate>')
    xml.append(f'      <timebase>{fps}</timebase>')
    xml.append('      <ntsc>FALSE</ntsc>')
    xml.append('    </rate>')
    xml.append('    <timecode>')
    xml.append('      <rate>')
    xml.append(f'        <timebase>{fps}</timebase>')
    xml.append('        <ntsc>FALSE</ntsc>')
    xml.append('      </rate>')
    xml.append('      <string>00:00:00:00</string>')
    xml.append('      <frame>0</frame>')
    xml.append('      <displayformat>NDF</displayformat>')
    xml.append('    </timecode>')

    for r in rows:
        name = r.get("marker_id", "M")
        t = r.get("t_new_est", r.get("t_ref", "0"))
        try:
            t_sec = float(t)
        except Exception:
            t_sec = 0.0

        frames = int(round(t_sec * fps))
        comment = r.get("label", "") or r.get("name", "")

        xml.append('    <marker>')
        xml.append(f'      <in>{frames}</in>')
        xml.append(f'      <out>{frames}</out>')
        xml.append(f'      <name>{_xml_escape(str(name))}</name>')
        if comment:
            xml.append(f'      <comment>{_xml_escape(comment)}</comment>')
        xml.append('    </marker>')

    xml.append('  </sequence>')
    xml.append('</xmeml>')

    out.write_text("\n".join(xml) + "\n", encoding="utf-8")
    return out

def export_markers(
    in_csv: str | Path,
    out: str | Path,
    fmt: str = "csv",
    fps: int = 30,
    reel: str = "AX",
    label: str | None = None,
) -> Path:
    """
    Export markers to the requested format.
    fmt ∈ {csv, json, edl, xml, fcpxml, premiere}
    """
    fmt = (fmt or "csv").lower()
    if fmt == "csv":
        return export_csv(in_csv, out)
    if fmt == "json":
        return export_json(in_csv, out, label=label)
    if fmt == "edl":
        return export_edl(in_csv, out, fps=fps, reel=reel)
    if fmt == "xml":
        return export_xml(in_csv, out, fps=fps)
    if fmt == "fcpxml":
        return export_fcpxml(in_csv, out, fps=fps)
    if fmt == "premiere":
        return export_premiere_xml(in_csv, out, fps=fps)
    raise ValueError(f"Unsupported export format: {fmt}")
