"""Keyframe processing utilities."""

import cv2
import numpy as np


def resize_max(frame: np.ndarray, max_dim: int = 960) -> np.ndarray:
    """
    Resize frame so that the maximum dimension is max_dim pixels.

    Preserves aspect ratio. If both dimensions are already smaller than max_dim,
    returns the frame unchanged.

    Args:
        frame: Input frame as numpy array (height, width, channels)
        max_dim: Maximum dimension in pixels (default: 960)

    Returns:
        Resized frame with same number of channels

    Raises:
        ValueError: If frame is invalid or max_dim is not positive
    """
    if max_dim <= 0:
        raise ValueError(f"max_dim must be positive, got: {max_dim}")

    if frame is None or frame.size == 0:
        raise ValueError("Frame is empty or None")

    if len(frame.shape) < 2:
        raise ValueError(f"Frame must be at least 2D, got shape: {frame.shape}")

    height, width = frame.shape[:2]

    # Check if resize is needed
    if height <= max_dim and width <= max_dim:
        return frame

    # Calculate new dimensions preserving aspect ratio
    if height > width:
        new_height = max_dim
        new_width = int(width * max_dim / height)
    else:
        new_width = max_dim
        new_height = int(height * max_dim / width)

    # Use INTER_AREA for downsampling (best quality for shrinking)
    resized = cv2.resize(frame, (new_width, new_height), interpolation=cv2.INTER_AREA)

    return resized
