from __future__ import annotations

import csv
import re
import xml.etree.ElementTree as ET
from pathlib import Path
from typing import List, Dict, Optional

def _tc_to_seconds(tc: str, fps: int = 30) -> float:
    """
    Convert timecode HH:MM:SS:FF to seconds.

    Args:
        tc: Timecode string in format HH:MM:SS:FF
        fps: Frame rate (default 30)

    Returns:
        Time in seconds
    """
    parts = tc.split(":")
    if len(parts) != 4:
        return 0.0
    try:
        hh, mm, ss, ff = map(int, parts)
        total_seconds = hh * 3600 + mm * 60 + ss
        total_seconds += ff / fps
        return total_seconds
    except (ValueError, ZeroDivisionError):
        return 0.0

def import_edl(in_path: str | Path, fps: int = 30) -> List[Dict[str, str]]:
    """
    Import markers from CMX-3600 EDL format.

    Args:
        in_path: Path to EDL file
        fps: Frame rate for timecode conversion (default 30)

    Returns:
        List of marker dictionaries with marker_id, t_ref, label
    """
    in_path = Path(in_path)
    if not in_path.exists():
        raise FileNotFoundError(f"EDL file not found: {in_path}")

    markers = []
    lines = in_path.read_text(encoding="utf-8").splitlines()

    i = 0
    while i < len(lines):
        line = lines[i].strip()

        # EDL event line format: 001  AX       V     C        00:00:00:00 00:00:00:00 00:00:00:00 00:00:00:00
        # Pattern: event_num  reel  track  edit_type  src_in src_out rec_in rec_out
        event_match = re.match(r"^(\d+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)", line)

        if event_match:
            event_num = event_match.group(1)
            rec_in_tc = event_match.group(7)  # Record in timecode

            # Parse time from record-in timecode
            t_ref = _tc_to_seconds(rec_in_tc, fps)

            # Look for comment line with marker label (next line starting with *)
            label = f"Marker_{event_num}"
            if i + 1 < len(lines):
                next_line = lines[i + 1].strip()
                if next_line.startswith("*"):
                    # Extract label from comment line
                    # Format: * FROM CLIP NAME: label
                    comment_match = re.search(r"\*\s*FROM CLIP NAME:\s*(.+)", next_line)
                    if comment_match:
                        label = comment_match.group(1).strip()
                    i += 1  # Skip comment line

            markers.append({
                "marker_id": event_num,
                "t_ref": str(t_ref),
                "label": label
            })

        i += 1

    return markers

def import_xml(in_path: str | Path, fps: int = 30) -> List[Dict[str, str]]:
    """
    Import markers from XML format (Premiere xmeml or FCPXML).
    Attempts to detect format and parse accordingly.

    Args:
        in_path: Path to XML file
        fps: Frame rate for timecode/frame conversion (default 30)

    Returns:
        List of marker dictionaries with marker_id, t_ref, label
    """
    in_path = Path(in_path)
    if not in_path.exists():
        raise FileNotFoundError(f"XML file not found: {in_path}")

    try:
        tree = ET.parse(in_path)
        root = tree.getroot()
    except ET.ParseError as e:
        raise ValueError(f"Invalid XML file: {e}")

    markers = []

    # Detect format by root element
    if root.tag == "fcpxml":
        # Modern FCPXML 1.8+ format
        markers = _parse_fcpxml(root, fps)
    elif root.tag == "xmeml":
        # Premiere/FCP7 xmeml format
        markers = _parse_xmeml(root, fps)
    else:
        raise ValueError(f"Unsupported XML format. Root element: {root.tag}")

    return markers

def _parse_fcpxml(root: ET.Element, fps: int) -> List[Dict[str, str]]:
    """Parse FCPXML 1.8+ format for markers."""
    markers = []
    marker_num = 1

    # Find all marker elements (can be in spine or other locations)
    for marker_elem in root.findall(".//marker"):
        start_attr = marker_elem.get("start", "0/30s")
        value_attr = marker_elem.get("value", f"M{marker_num}")

        # Parse rational time format: "frames/fps s" (e.g., "30/30s", "60/30s")
        # Extract frames and fps from format like "30/30s"
        time_match = re.match(r"(\d+)/(\d+)s", start_attr)
        if time_match:
            frames = int(time_match.group(1))
            time_fps = int(time_match.group(2))
            t_ref = frames / time_fps
        else:
            t_ref = 0.0

        # Look for note element
        note_elem = marker_elem.find("note")
        label = note_elem.text if note_elem is not None and note_elem.text else value_attr

        markers.append({
            "marker_id": value_attr,
            "t_ref": str(t_ref),
            "label": label
        })
        marker_num += 1

    return markers

def _parse_xmeml(root: ET.Element, fps: int) -> List[Dict[str, str]]:
    """Parse Premiere/FCP7 xmeml format for markers."""
    markers = []

    # Find sequence/marker elements
    for seq in root.findall(".//sequence"):
        # Get timebase if available
        rate_elem = seq.find(".//rate/timebase")
        if rate_elem is not None and rate_elem.text:
            try:
                fps = int(rate_elem.text)
            except ValueError:
                pass

        # Find marker collection or individual markers
        marker_collection = seq.find(".//marker-collection")
        if marker_collection is not None:
            # FCP7 style with marker-collection
            for marker_elem in marker_collection.findall("marker"):
                markers.append(_parse_marker_element(marker_elem, fps))
        else:
            # Premiere style - check for multiple separate marker elements first
            marker_elements = seq.findall(".//marker")
            if marker_elements:
                # Check if each marker has direct children (new format with separate blocks)
                first_marker = marker_elements[0]
                has_own_children = (first_marker.find("in") is not None and
                                    first_marker.find("name") is not None)

                if has_own_children:
                    # New format: each marker is a separate block
                    for marker_elem in marker_elements:
                        markers.append(_parse_marker_element(marker_elem, fps))
                else:
                    # Old format: single marker container with multiple in/out/name sets
                    marker_container = marker_elements[0]
                    names = marker_container.findall("name")
                    ins = marker_container.findall("in")
                    outs = marker_container.findall("out")
                    comments = marker_container.findall("comment")

                    for i, (name_elem, in_elem) in enumerate(zip(names, ins)):
                        marker_id = name_elem.text if name_elem.text else f"M{i+1}"

                        # Parse frame number
                        if in_elem.text:
                            try:
                                frames = int(in_elem.text)
                                t_ref = frames / fps
                            except ValueError:
                                t_ref = 0.0
                        else:
                            t_ref = 0.0

                        # Get comment if available
                        label = marker_id
                        if i < len(comments) and comments[i].text:
                            label = comments[i].text

                        markers.append({
                            "marker_id": marker_id,
                            "t_ref": str(t_ref),
                            "label": label
                        })

    return markers

def _parse_marker_element(marker_elem: ET.Element, fps: int) -> Dict[str, str]:
    """Parse a single marker element from xmeml."""
    name_elem = marker_elem.find("name")
    in_elem = marker_elem.find("in")
    comment_elem = marker_elem.find("comment")

    marker_id = name_elem.text if name_elem is not None and name_elem.text else "M"

    # Parse timecode or frame number
    if in_elem is not None and in_elem.text:
        # Could be timecode (HH:MM:SS:FF) or frame number
        if ":" in in_elem.text:
            t_ref = _tc_to_seconds(in_elem.text, fps)
        else:
            try:
                frames = int(in_elem.text)
                t_ref = frames / fps
            except ValueError:
                t_ref = 0.0
    else:
        t_ref = 0.0

    label = comment_elem.text if comment_elem is not None and comment_elem.text else marker_id

    return {
        "marker_id": marker_id,
        "t_ref": str(t_ref),
        "label": label
    }

def save_markers_csv(markers: List[Dict[str, str]], out_path: str | Path) -> Path:
    """
    Save markers to CSV format.

    Args:
        markers: List of marker dictionaries
        out_path: Output CSV path

    Returns:
        Path to written CSV file
    """
    out_path = Path(out_path)

    if not markers:
        # Write empty CSV with headers
        with open(out_path, "w", newline="", encoding="utf-8") as f:
            writer = csv.DictWriter(f, fieldnames=["marker_id", "t_ref", "label"])
            writer.writeheader()
        return out_path

    # Get all unique keys from markers
    fieldnames = list(markers[0].keys())

    with open(out_path, "w", newline="", encoding="utf-8") as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames)
        writer.writeheader()
        writer.writerows(markers)

    return out_path
