"""
Marker Transfer utilities for MTB Sync.

This module applies the fitted affine time-warp to transfer or resample markers
from a reference video onto a newly aligned video timeline.

Workflow:
1. Load `timewarp.json` (and optionally `gps_alignment.json`).
2. Apply affine mapping t_ref = a * t_new + b.
3. Produce a CSV or list of (marker_id, t_ref, t_new_est).
4. Optionally smooth or bound estimates using known video duration.

Example:
    >>> from mtbsync.match.marker_transfer import transfer_markers
    >>> transfer_markers("ref_markers.csv", "cache/timewarp.json", out="cache/new_markers.csv")
"""

from __future__ import annotations
import json
import csv
from pathlib import Path
from typing import List, Dict, Tuple

def load_timewarp_json(path: str | Path) -> Dict:
    """Load timewarp.json and return dict with a, b, and metadata."""
    p = Path(path)
    with open(p, "r", encoding="utf-8") as f:
        data = json.load(f)
    params = data.get("params", {})
    return {"a": float(params.get("a", 1.0)), "b": float(params.get("b", 0.0)), "meta": data}

def apply_warp(a: float, b: float, t_ref: float) -> float:
    """Apply affine warp mapping t_new = (t_ref - b) / a."""
    return (t_ref - b) / a

def transfer_markers(ref_csv: str | Path, timewarp_json: str | Path, out: str | Path) -> Path:
    """Transfer reference markers to new-video timeline.

    Args:
        ref_csv: CSV with at least columns [marker_id, t_ref].
        timewarp_json: Path to timewarp.json artefact.
        out: Output CSV path for new markers.

    Returns:
        Path to the written CSV.
    """
    tw = load_timewarp_json(timewarp_json)
    a, b = tw["a"], tw["b"]

    p_in, p_out = Path(ref_csv), Path(out)
    out_rows: List[Dict[str, any]] = []

    with open(p_in, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        for row in reader:
            marker_id = row.get("marker_id") or row.get("id") or str(len(out_rows))
            t_ref = float(row["t_ref"])
            t_new = apply_warp(a, b, t_ref)
            extra = {k: v for k, v in row.items() if k not in {"marker_id", "id", "t_ref"}}
            out_rows.append({"marker_id": marker_id, "t_ref": t_ref, "t_new_est": t_new, **extra})

    fieldnames = list(out_rows[0].keys()) if out_rows else ["marker_id", "t_ref", "t_new_est"]
    with open(p_out, "w", newline="", encoding="utf-8") as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames)
        writer.writeheader()
        writer.writerows(out_rows)

    print(f"[Marker-Transfer] {len(out_rows)} markers written to {p_out}")
    return p_out

def plot_marker_overlay(ref_csv: str | Path, new_csv: str | Path, out_png: str | Path):
    """Plot simple visual comparison of reference vs transferred markers."""
    import csv
    try:
        import matplotlib
        # Use default interactive backend if available here; viewer will override to Agg for headless.
        import matplotlib.pyplot as plt
    except Exception as e:
        raise RuntimeError("matplotlib is required for --plot-overlay; install with `pip install matplotlib`.") from e
    ref_ts, new_ts = [], []
    with open(ref_csv, newline="", encoding="utf-8") as f:
        ref_reader = csv.DictReader(f)
        for r in ref_reader:
            ref_ts.append(float(r["t_ref"]))
    with open(new_csv, newline="", encoding="utf-8") as f:
        new_reader = csv.DictReader(f)
        for r in new_reader:
            new_ts.append(float(r["t_new_est"]))
    plt.figure(figsize=(8, 3))
    plt.scatter(ref_ts, [1]*len(ref_ts), label="Reference", marker="o")
    plt.scatter(new_ts, [0]*len(new_ts), label="Transferred", marker="x", color="red")
    plt.yticks([0, 1], ["New", "Ref"])
    plt.xlabel("Time (s)")
    plt.title("Marker Transfer Preview")
    plt.legend()
    plt.tight_layout()
    plt.savefig(out_png)
    plt.close()
    print(f"[Marker-Transfer] Overlay written to {out_png}")
    return out_png
