from __future__ import annotations

import csv
import json
from dataclasses import dataclass
from pathlib import Path
from typing import Optional, List, Tuple

import matplotlib.pyplot as plt

# Optional GUI imports (guarded)
try:
    import tkinter as tk
    from tkinter import filedialog, messagebox
    TK_AVAILABLE = True
except Exception:
    TK_AVAILABLE = False

# Reuse existing transfer utilities
from mtbsync.match.marker_transfer import (
    load_timewarp_json,
    transfer_markers,
    plot_marker_overlay,  # used in headless mode
)


@dataclass
class ViewerState:
    ref_markers: Optional[Path] = None
    new_markers: Optional[Path] = None
    timewarp_json: Optional[Path] = None
    fig: Optional[plt.Figure] = None
    ax: Optional[plt.Axes] = None


def _read_ref_times(path: Path) -> List[float]:
    with open(path, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        return [float(r["t_ref"]) for r in reader]


def _read_new_times(path: Path) -> List[float]:
    with open(path, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        return [float(r["t_new_est"]) for r in reader]


def _render_plot(ax: plt.Axes, ref_ts: List[float], new_ts: List[float]) -> None:
    ax.clear()
    ax.scatter(ref_ts, [1] * len(ref_ts), marker="o", label="Reference")
    ax.scatter(new_ts, [0] * len(new_ts), marker="x", label="Transferred")
    ax.set_yticks([0, 1], ["New", "Ref"])
    ax.set_xlabel("Time (s)")
    ax.set_title("Synchronisation Viewer")
    ax.legend(loc="best")
    ax.grid(True, alpha=0.25)


def render_overlay_headless(
    ref_csv: Path, timewarp_json: Path, out_png: Path, new_csv: Optional[Path] = None
) -> Path:
    """
    Headless render: if new_csv not provided, it will be generated next to out_png as a temp.
    Returns path to the saved PNG.
    """
    # Force non-interactive backend to avoid TkAgg import on headless systems
    try:
        import matplotlib
        matplotlib.use("Agg", force=True)
    except Exception:
        # If matplotlib is entirely missing, plot_marker_overlay will raise a clear error
        pass
    if new_csv is None:
        new_csv = out_png.with_name(out_png.stem.replace("_overlay", "") + "_tmp_new.csv")
        transfer_markers(ref_csv, timewarp_json, new_csv)
    plot_marker_overlay(ref_csv, new_csv, out_png)
    return out_png


def launch_viewer(
    ref_markers: Optional[str | Path] = None,
    timewarp_json: Optional[str | Path] = None,
    new_markers: Optional[str | Path] = None,
) -> None:
    """
    Launch an interactive Tk viewer for marker alignment.
    If Tk is unavailable, raises a clear RuntimeError.
    """
    if not TK_AVAILABLE:
        raise RuntimeError("Tkinter not available in this environment; use --headless to render PNG.")

    # Import TkAgg backend only when actually launching GUI
    try:
        from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
    except ImportError:
        raise RuntimeError("Matplotlib TkAgg backend not available; use --headless to render PNG.")

    st = ViewerState(
        ref_markers=Path(ref_markers) if ref_markers else None,
        new_markers=Path(new_markers) if new_markers else None,
        timewarp_json=Path(timewarp_json) if timewarp_json else None,
    )

    root = tk.Tk()
    root.title("MTB Sync — Viewer")

    # Top controls
    frm = tk.Frame(root)
    frm.pack(side=tk.TOP, fill=tk.X, padx=8, pady=6)

    def pick_ref():
        p = filedialog.askopenfilename(title="Select reference markers CSV", filetypes=[("CSV", "*.csv")])
        if p:
            st.ref_markers = Path(p)
            lbl_ref_val.config(text=str(st.ref_markers))

    def pick_tw():
        p = filedialog.askopenfilename(title="Select timewarp.json", filetypes=[("JSON", "*.json")])
        if p:
            st.timewarp_json = Path(p)
            lbl_tw_val.config(text=str(st.timewarp_json))

    def pick_new():
        p = filedialog.askopenfilename(title="Select transferred markers CSV (optional)", filetypes=[("CSV", "*.csv")])
        if p:
            st.new_markers = Path(p)
            lbl_new_val.config(text=str(st.new_markers))

    tk.Button(frm, text="Ref CSV…", command=pick_ref).grid(row=0, column=0, padx=4, pady=2, sticky="w")
    lbl_ref_val = tk.Label(frm, text=str(st.ref_markers) if st.ref_markers else "—")
    lbl_ref_val.grid(row=0, column=1, padx=4, sticky="w")

    tk.Button(frm, text="Timewarp…", command=pick_tw).grid(row=1, column=0, padx=4, pady=2, sticky="w")
    lbl_tw_val = tk.Label(frm, text=str(st.timewarp_json) if st.timewarp_json else "—")
    lbl_tw_val.grid(row=1, column=1, padx=4, sticky="w")

    tk.Button(frm, text="New CSV…", command=pick_new).grid(row=2, column=0, padx=4, pady=2, sticky="w")
    lbl_new_val = tk.Label(frm, text=str(st.new_markers) if st.new_markers else "—")
    lbl_new_val.grid(row=2, column=1, padx=4, sticky="w")

    def plot_now():
        if not st.ref_markers or not st.timewarp_json:
            messagebox.showwarning("Missing inputs", "Please select at least Ref CSV and timewarp.json.")
            return
        if not st.new_markers:
            # auto-generate new markers next to ref file
            st.new_markers = st.ref_markers.with_name("new_markers_preview.csv")
            transfer_markers(st.ref_markers, st.timewarp_json, st.new_markers)

        ref_ts = _read_ref_times(st.ref_markers)
        new_ts = _read_new_times(st.new_markers)
        _render_plot(st.ax, ref_ts, new_ts)
        st.fig.tight_layout()
        canvas.draw()

    def save_png():
        if st.fig is None:
            return
        p = filedialog.asksaveasfilename(defaultextension=".png", filetypes=[("PNG", "*.png")])
        if p:
            st.fig.savefig(p)
            messagebox.showinfo("Saved", f"Overlay saved to {p}")

    tk.Button(frm, text="Plot", command=plot_now).grid(row=3, column=0, padx=4, pady=6, sticky="w")
    tk.Button(frm, text="Save PNG…", command=save_png).grid(row=3, column=1, padx=4, pady=6, sticky="w")

    # Figure area
    st.fig, st.ax = plt.subplots(figsize=(8, 3))
    canvas = FigureCanvasTkAgg(st.fig, master=root)
    canvas.get_tk_widget().pack(side=tk.TOP, fill=tk.BOTH, expand=True)

    # Initial draw (empty)
    _render_plot(st.ax, [], [])
    st.fig.tight_layout()
    canvas.draw()

    root.mainloop()
