"""Test batch --no-gpu flag disables GPU telemetry probes."""
from pathlib import Path
import json
from unittest.mock import patch
import pandas as pd


def test_batch_no_gpu_disables_gpu_probe(tmp_path: Path):
    """Ensure batch --no-gpu writes perf.json with gpu fields = None."""
    from mtbsync.batch import run_batch

    # Setup test directory structure
    job_dir = tmp_path / "job"
    job_dir.mkdir()
    (job_dir / "foo_ref.mp4").write_text("dummy")
    (job_dir / "foo_new.mp4").write_text("dummy")

    out_root = tmp_path / "out"

    # Mock retrieve_coarse_pairs to avoid ffmpeg dependency
    def mock_retrieve(**kwargs):
        return pd.DataFrame({"t_new": [1.0], "t_ref": [1.0], "score": [0.9], "n_matches": [50]}), {
            "retrieval_sec": 0.01,
            "warp_sec": 0.01,
            "total_sec": 0.02,
            "frames_processed": 10,
        }

    # Mock discover_pairs to return our test pair
    def mock_discover(input_dir, ref_suffix, new_suffix):
        return [(job_dir / "foo_ref.mp4", job_dir / "foo_new.mp4")]

    # Mock attach_sidecar_files to avoid filesystem scanning
    def mock_attach(base_dir, ref, new, **kwargs):
        return None, None, None

    with patch("mtbsync.match.retrieval.retrieve_coarse_pairs", mock_retrieve):
        with patch("mtbsync.batch.discover_pairs", mock_discover):
            with patch("mtbsync.batch.attach_sidecar_files", mock_attach):
                # Run batch with --no-gpu
                results = run_batch(
                    input_dir=str(job_dir),
                    out_root=str(out_root),
                    ref_suffix="_ref.mp4",
                    new_suffix="_new.mp4",
                    dry_run=False,
                    no_gpu=True,
                    threads=1,
                    fast=False,
                )

    # Verify perf.json exists and has None gpu fields
    perf_path = out_root / "foo" / "perf.json"
    assert perf_path.exists(), f"Expected perf.json at {perf_path}"

    perf = json.loads(perf_path.read_text())
    assert "gpu_util" in perf
    assert "gpu_mem_mb" in perf
    # When --no-gpu is set, these should be None (disabled)
    assert perf["gpu_util"] is None
    assert perf["gpu_mem_mb"] is None
    assert perf["ok"] is True


def test_batch_gpu_enabled_by_default(tmp_path: Path, monkeypatch):
    """Ensure batch without --no-gpu attempts GPU probe (gracefully fails if NVML absent)."""
    from mtbsync.batch import run_batch

    # Setup test directory structure
    job_dir = tmp_path / "job"
    job_dir.mkdir()
    (job_dir / "bar_ref.mp4").write_text("dummy")
    (job_dir / "bar_new.mp4").write_text("dummy")

    out_root = tmp_path / "out2"

    # Mock retrieve_coarse_pairs
    def mock_retrieve(**kwargs):
        return pd.DataFrame({"t_new": [1.0], "t_ref": [1.0], "score": [0.9], "n_matches": [50]}), {
            "retrieval_sec": 0.01,
            "warp_sec": 0.01,
            "total_sec": 0.02,
            "frames_processed": 10,
        }

    # Mock discover_pairs
    def mock_discover(input_dir, ref_suffix, new_suffix):
        return [(job_dir / "bar_ref.mp4", job_dir / "bar_new.mp4")]

    # Mock attach_sidecar_files
    def mock_attach(base_dir, ref, new, **kwargs):
        return None, None, None

    # Mock pynvml to be None (simulating no NVML installed)
    monkeypatch.setattr("mtbsync.telemetry.pynvml", None, raising=False)

    with patch("mtbsync.match.retrieval.retrieve_coarse_pairs", mock_retrieve):
        with patch("mtbsync.batch.discover_pairs", mock_discover):
            with patch("mtbsync.batch.attach_sidecar_files", mock_attach):
                # Run batch without --no-gpu (default: gpu enabled)
                results = run_batch(
                    input_dir=str(job_dir),
                    out_root=str(out_root),
                    ref_suffix="_ref.mp4",
                    new_suffix="_new.mp4",
                    dry_run=False,
                    no_gpu=False,  # GPU probe enabled
                    threads=1,
                    fast=False,
                )

    # Verify perf.json exists
    perf_path = out_root / "bar" / "perf.json"
    assert perf_path.exists()

    perf = json.loads(perf_path.read_text())
    # GPU fields should be None because NVML is not available (mocked as None)
    # But the code attempted to probe (enable_gpu=True)
    assert "gpu_util" in perf
    assert "gpu_mem_mb" in perf
    # With no NVML, graceful fallback gives None
    assert perf["gpu_util"] is None
    assert perf["gpu_mem_mb"] is None
