"""Tests for CLI argument parsing to prevent regression."""

from typer.testing import CliRunner

from mtbsync.cli import app

runner = CliRunner()


def test_sync_help_runs():
    """Test that sync help command runs successfully."""
    r = runner.invoke(app, ["sync", "--help"])
    assert r.exit_code == 0
    assert "sync" in r.stdout.lower()


def test_index_help_runs():
    """Test that index help command runs successfully."""
    r = runner.invoke(app, ["index", "--help"])
    assert r.exit_code == 0
    assert "index" in r.stdout.lower()


def test_index_parses_options_types():
    """Test that index command parses option types correctly (not as boolean flags)."""
    r = runner.invoke(
        app,
        [
            "index",
            "--reference",
            "data/ref.mp4",
            "--fps",
            "3.0",
            "--out",
            "cache/ref_index.npz",
            "--n-features",
            "1500",
            # --no-clahe present => bool flag toggles
            "--no-clahe",
        ],
    )
    # We only check it reaches argument validation; command will fail on missing files.
    # But importantly, Typer should not error by treating non-bools as flags.
    # Exit code 0 = success, 1 = application error (e.g., file not found), 2 = argument parsing error
    assert r.exit_code in (0, 1), f"Got exit code {r.exit_code}, stderr: {r.stdout}"
    # Should not have "unexpected extra arguments" error
    assert "unexpected extra arguments" not in r.stdout.lower()


def test_sync_parses_options_types():
    """Test that sync command parses option types correctly."""
    r = runner.invoke(
        app,
        [
            "sync",
            "--reference",
            "data/ref.mp4",
            "--new",
            "data/new.mp4",
            "--index",
            "cache/ref_index.npz",
            "--out",
            "cache/pairs.csv",
            "--fps",
            "3.0",
            "--top-k",
            "5",
            "--no-refine",
        ],
    )
    # Should reach application logic, not fail on argument parsing
    assert r.exit_code in (0, 1), f"Got exit code {r.exit_code}, stderr: {r.stdout}"
    assert "unexpected extra arguments" not in r.stdout.lower()


def test_app_has_multiple_commands():
    """Test that the app exposes multiple commands."""
    r = runner.invoke(app, ["--help"])
    assert r.exit_code == 0
    # Check for main commands
    assert "index" in r.stdout.lower()
    assert "sync" in r.stdout.lower()
    assert "warp" in r.stdout.lower()
