import csv
import json
import socket
import threading
import time
from pathlib import Path
from urllib.request import urlopen, Request
from urllib.error import HTTPError

from mtbsync.dashboard import run_dashboard


def _free_port():
    s = socket.socket()
    s.bind(("127.0.0.1", 0))
    addr, port = s.getsockname()
    s.close()
    return port


def test_marker_list_and_selector(tmp_path: Path):
    """Test /api/marker-list returns multiple marker files and /api/markers?file= selects specific one."""
    # Create multiple marker files
    (tmp_path / "new_markers_A.csv").write_text("marker_id,t_ref,t_new_est\nM1,0.0,0.5\n", encoding="utf-8")
    (tmp_path / "new_markers_B.csv").write_text("marker_id,t_ref,t_new_est\nM2,1.0,1.5\n", encoding="utf-8")
    sub = tmp_path / "sub"
    sub.mkdir()
    (sub / "new_markers_C.csv").write_text("marker_id,t_ref,t_new_est\nM3,2.0,2.5\n", encoding="utf-8")

    port = _free_port()
    thr = threading.Thread(
        target=run_dashboard,
        kwargs={"root": tmp_path, "host": "127.0.0.1", "port": port, "open_browser": False, "allow_write": False},
        daemon=True,
    )
    thr.start()
    time.sleep(0.6)

    base = f"http://127.0.0.1:{port}"

    # Test /api/marker-list
    with urlopen(base + "/api/marker-list") as r:
        marker_list = json.loads(r.read().decode("utf-8"))
    assert "files" in marker_list
    assert len(marker_list["files"]) == 3
    assert any("new_markers_A.csv" in f for f in marker_list["files"])

    # Test /api/markers?file=...
    file_to_read = marker_list["files"][0]
    with urlopen(base + f"/api/markers?file={file_to_read}") as r:
        markers = json.loads(r.read().decode("utf-8"))
    assert "rows" in markers
    assert isinstance(markers["rows"], list)
    assert len(markers["rows"]) == 1


def test_export_json_denied_without_flag(tmp_path: Path):
    """Test POST /api/export-json returns 403 when allow_write=False."""
    (tmp_path / "new_markers.csv").write_text("marker_id,t_ref,t_new_est\nA,0.0,0.5\n", encoding="utf-8")

    port = _free_port()
    thr = threading.Thread(
        target=run_dashboard,
        kwargs={"root": tmp_path, "host": "127.0.0.1", "port": port, "open_browser": False, "allow_write": False},
        daemon=True,
    )
    thr.start()
    time.sleep(0.6)

    base = f"http://127.0.0.1:{port}"

    # Attempt POST /api/export-json
    payload = json.dumps({"in_csv": "new_markers.csv", "out_json": "new_markers.json"}).encode("utf-8")
    req = Request(base + "/api/export-json", data=payload, headers={"Content-Type": "application/json"}, method="POST")

    try:
        with urlopen(req) as r:
            result = json.loads(r.read().decode("utf-8"))
        assert False, "Expected 403 error but request succeeded"
    except HTTPError as e:
        assert e.code == 403
        result = json.loads(e.read().decode("utf-8"))
        assert "error" in result


def test_export_json_allowed_with_flag(tmp_path: Path):
    """Test POST /api/export-json creates JSON file when allow_write=True."""
    (tmp_path / "new_markers.csv").write_text("marker_id,t_ref,t_new_est\nA,0.0,0.5\nB,1.0,1.5\n", encoding="utf-8")

    port = _free_port()
    thr = threading.Thread(
        target=run_dashboard,
        kwargs={"root": tmp_path, "host": "127.0.0.1", "port": port, "open_browser": False, "allow_write": True},
        daemon=True,
    )
    thr.start()
    time.sleep(0.6)

    base = f"http://127.0.0.1:{port}"

    # POST /api/export-json
    payload = json.dumps({"in_csv": "new_markers.csv", "out_json": "new_markers.json"}).encode("utf-8")
    req = Request(base + "/api/export-json", data=payload, headers={"Content-Type": "application/json"}, method="POST")

    with urlopen(req) as r:
        result = json.loads(r.read().decode("utf-8"))

    assert result.get("ok") is True
    assert "path" in result

    # Verify JSON file was created
    json_path = tmp_path / "new_markers.json"
    assert json_path.exists()
    with open(json_path, encoding="utf-8") as f:
        exported = json.load(f)
    assert "markers" in exported
    assert len(exported["markers"]) == 2
    assert exported["markers"][0]["marker_id"] == "A"


def test_batch_summary_with_timing_data(tmp_path: Path):
    """Test /api/batch returns timing fields for sparkline rendering."""
    batch_csv = "ref_video,new_video,out_dir,ok,retrieval_sec,warp_sec\nr1,n1,.,True,1.2,0.05\nr2,n2,.,True,1.5,0.06\n"
    (tmp_path / "batch_summary.csv").write_text(batch_csv, encoding="utf-8")

    port = _free_port()
    thr = threading.Thread(
        target=run_dashboard,
        kwargs={"root": tmp_path, "host": "127.0.0.1", "port": port, "open_browser": False, "allow_write": False},
        daemon=True,
    )
    thr.start()
    time.sleep(0.6)

    base = f"http://127.0.0.1:{port}"

    with urlopen(base + "/api/batch") as r:
        batch = json.loads(r.read().decode("utf-8"))

    assert "rows" in batch
    assert len(batch["rows"]) == 2
    assert "retrieval_sec" in batch["rows"][0]
    assert "warp_sec" in batch["rows"][0]
    assert batch["rows"][0]["retrieval_sec"] == "1.2"
    assert batch["rows"][1]["warp_sec"] == "0.06"


def test_dashboard_blocks_traversal(tmp_path: Path):
    """Test that dashboard blocks path traversal attempts in GET and POST."""
    (tmp_path / "new_markers.csv").write_text("marker_id,t_ref,t_new_est\nA,0.0,0.5\n", encoding="utf-8")

    port = _free_port()
    thr = threading.Thread(
        target=run_dashboard,
        kwargs={"root": tmp_path, "host": "127.0.0.1", "port": port, "open_browser": False, "allow_write": True},
        daemon=True,
    )
    thr.start()
    time.sleep(0.6)

    base = f"http://127.0.0.1:{port}"

    # GET traversal attempt
    try:
        with urlopen(base + "/api/markers?file=/etc/passwd") as r:
            data = json.loads(r.read().decode("utf-8"))
        assert "error" in data
    except HTTPError:
        # Either error in response or HTTP error is acceptable
        pass

    # POST traversal attempt
    bad = json.dumps({"in_csv": "/etc/passwd", "out_json": "../../pwn.json"}).encode("utf-8")
    req = Request(base + "/api/export-json", data=bad, method="POST", headers={"Content-Type": "application/json"})

    try:
        with urlopen(req) as r:
            data = json.loads(r.read().decode("utf-8"))
        assert "error" in data
    except HTTPError as e:
        # HTTP error is also acceptable (400 Bad Request)
        assert e.code == 400
