import csv, json
from pathlib import Path
from mtbsync.match.marker_transfer import transfer_markers
from mtbsync.export import export_markers

def _make_markers(tmp_path: Path):
    # Build a tiny ref markers file + timewarp.json, then transfer to produce input CSV
    ref = tmp_path / "ref_markers.csv"
    ref.write_text("marker_id,t_ref,label\nA,0.0,start\nB,1.0,mid\nC,2.0,end\n", encoding="utf-8")
    tw = tmp_path / "timewarp.json"
    tw.write_text(json.dumps({"params": {"a": 1.0, "b": 0.5}}), encoding="utf-8")
    out_csv = tmp_path / "new_markers.csv"
    transfer_markers(ref, tw, out_csv)
    return out_csv

def test_export_csv_json_edl_xml(tmp_path: Path):
    in_csv = _make_markers(tmp_path)

    # CSV
    out_csv = tmp_path / "export.csv"
    export_markers(in_csv, out_csv, fmt="csv")
    assert out_csv.exists()
    rows = list(csv.DictReader(open(out_csv)))
    assert len(rows) == 3

    # JSON
    out_json = tmp_path / "export.json"
    export_markers(in_csv, out_json, fmt="json", label="test")
    data = json.loads(out_json.read_text())
    assert data["label"] == "test"
    assert len(data["markers"]) == 3

    # EDL
    out_edl = tmp_path / "export.edl"
    export_markers(in_csv, out_edl, fmt="edl", fps=30, reel="AX")
    text = out_edl.read_text()
    assert "TITLE: MARKERS" in text
    assert "FCM: NON-DROP FRAME" in text
    assert "AX" in text
    assert "00:00:00:00" in text  # basic timecode presence

    # XML
    out_xml = tmp_path / "export.xml"
    export_markers(in_csv, out_xml, fmt="xml", fps=30)
    xml = out_xml.read_text()
    assert "<xmeml" in xml and "<sequence>" in xml and "<marker>" in xml

def test_export_fcpxml(tmp_path: Path):
    """Test FCPXML (Final Cut Pro X) export format."""
    in_csv = _make_markers(tmp_path)
    out_fcpxml = tmp_path / "export.fcpxml"
    export_markers(in_csv, out_fcpxml, fmt="fcpxml", fps=30)

    assert out_fcpxml.exists()
    xml = out_fcpxml.read_text()

    # Validate FCPXML structure
    assert '<?xml version="1.0" encoding="UTF-8"?>' in xml
    assert '<!DOCTYPE fcpxml>' in xml
    assert '<fcpxml version="1.8">' in xml
    assert '<library>' in xml
    assert '<event name="Markers">' in xml
    assert '<project name="Markers">' in xml
    assert '<sequence' in xml
    assert '<spine>' in xml
    assert '<marker start=' in xml

    # Check for rational time format (frames/fps)
    assert '/30s' in xml  # e.g., "0/30s", "30/30s", "60/30s"

    # Check for marker values and notes
    assert 'value="A"' in xml or 'value="B"' in xml or 'value="C"' in xml
    assert '<note>' in xml  # Should have notes for labels

def test_export_premiere_xml(tmp_path: Path):
    """Test Adobe Premiere Pro XML export format."""
    in_csv = _make_markers(tmp_path)
    out_premiere = tmp_path / "export.xml"
    export_markers(in_csv, out_premiere, fmt="premiere", fps=30)

    assert out_premiere.exists()
    xml = out_premiere.read_text()

    # Validate Premiere XML structure
    assert '<?xml version="1.0" encoding="UTF-8"?>' in xml
    assert '<!DOCTYPE xmeml>' in xml
    assert '<xmeml version="4">' in xml
    assert '<sequence id="sequence-1">' in xml
    assert '<name>Markers</name>' in xml
    assert '<rate>' in xml
    assert '<timebase>30</timebase>' in xml
    assert '<ntsc>FALSE</ntsc>' in xml
    assert '<timecode>' in xml
    assert '<marker>' in xml

    # Check for frame-based timing
    assert '<in>' in xml
    assert '<out>' in xml

    # Check for marker names and comments
    assert '<name>A</name>' in xml or '<name>B</name>' in xml or '<name>C</name>' in xml
    assert '<comment>' in xml  # Should have comments for labels

def test_fcpxml_timebase_matches_fps(tmp_path: Path):
    """Test that FCPXML format string matches requested fps."""
    in_csv = _make_markers(tmp_path)
    out_xml_24 = tmp_path / "export_24.fcpxml"
    out_xml_30 = tmp_path / "export_30.fcpxml"
    export_markers(in_csv, out_xml_24, fmt="fcpxml", fps=24)
    export_markers(in_csv, out_xml_30, fmt="fcpxml", fps=30)
    t24 = out_xml_24.read_text()
    t30 = out_xml_30.read_text()
    assert 'FFVideoFormat1080p24' in t24 and '100/2400s' in t24
    assert 'FFVideoFormat1080p30' in t30 and '100/3000s' in t30

def test_premiere_xml_emits_multiple_markers(tmp_path: Path):
    """Test that Premiere XML format emits separate marker blocks."""
    in_csv = _make_markers(tmp_path)
    out_xml = tmp_path / "export_premiere.xml"
    export_markers(in_csv, out_xml, fmt="premiere", fps=30)
    text = out_xml.read_text()
    # Expect multiple <marker> blocks
    assert text.count("<marker>") >= 3
