import json
import os
from pathlib import Path
from mtbsync.match.marker_transfer import transfer_markers


def _make_markers_csv(tmp_path: Path) -> Path:
    """Create a test markers CSV file."""
    ref = tmp_path / "ref_markers.csv"
    ref.write_text("marker_id,t_ref,label\nA,0.0,start\nB,1.0,mid\n", encoding="utf-8")
    tw = tmp_path / "timewarp.json"
    tw.write_text(json.dumps({"params": {"a": 1.0, "b": 0.0}}), encoding="utf-8")
    out_csv = tmp_path / "markers.csv"
    transfer_markers(ref, tw, out_csv)
    return out_csv


def test_preset_from_config(tmp_path: Path, monkeypatch):
    """Test loading export preset from config.toml."""
    from mtbsync.cli import export_markers_cmd
    from pathlib import Path as StdPath
    import io
    import sys

    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Create mock config directory
    config_dir = tmp_path / ".mtbsync"
    config_dir.mkdir()
    config_path = config_dir / "config.toml"

    # Write config with preset
    config_path.write_text(
        """
[presets.my-fcpxml]
format = "fcpxml"
fps = 24

[presets.my-edl]
format = "edl"
fps = 25
reel = "B1"
""",
        encoding="utf-8",
    )

    # Mock Path.home() to return our tmp_path
    def mock_home():
        return tmp_path

    monkeypatch.setattr(StdPath, "home", mock_home)

    # Capture stdout to verify output
    out_path = tmp_path / "output.fcpxml"

    # Call export with preset (this should load from config)
    try:
        # Temporarily redirect stdout
        old_stdout = sys.stdout
        sys.stdout = io.StringIO()

        export_markers_cmd(
            in_csv=str(markers_csv),
            fmt="csv",  # Default, should be overridden by preset
            preset="my-fcpxml",
            out=str(out_path),
            fps=30,  # Default, should be overridden by preset to 24
            reel="AX",
            label=None,
        )

        output = sys.stdout.getvalue()
        sys.stdout = old_stdout

    except SystemExit:
        sys.stdout = old_stdout
        pass

    # Verify output file was created
    assert out_path.exists(), f"Output file not created: {out_path}"

    # Verify it's FCPXML format
    content = out_path.read_text()
    assert "<!DOCTYPE fcpxml>" in content
    assert "<fcpxml version=" in content


def test_preset_not_found_warning(tmp_path: Path, monkeypatch):
    """Test warning when preset not found in config."""
    from mtbsync.cli import export_markers_cmd
    from pathlib import Path as StdPath
    import io
    import sys

    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Create empty config
    config_dir = tmp_path / ".mtbsync"
    config_dir.mkdir()
    config_path = config_dir / "config.toml"
    config_path.write_text("[presets]\n", encoding="utf-8")

    # Mock Path.home()
    def mock_home():
        return tmp_path

    monkeypatch.setattr(StdPath, "home", mock_home)

    out_path = tmp_path / "output.csv"

    # Capture stderr
    old_stderr = sys.stderr
    sys.stderr = io.StringIO()

    try:
        export_markers_cmd(
            in_csv=str(markers_csv),
            fmt="csv",
            preset="nonexistent-preset",
            out=str(out_path),
            fps=30,
            reel="AX",
            label=None,
        )
        stderr_output = sys.stderr.getvalue()
        sys.stderr = old_stderr
    except SystemExit:
        stderr_output = sys.stderr.getvalue()
        sys.stderr = old_stderr

    # Should show warning about preset not found
    assert "not found" in stderr_output.lower() or "warning" in stderr_output.lower()


def test_preset_cli_precedence(tmp_path: Path, monkeypatch):
    """Test that CLI args take precedence over preset values when explicitly set."""
    from mtbsync.cli import export_markers_cmd
    from pathlib import Path as StdPath
    import io
    import sys

    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Create config with preset
    config_dir = tmp_path / ".mtbsync"
    config_dir.mkdir()
    config_path = config_dir / "config.toml"
    config_path.write_text(
        """
[presets.test-preset]
format = "edl"
fps = 24
reel = "AA"
""",
        encoding="utf-8",
    )

    # Mock Path.home()
    def mock_home():
        return tmp_path

    monkeypatch.setattr(StdPath, "home", mock_home)

    out_path = tmp_path / "output.edl"

    # Call with preset but different CLI args
    # The preset format should be used since preset is specified
    old_stdout = sys.stdout
    sys.stdout = io.StringIO()

    try:
        export_markers_cmd(
            in_csv=str(markers_csv),
            fmt="csv",  # Will be overridden by preset's format
            preset="test-preset",
            out=str(out_path),
            fps=30,  # Note: Currently preset fps overrides this
            reel="AX",  # Note: Currently preset reel overrides this
            label=None,
        )
        sys.stdout = old_stdout
    except SystemExit:
        sys.stdout = old_stdout

    # Verify EDL was created (from preset format)
    assert out_path.exists()
    content = out_path.read_text()
    assert "TITLE: MARKERS" in content
    assert "FCM: NON-DROP FRAME" in content


def test_no_config_file_warning(tmp_path: Path, monkeypatch):
    """Test error when unknown preset requested and config file doesn't exist."""
    from mtbsync.cli import export_markers_cmd
    from pathlib import Path as StdPath

    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Mock Path.home() to point to directory without config
    def mock_home():
        return tmp_path

    monkeypatch.setattr(StdPath, "home", mock_home)

    out_path = tmp_path / "output.csv"

    # Should raise ValueError for unknown preset when config doesn't exist
    try:
        export_markers_cmd(
            in_csv=str(markers_csv),
            fmt="csv",
            preset="any-preset",
            out=str(out_path),
            fps=30,
            reel="AX",
            label=None,
        )
        assert False, "Expected ValueError for unknown preset"
    except ValueError as e:
        # Should mention the unknown preset and suggest alternatives
        assert "unknown preset" in str(e).lower()
        assert "any-preset" in str(e).lower()
