import json, time
import numpy as np
import pandas as pd
from pathlib import Path
from mtbsync.io.gps import estimate_tref_for_tnew

def _make_tracks(n=20000, offset=2.5):
    t_rel = np.linspace(0, 600, n)  # 10 min span
    speed = 7.0 + 2.0*np.sin(2*np.pi*t_rel/30.0)
    dist = np.cumsum(speed * np.gradient(t_rel))
    ref_df = pd.DataFrame({"t_rel": t_rel, "dist_m": dist})
    new_df = pd.DataFrame({"t_rel": t_rel, "dist_m": np.interp(t_rel, t_rel, dist)})
    # Just reuse same dist curve; offset is applied at query time
    return ref_df, new_df

def test_vectorised_equivalence_and_speed():
    ref_df, new_df = _make_tracks(n=8000, offset=2.5)
    tnew = np.linspace(0, 600, 5000)

    # Scalar path (force by slicing to small chunks and concatenating)
    # We call the function per chunk to bypass the vectorised fast-path threshold.
    start = time.time()
    parts = []
    for chunk in np.array_split(tnew, 20):  # 250-length chunks < 512
        parts.append(estimate_tref_for_tnew(chunk, ref_df, new_df, offset_sec=1.25))
    tref_scalar = np.concatenate(parts)
    t_scalar = time.time() - start

    # Vectorised path (single call)
    start = time.time()
    tref_vec = estimate_tref_for_tnew(tnew, ref_df, new_df, offset_sec=1.25)
    t_vec = time.time() - start

    # Numeric equivalence within a small tolerance
    assert np.allclose(tref_vec, tref_scalar, atol=1e-6)

    # Expect a reasonable speedup (allow flaky envs, just assert it's not slower)
    assert t_vec <= t_scalar
