import csv
import json
from pathlib import Path
from mtbsync.importers import import_edl, import_xml, save_markers_csv
from mtbsync.export import export_edl, export_xml, export_fcpxml, export_premiere_xml
from mtbsync.match.marker_transfer import transfer_markers


def _make_markers_csv(tmp_path: Path) -> Path:
    """Create a test markers CSV file."""
    ref = tmp_path / "ref_markers.csv"
    ref.write_text("marker_id,t_ref,label\nA,0.0,start\nB,1.5,middle\nC,3.0,end\n", encoding="utf-8")
    tw = tmp_path / "timewarp.json"
    tw.write_text(json.dumps({"params": {"a": 1.0, "b": 0.0}}), encoding="utf-8")
    out_csv = tmp_path / "markers.csv"
    transfer_markers(ref, tw, out_csv)
    return out_csv


def test_import_edl_roundtrip(tmp_path: Path):
    """Test EDL export → import roundtrip."""
    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Export to EDL
    edl_path = tmp_path / "markers.edl"
    export_edl(markers_csv, edl_path, fps=30, reel="AX")
    assert edl_path.exists()

    # Import back from EDL
    imported = import_edl(edl_path, fps=30)
    assert len(imported) == 3

    # Verify marker content
    assert imported[0]["marker_id"] == "001"
    assert float(imported[0]["t_ref"]) == 0.0
    assert "start" in imported[0]["label"]

    assert imported[1]["marker_id"] == "002"
    assert abs(float(imported[1]["t_ref"]) - 1.5) < 0.1  # Allow some rounding error
    assert "middle" in imported[1]["label"]

    assert imported[2]["marker_id"] == "003"
    assert abs(float(imported[2]["t_ref"]) - 3.0) < 0.1
    assert "end" in imported[2]["label"]


def test_import_xml_fcp7_roundtrip(tmp_path: Path):
    """Test FCP7-style XML export → import roundtrip."""
    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Export to XML (FCP7 style)
    xml_path = tmp_path / "markers.xml"
    export_xml(markers_csv, xml_path, fps=30)
    assert xml_path.exists()

    # Import back from XML
    imported = import_xml(xml_path, fps=30)
    assert len(imported) == 3

    # Verify marker content
    assert imported[0]["marker_id"] == "A"
    assert float(imported[0]["t_ref"]) == 0.0
    assert imported[0]["label"] in ["start", "A"]

    assert imported[1]["marker_id"] == "B"
    assert abs(float(imported[1]["t_ref"]) - 1.5) < 0.1
    assert imported[1]["label"] in ["middle", "B"]


def test_import_fcpxml_roundtrip(tmp_path: Path):
    """Test FCPXML export → import roundtrip."""
    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Export to FCPXML
    fcpxml_path = tmp_path / "markers.fcpxml"
    export_fcpxml(markers_csv, fcpxml_path, fps=30)
    assert fcpxml_path.exists()

    # Import back from FCPXML
    imported = import_xml(fcpxml_path, fps=30)
    assert len(imported) == 3

    # Verify marker content (FCPXML uses value attribute for marker_id)
    assert imported[0]["marker_id"] == "A"
    assert float(imported[0]["t_ref"]) == 0.0
    assert "start" in imported[0]["label"]

    assert imported[1]["marker_id"] == "B"
    assert abs(float(imported[1]["t_ref"]) - 1.5) < 0.1
    assert "middle" in imported[1]["label"]


def test_import_premiere_xml_roundtrip(tmp_path: Path):
    """Test Premiere XML export → import roundtrip."""
    # Create markers CSV
    markers_csv = _make_markers_csv(tmp_path)

    # Export to Premiere XML
    premiere_path = tmp_path / "markers_premiere.xml"
    export_premiere_xml(markers_csv, premiere_path, fps=30)
    assert premiere_path.exists()

    # Import back from Premiere XML
    imported = import_xml(premiere_path, fps=30)
    assert len(imported) == 3

    # Verify marker content
    assert imported[0]["marker_id"] == "A"
    assert float(imported[0]["t_ref"]) == 0.0
    assert "start" in imported[0]["label"]

    assert imported[1]["marker_id"] == "B"
    assert abs(float(imported[1]["t_ref"]) - 1.5) < 0.1
    assert "middle" in imported[1]["label"]


def test_save_markers_csv(tmp_path: Path):
    """Test saving markers to CSV."""
    markers = [
        {"marker_id": "M1", "t_ref": "0.0", "label": "first"},
        {"marker_id": "M2", "t_ref": "1.5", "label": "second"},
    ]

    out_path = tmp_path / "output.csv"
    save_markers_csv(markers, out_path)

    assert out_path.exists()

    # Read back and verify
    with open(out_path, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        rows = list(reader)

    assert len(rows) == 2
    assert rows[0]["marker_id"] == "M1"
    assert rows[0]["t_ref"] == "0.0"
    assert rows[0]["label"] == "first"


def test_import_edl_missing_file():
    """Test import_edl raises FileNotFoundError for missing file."""
    try:
        import_edl("/nonexistent/file.edl")
        assert False, "Expected FileNotFoundError"
    except FileNotFoundError:
        pass


def test_import_xml_missing_file():
    """Test import_xml raises FileNotFoundError for missing file."""
    try:
        import_xml("/nonexistent/file.xml")
        assert False, "Expected FileNotFoundError"
    except FileNotFoundError:
        pass


def test_import_xml_invalid_format(tmp_path: Path):
    """Test import_xml raises ValueError for invalid XML."""
    bad_xml = tmp_path / "bad.xml"
    bad_xml.write_text("Not valid XML at all!", encoding="utf-8")

    try:
        import_xml(bad_xml)
        assert False, "Expected ValueError for invalid XML"
    except ValueError as e:
        assert "Invalid XML" in str(e)


def test_import_xml_unsupported_root(tmp_path: Path):
    """Test import_xml raises ValueError for unsupported XML root element."""
    unknown_xml = tmp_path / "unknown.xml"
    unknown_xml.write_text('<?xml version="1.0"?><unknown></unknown>', encoding="utf-8")

    try:
        import_xml(unknown_xml)
        assert False, "Expected ValueError for unsupported format"
    except ValueError as e:
        assert "Unsupported XML format" in str(e)
