"""Tests for keyframe processing functionality."""

import numpy as np
import pytest

from mtbsync.features.keyframes import resize_max


class TestResizeMax:
    """Tests for resize_max function."""

    def test_resize_larger_height(self):
        """Test resizing when height is larger than width."""
        # Create a 1920x1080 frame (width x height, but numpy is height x width)
        frame = np.zeros((1920, 1080, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Height should be 960, width should be proportional
        assert resized.shape[0] == 960  # height
        assert resized.shape[1] == 540  # width (1080 * 960 / 1920)
        assert resized.shape[2] == 3  # channels preserved

    def test_resize_larger_width(self):
        """Test resizing when width is larger than height."""
        # Create a 1080x1920 frame
        frame = np.zeros((1080, 1920, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Width should be 960, height should be proportional
        assert resized.shape[0] == 540  # height (1080 * 960 / 1920)
        assert resized.shape[1] == 960  # width
        assert resized.shape[2] == 3  # channels preserved

    def test_no_resize_needed(self):
        """Test that frame is unchanged when already smaller than max_dim."""
        frame = np.zeros((480, 640, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Should be unchanged
        assert resized.shape == frame.shape
        assert np.array_equal(resized, frame)

    def test_square_frame(self):
        """Test resizing a square frame."""
        frame = np.zeros((1920, 1920, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Both dimensions should be 960
        assert resized.shape[0] == 960
        assert resized.shape[1] == 960
        assert resized.shape[2] == 3

    def test_grayscale_frame(self):
        """Test resizing a grayscale (2D) frame."""
        frame = np.zeros((1920, 1080), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Should preserve 2D shape
        assert len(resized.shape) == 2
        assert resized.shape[0] == 960
        assert resized.shape[1] == 540

    def test_invalid_max_dim(self):
        """Test error handling for invalid max_dim values."""
        frame = np.zeros((100, 100, 3), dtype=np.uint8)

        with pytest.raises(ValueError, match="max_dim must be positive"):
            resize_max(frame, max_dim=0)

        with pytest.raises(ValueError, match="max_dim must be positive"):
            resize_max(frame, max_dim=-100)

    def test_empty_frame(self):
        """Test error handling for empty frame."""
        with pytest.raises(ValueError, match="Frame is empty"):
            resize_max(np.array([]), max_dim=960)

    def test_none_frame(self):
        """Test error handling for None frame."""
        with pytest.raises(ValueError, match="Frame is empty or None"):
            resize_max(None, max_dim=960)

    def test_1d_frame(self):
        """Test error handling for 1D frame."""
        frame = np.zeros((100,), dtype=np.uint8)

        with pytest.raises(ValueError, match="Frame must be at least 2D"):
            resize_max(frame, max_dim=960)

    def test_aspect_ratio_preservation(self):
        """Test that aspect ratio is preserved during resize."""
        # Create a 16:9 frame
        frame = np.zeros((1080, 1920, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=960)

        # Calculate aspect ratios
        original_aspect = 1920 / 1080
        resized_aspect = resized.shape[1] / resized.shape[0]

        # Should be very close (within rounding error)
        assert abs(original_aspect - resized_aspect) < 0.01

    def test_custom_max_dim(self):
        """Test resizing with custom max_dim value."""
        frame = np.zeros((2000, 3000, 3), dtype=np.uint8)

        resized = resize_max(frame, max_dim=500)

        # Width (3000) is larger, so it should be 500
        assert resized.shape[1] == 500
        # Height should be proportional: 2000 * 500 / 3000 = 333
        assert resized.shape[0] == 333
