from pathlib import Path
import json, threading, time
from urllib.request import urlopen

from mtbsync.telemetry import TelemetryRecorder
from mtbsync.dashboard import run_dashboard

def test_perf_json_written(tmp_path: Path):
    data = {"ok": True, "cmd": "test", "timings": {"retrieval_sec": 0.12}}
    p = TelemetryRecorder.write_perf_json(tmp_path, data)
    assert p.exists()
    obj = json.loads(p.read_text())
    assert obj["ok"] is True
    assert "timings" in obj

def _free_port():
    import socket
    s=socket.socket(); s.bind(("127.0.0.1",0))
    _,port=s.getsockname(); s.close(); return port

def test_dashboard_perf_endpoint(tmp_path: Path):
    # create two perf.json
    (tmp_path/"a").mkdir()
    (tmp_path/"a"/"perf.json").write_text('{"ok":true,"cmd":"sync","fps":12.3}', encoding="utf-8")
    (tmp_path/"b").mkdir()
    (tmp_path/"b"/"perf.json").write_text('{"ok":true,"cmd":"batch_item"}', encoding="utf-8")
    port=_free_port()
    import threading
    t = threading.Thread(target=run_dashboard, kwargs={"root": tmp_path, "host":"127.0.0.1","port":port,"open_browser":False}, daemon=True)
    t.start(); time.sleep(0.6)
    with urlopen(f"http://127.0.0.1:{port}/api/perf") as r:
        obj = json.loads(r.read().decode("utf-8"))
        assert obj["count"] == 2
        assert isinstance(obj["items"], list)

def test_dashboard_perf_stream_once(tmp_path: Path):
    # prepare one perf file
    (tmp_path/"x").mkdir()
    (tmp_path/"x"/"perf.json").write_text('{"ok":true,"cmd":"sync","fps":10.0,"run":{"wall_sec":0.1,"rss_bytes":1000000}}', encoding="utf-8")
    port = _free_port()
    import threading, time, urllib.request
    from mtbsync.dashboard import run_dashboard
    t = threading.Thread(target=run_dashboard, kwargs={"root": tmp_path, "host":"127.0.0.1","port":port,"open_browser":False}, daemon=True)
    t.start(); time.sleep(0.6)
    # open stream and read a small chunk
    req = urllib.request.Request(f"http://127.0.0.1:{port}/api/perf/stream")
    with urllib.request.urlopen(req, timeout=2) as r:
        data = r.read(128).decode("utf-8")
        assert "event: init" in data or "event: perf" in data

def test_sse_does_not_block_other_endpoints(tmp_path: Path):
    # Prepare a perf.json so /api/perf & /api/files return something
    (tmp_path/"out").mkdir()
    (tmp_path/"out"/"perf.json").write_text('{"ok":true,"cmd":"sync"}', encoding="utf-8")
    port = _free_port()
    from mtbsync.dashboard import run_dashboard
    t = threading.Thread(target=run_dashboard, kwargs={"root": tmp_path, "host":"127.0.0.1", "port": port, "open_browser": False}, daemon=True)
    t.start(); time.sleep(0.6)
    base = f"http://127.0.0.1:{port}"
    # Open SSE stream in a background thread and keep it open
    import urllib.request
    sse_started = []
    def _open_sse():
        try:
            with urllib.request.urlopen(base + "/api/perf/stream", timeout=5) as resp:
                sse_started.append(True)
                # read a bit to ensure connection stays open
                _ = resp.read(64)
        except Exception:
            pass
    th = threading.Thread(target=_open_sse, daemon=True)
    th.start()
    time.sleep(0.3)
    # While SSE is connected, we should still be able to call other endpoints quickly
    with urllib.request.urlopen(base + "/api/files", timeout=2) as r:
        txt = r.read().decode("utf-8")
        assert '"items"' in txt
