from pathlib import Path
import json
import types
from mtbsync.telemetry import TelemetryRecorder


def test_collect_cpu_pct_rss_graceful(monkeypatch):
    """Test CPU/RSS collection with mocked psutil."""
    # Simulate psutil presence
    fake_psutil = types.SimpleNamespace()

    class _Mem:
        rss = 123456789

    class _Proc:
        def __init__(self, *_a, **_k):
            pass

        def cpu_percent(self, interval=None):
            return 12.5

        def memory_info(self):
            return _Mem()

    fake_psutil.Process = _Proc
    monkeypatch.setattr("mtbsync.telemetry.psutil", fake_psutil, raising=False)

    data = TelemetryRecorder.collect_cpu_pct_rss()
    assert data["cpu_pct"] == 12.5
    assert round(data["rss_mb"], 1) == round(123456789 / 1e6, 1)


def test_collect_cpu_pct_rss_no_psutil(monkeypatch):
    """Test CPU/RSS collection gracefully handles missing psutil."""
    monkeypatch.setattr("mtbsync.telemetry.psutil", None, raising=False)
    data = TelemetryRecorder.collect_cpu_pct_rss()
    assert data["cpu_pct"] is None
    assert data["rss_mb"] is None


def test_collect_gpu_metrics_graceful(monkeypatch):
    """Test GPU collection with mocked NVML."""
    # Simulate NVML presence with one device
    fake = types.SimpleNamespace()

    class _U:
        gpu = 42

    class _M:
        total = 8000 * 1024 * 1024
        used = 3000 * 1024 * 1024

    def nvmlInit():
        pass

    def nvmlShutdown():
        pass

    def nvmlDeviceGetCount():
        return 1

    def nvmlDeviceGetHandleByIndex(i):
        return object()

    def nvmlDeviceGetUtilizationRates(_h):
        return _U()

    def nvmlDeviceGetMemoryInfo(_h):
        return _M()

    fake.nvmlInit = nvmlInit
    fake.nvmlShutdown = nvmlShutdown
    fake.nvmlDeviceGetCount = nvmlDeviceGetCount
    fake.nvmlDeviceGetHandleByIndex = nvmlDeviceGetHandleByIndex
    fake.nvmlDeviceGetUtilizationRates = nvmlDeviceGetUtilizationRates
    fake.nvmlDeviceGetMemoryInfo = nvmlDeviceGetMemoryInfo

    monkeypatch.setattr("mtbsync.telemetry.pynvml", fake, raising=False)

    out = TelemetryRecorder.collect_gpu_metrics(enable_gpu=True)
    assert out["gpu_util"] == 42
    # 3000 * 1024 * 1024 bytes / 1e6 = 3145.728 MB
    assert round(out["gpu_mem_mb"]) == 3146


def test_collect_gpu_metrics_no_nvml(monkeypatch):
    """Test GPU collection gracefully handles missing NVML."""
    monkeypatch.setattr("mtbsync.telemetry.pynvml", None, raising=False)
    out = TelemetryRecorder.collect_gpu_metrics(enable_gpu=True)
    assert out["gpu_util"] is None
    assert out["gpu_mem_mb"] is None


def test_collect_gpu_metrics_disabled():
    """Test GPU collection respects enable_gpu=False."""
    out = TelemetryRecorder.collect_gpu_metrics(enable_gpu=False)
    assert out["gpu_util"] is None
    assert out["gpu_mem_mb"] is None


def test_perf_payload_contains_extended_fields(tmp_path: Path, monkeypatch):
    """Test that perf.json can be written with extended fields."""
    # No psutil/nvml -> should still write perf.json with None fields
    monkeypatch.setattr("mtbsync.telemetry.psutil", None, raising=False)
    monkeypatch.setattr("mtbsync.telemetry.pynvml", None, raising=False)

    payload = {
        "ok": True,
        "cmd": "sync",
        "cpu_pct": None,
        "rss_mb": None,
        "gpu_util": None,
        "gpu_mem_mb": None,
    }
    p = TelemetryRecorder.write_perf_json(tmp_path, payload)
    assert p.exists()

    obj = json.loads(p.read_text())
    assert obj["ok"] is True
    assert obj["cmd"] == "sync"
    assert obj["cpu_pct"] is None
    assert obj["rss_mb"] is None
    assert obj["gpu_util"] is None
    assert obj["gpu_mem_mb"] is None
