import math
from mtbsync.match.timewarp import fit_timewarp, gate_by_warp, TimeWarp

def _synthetic_pairs(a: float, b: float, n: int = 100, noise: float = 0.02, outlier_frac: float = 0.05):
    """Generate synthetic (t_new, t_ref) pairs with Gaussian-like noise and optional outliers."""
    pairs = []
    step = 1.0
    for i in range(n):
        t_new = i * step
        t_ref = a * t_new + b + ((-1)**i) * noise * (i % 5) * 0.2
        pairs.append((t_new, t_ref))
    # inject simple outliers
    outliers = max(1, int(outlier_frac * n))
    for i in range(outliers):
        pairs[i] = (pairs[i][0], pairs[i][1] + 5.0)  # 5s offset
    return pairs

def test_affine_recovery_basic():
    pairs = _synthetic_pairs(1.0003, 2.4, n=200, noise=0.01, outlier_frac=0.05)
    result = fit_timewarp(pairs, rng_seed=1)
    assert result["ok"]
    a, b = result["params"]["a"], result["params"]["b"]
    assert abs(a - 1.0003) < 5e-4
    assert abs(b - 2.4) < 0.1
    assert result["residuals"]["p90"] < 0.15

def test_outlier_robustness():
    pairs = _synthetic_pairs(1.0005, -1.2, n=150, noise=0.02, outlier_frac=0.4)
    result = fit_timewarp(pairs, rng_seed=2, min_inlier_frac=0.25)
    assert result["ok"]
    assert result["inlier_frac"] >= 0.25

def test_ppm_clamp():
    pairs = _synthetic_pairs(1.005, 0.0, n=120, noise=0.01)
    result = fit_timewarp(pairs, rng_seed=3, max_ppm=1000)
    assert not result["ok"]
    assert result["ppm"] > 1000

def test_low_support_returns_false():
    pairs = [(0.0, 0.1)]  # only one pair
    result = fit_timewarp(pairs)
    assert not result["ok"]
    assert math.isclose(result["params"]["a"], 1.0)
    assert math.isclose(result["params"]["b"], 0.0)

def test_gate_by_warp_filters_correctly():
    warp = TimeWarp("affine", a=1.0, b=1.0)
    candidates = [(1.9, 0.9), (3.0, 2.0), (5.8, 4.4)]  # (t_ref, t_new)
    kept = gate_by_warp(candidates, warp, window_sec=0.2)
    assert len(kept) == 2  # first two within ±0.2s of t_ref=a*t_new+b
