"""Tests for video I/O functionality."""

import tempfile
from pathlib import Path
from unittest.mock import MagicMock, patch

import numpy as np
import pytest

from mtbsync.io.video import extract_keyframes, get_video_duration, video_fps


class TestVideoFps:
    """Tests for video_fps function."""

    def test_video_fps_valid(self):
        """Test fps extraction from valid video metadata."""
        mock_probe = {
            "streams": [
                {"codec_type": "video", "r_frame_rate": "30000/1001"}  # 29.97 fps
            ]
        }

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            fps = video_fps("dummy.mp4")
            assert abs(fps - 29.97) < 0.01

    def test_video_fps_exact(self):
        """Test fps extraction with exact frame rate."""
        mock_probe = {"streams": [{"codec_type": "video", "r_frame_rate": "30/1"}]}

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            fps = video_fps("dummy.mp4")
            assert fps == 30.0

    def test_video_fps_no_streams(self):
        """Test error handling when no video stream is found."""
        mock_probe = {"streams": [{"codec_type": "audio"}]}

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            with pytest.raises(RuntimeError, match="No video stream found"):
                video_fps("dummy.mp4")

    def test_video_fps_invalid_file(self):
        """Test error handling for invalid video file."""
        import ffmpeg

        with patch(
            "mtbsync.io.video.ffmpeg.probe",
            side_effect=ffmpeg.Error("ffmpeg", b"", b"File not found"),
        ):
            with pytest.raises(RuntimeError, match="Cannot read video file"):
                video_fps("nonexistent.mp4")


class TestGetVideoDuration:
    """Tests for get_video_duration function."""

    def test_duration_from_format(self):
        """Test duration extraction from format metadata."""
        mock_probe = {
            "format": {"duration": "120.5"},
            "streams": [{"codec_type": "video"}],
        }

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            duration = get_video_duration("dummy.mp4")
            assert duration == 120.5

    def test_duration_from_stream(self):
        """Test duration extraction from stream metadata."""
        mock_probe = {
            "streams": [{"codec_type": "video", "duration": "60.0"}],
        }

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            duration = get_video_duration("dummy.mp4")
            assert duration == 60.0


class TestExtractKeyframes:
    """Tests for extract_keyframes function."""

    def test_invalid_fps(self):
        """Test error handling for invalid fps values."""
        with pytest.raises(ValueError, match="FPS must be positive"):
            extract_keyframes("dummy.mp4", fps=0)

        with pytest.raises(ValueError, match="FPS must be positive"):
            extract_keyframes("dummy.mp4", fps=-1)

    def test_nonexistent_file(self):
        """Test error handling for nonexistent file."""
        with pytest.raises(RuntimeError, match="Video file not found"):
            extract_keyframes("nonexistent.mp4", fps=3.0)

    def test_empty_file(self):
        """Test error handling for empty file."""
        with tempfile.NamedTemporaryFile(suffix=".mp4", delete=False) as f:
            empty_file = f.name

        try:
            with pytest.raises(RuntimeError, match="Video file is empty"):
                extract_keyframes(empty_file, fps=3.0)
        finally:
            Path(empty_file).unlink()

    def test_deterministic_timestamps(self):
        """Test that timestamp extraction is deterministic across multiple reads."""
        # Mock video metadata
        mock_probe = {
            "format": {"duration": "10.0"},
            "streams": [
                {
                    "codec_type": "video",
                    "width": 1920,
                    "height": 1080,
                    "r_frame_rate": "30/1",
                }
            ],
        }

        # Create a dummy frame
        dummy_frame = np.zeros((1080, 1920, 3), dtype=np.uint8)
        dummy_frame_bytes = dummy_frame.tobytes()

        # Mock ffmpeg operations
        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            with patch("mtbsync.io.video.get_video_duration", return_value=10.0):
                with patch("mtbsync.io.video.Path.exists", return_value=True):
                    with patch("mtbsync.io.video.Path.stat") as mock_stat:
                        mock_stat.return_value.st_size = 1000

                        # Mock ffmpeg.run to return dummy frames
                        with patch("mtbsync.io.video.ffmpeg.input") as mock_input:
                            mock_output = MagicMock()
                            mock_output.run.return_value = (dummy_frame_bytes, b"")
                            mock_input.return_value.output.return_value = mock_output

                            # First extraction
                            keyframes1 = extract_keyframes("dummy.mp4", fps=3.0)
                            timestamps1 = [t for t, _ in keyframes1]

                            # Second extraction
                            keyframes2 = extract_keyframes("dummy.mp4", fps=3.0)
                            timestamps2 = [t for t, _ in keyframes2]

                            # Verify timestamps are identical
                            assert len(timestamps1) == len(timestamps2)
                            assert timestamps1 == timestamps2

                            # Verify timestamps are at expected intervals
                            expected_interval = 1.0 / 3.0
                            for i, t in enumerate(timestamps1):
                                expected_t = i * expected_interval
                                assert abs(t - expected_t) < 1e-6

    def test_timestamp_calculation(self):
        """Test that timestamps are calculated correctly for given fps."""
        mock_probe = {
            "format": {"duration": "5.0"},
            "streams": [
                {
                    "codec_type": "video",
                    "width": 640,
                    "height": 480,
                    "r_frame_rate": "30/1",
                }
            ],
        }

        dummy_frame = np.zeros((480, 640, 3), dtype=np.uint8)
        dummy_frame_bytes = dummy_frame.tobytes()

        with patch("mtbsync.io.video.ffmpeg.probe", return_value=mock_probe):
            with patch("mtbsync.io.video.get_video_duration", return_value=5.0):
                with patch("mtbsync.io.video.Path.exists", return_value=True):
                    with patch("mtbsync.io.video.Path.stat") as mock_stat:
                        mock_stat.return_value.st_size = 1000

                        with patch("mtbsync.io.video.ffmpeg.input") as mock_input:
                            mock_output = MagicMock()
                            mock_output.run.return_value = (dummy_frame_bytes, b"")
                            mock_input.return_value.output.return_value = mock_output

                            # Extract at 2 fps (one frame every 0.5 seconds)
                            keyframes = extract_keyframes("dummy.mp4", fps=2.0)
                            timestamps = [t for t, _ in keyframes]

                            # Should have frames at: 0.0, 0.5, 1.0, 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5
                            expected_count = 10
                            assert len(timestamps) == expected_count

                            # Check first few timestamps
                            assert abs(timestamps[0] - 0.0) < 1e-6
                            assert abs(timestamps[1] - 0.5) < 1e-6
                            assert abs(timestamps[2] - 1.0) < 1e-6
