use serde::{de::DeserializeOwned, Serialize};
use std::time::Duration;
use tonic::Status;
use tracing::error;

pub const MAX_VECTOR_DIMENSION: u32 = 16_384; // Double the size of `8192` which is the largest widely used vector dimension.

#[derive(Debug, thiserror::Error)]
pub enum Error {
    #[error("lsn timeout")]
    QueryLsnTimeout,

    #[error("retry timeout")]
    RetryTimeout,

    #[error("collection already exists")]
    CollectionAlreadyExists,

    #[error("collection not found")]
    CollectionNotFound,

    #[error("not found")]
    NotFound,

    #[error("invalid collection schema")]
    SchemaValidationError(ValidationErrorBag<SchemaValidationError>),

    #[error("document validation error")]
    DocumentValidationError(ValidationErrorBag<DocumentValidationError>),

    #[error("collection validation error")]
    CollectionValidationError(ValidationErrorBag<CollectionValidationError>),

    #[error("invalid argument: {0}")]
    InvalidArgument(String),

    #[error("permission denied")]
    PermissionDenied,

    #[error("quota exceeded: {0}")]
    QuotaExceeded(String),

    #[error("request too large: {0}")]
    RequestTooLarge(String),

    #[error("internal error: {0}")]
    Internal(String),

    #[error("unexpected error: {0}")]
    Unexpected(String),

    #[error("slow down: {0}")]
    SlowDown(String),

    #[error("tonic transport error")]
    TransportError(#[from] tonic::transport::Error),

    #[error("malformed response: {0}")]
    MalformedResponse(String),
}

impl Error {
    pub fn is_retryable(&self) -> bool {
        match self {
            // Retryable
            Error::QueryLsnTimeout => true,
            Error::SlowDown(_) => true,
            // Not retryable
            Error::RetryTimeout => false,
            Error::CollectionAlreadyExists => false,
            Error::CollectionNotFound => false,
            Error::NotFound => false,
            Error::SchemaValidationError(_) => false,
            Error::DocumentValidationError(_) => false,
            Error::CollectionValidationError(_) => false,
            Error::InvalidArgument(_) => false,
            Error::PermissionDenied => false,
            Error::QuotaExceeded(_) => false,
            Error::RequestTooLarge(_) => false,
            Error::TransportError(_) => false,
            Error::MalformedResponse(_) => false,
            Error::Unexpected(_) => false,
            Error::Internal(_) => false,
        }
    }

    /// Returns how long the client should keep retrying this error, if applicable.
    /// Returns `Some(duration)` for errors that require retrying for a minimum time,
    /// or `None` if no specific retry duration is required.
    pub fn retry_duration(&self) -> Option<Duration> {
        match self {
            // When quering using `lsn=N`, the client should retry for at least 2 seconds,
            // since the guarantee for "default" consistency mode is ~1 second.
            Error::QueryLsnTimeout => Some(Duration::from_millis(2_000)),
            _ => None,
        }
    }
}

impl From<Status> for Error {
    fn from(status: Status) -> Self {
        let request_id = status
            .metadata()
            .get("x-request-id")
            .and_then(|v| v.to_str().ok())
            .map(|v| v.to_string());

        let append_request_id = |message: &str| {
            if let Some(request_id) = request_id {
                format!("{message} (x-request-id: {request_id})")
            } else {
                message.to_string()
            }
        };

        match CustomError::try_from(status) {
            // Custom error
            Ok(error) => match error.code() {
                CustomErrorCode::RequiredLsnGreaterThanManifestMaxLsn => Error::QueryLsnTimeout,
                CustomErrorCode::SlowDown => Error::SlowDown(error.message().to_string()),
            },
            Err(e) => match e.code() {
                tonic::Code::NotFound => Error::NotFound,
                tonic::Code::ResourceExhausted => Error::QuotaExceeded(e.message().into()),
                tonic::Code::InvalidArgument => match ValidationErrorBag::try_from(e.clone()) {
                    Ok(errors) => Error::DocumentValidationError(errors),
                    Err(_) => match ValidationErrorBag::try_from(e.clone()) {
                        Ok(errors) => Error::SchemaValidationError(errors),
                        Err(_) => match ValidationErrorBag::try_from(e.clone()) {
                            Ok(errors) => Error::CollectionValidationError(errors),
                            Err(_) => Error::InvalidArgument(e.message().into()),
                        },
                    },
                },
                tonic::Code::OutOfRange => Error::RequestTooLarge(e.message().into()),
                tonic::Code::PermissionDenied => Error::PermissionDenied,
                tonic::Code::Internal => Error::Internal(append_request_id(e.message())),
                _ => Error::Unexpected(append_request_id(&format!("{:?}", e))),
            },
        }
    }
}

#[derive(Debug, thiserror::Error, PartialEq, Eq, serde::Serialize, serde::Deserialize)]
pub enum SchemaValidationError {
    #[error("field `{field}` has no data type")]
    MissingDataType { field: String },

    #[error("field name `{field}` cannot start with an underscore")]
    ReservedFieldName { field: String },

    #[error("Missing index spec for field `{field}`")]
    MissingIndexSpec { field: String },

    #[error("invalid index `{index}` for field `{field}` with type `{data_type}`")]
    InvalidIndex {
        field: String,
        index: String,
        data_type: String,
    },

    #[error("invalid vector index metric `{metric}` for field `{field}` with type `{data_type}`")]
    InvalidVectorIndexMetric {
        field: String,
        metric: String,
        data_type: String,
    },

    #[error("vector field `{field}` cannot be have zero dimension")]
    VectorDimensionCannotBeZero { field: String },

    #[error("vector field `{field}` cannot have dimension greater than {MAX_VECTOR_DIMENSION}")]
    VectorDimensionTooLarge { field: String, dimension: u32 },

    #[error("Invalid semantic index for field `{field}. Error: {error}`")]
    InvalidSemanticIndex { field: String, error: String },
}

#[derive(Debug, PartialEq, serde::Serialize, serde::Deserialize)]
pub enum DocumentValidationError {
    MissingId {
        doc_offset: usize,
    },

    InvalidId {
        doc_offset: usize,
        got: String,
    },

    MissingField {
        doc_id: String,
        field: String,
    },

    ReservedFieldName {
        doc_id: String,
        field: String,
    },

    InvalidFieldName {
        doc_id: String,
        field: String,
    },

    InvalidDataType {
        doc_id: String,
        field: String,
        expected_type: String,
        got_value: String,
    },

    InvalidVectorDimension {
        doc_id: String,
        field: String,
        expected_dimension: usize,
        got_dimension: usize,
    },

    InvalidSparseVector {
        doc_id: String,
        field: String,
        reason: String,
    },

    InvalidStructDepth {
        doc_id: String,
        field: String,
        reason: String,
    },

    NoDocuments,

    DocumentTooLarge {
        doc_id: String,
        max_size_bytes: u64,
        got_size_bytes: u64,
    },
}

#[derive(PartialEq, Debug, serde::Serialize, serde::Deserialize)]
pub enum CollectionValidationError {
    InvalidName(String),
}

#[derive(Debug, PartialEq, serde::Serialize, serde::Deserialize)]
pub struct ValidationErrorBag<T: Serialize>(Vec<T>);

impl<T: Serialize + DeserializeOwned> ValidationErrorBag<T> {
    pub fn new(errors: Vec<T>) -> Self {
        Self(errors)
    }

    pub fn from(errors: impl IntoIterator<Item = T>) -> Self {
        Self(errors.into_iter().collect())
    }

    pub fn empty() -> Self {
        Self(vec![])
    }

    pub fn push(&mut self, error: T) {
        self.0.push(error);
    }

    pub fn extend(&mut self, errors: impl IntoIterator<Item = T>) {
        self.0.extend(errors.into_iter());
    }

    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn iter(&self) -> std::slice::Iter<'_, T> {
        self.0.iter()
    }
}

impl<T: Serialize> From<ValidationErrorBag<T>> for tonic::Status {
    fn from(err: ValidationErrorBag<T>) -> Self {
        match serde_json::to_string(&err) {
            Ok(message) => tonic::Status::invalid_argument(message),
            Err(e) => {
                error!(?e, "failed to serialize response");
                tonic::Status::internal("failed to serialize response")
            }
        }
    }
}

impl<T: Serialize + DeserializeOwned> TryFrom<tonic::Status> for ValidationErrorBag<T> {
    type Error = serde_json::Error;

    fn try_from(status: tonic::Status) -> Result<Self, Self::Error> {
        serde_json::from_str(status.message())
    }
}

#[derive(Clone, PartialEq, Eq)]
pub struct CustomError {
    /// Error message
    message: String,
    /// Custom error code
    code: CustomErrorCode,
}

impl CustomError {
    pub fn new(message: String, code: CustomErrorCode) -> Self {
        Self { message, code }
    }

    pub fn message(&self) -> &str {
        &self.message
    }

    pub fn code(&self) -> &CustomErrorCode {
        &self.code
    }
}

impl TryFrom<Status> for CustomError {
    type Error = Status;

    fn try_from(status: Status) -> Result<Self, Self::Error> {
        let ddb_error_code = status
            .metadata()
            .get("x-topk-error-code")
            .ok_or(anyhow::anyhow!("x-topk-error-code not found"))
            .map_err(|_| status.clone())?;
        let ddb_error_code = ddb_error_code.to_str().map_err(|_| status.clone())?;
        let ddb_error_code: u32 = ddb_error_code.parse().map_err(|_| status.clone())?;
        let code = CustomErrorCode::try_from(ddb_error_code).map_err(|_| status.clone())?;

        Ok(CustomError {
            message: status.message().to_string(),
            code,
        })
    }
}

#[derive(Clone, PartialEq, Eq)]
pub enum CustomErrorCode {
    RequiredLsnGreaterThanManifestMaxLsn,
    SlowDown,
}

impl Into<u32> for CustomErrorCode {
    fn into(self) -> u32 {
        match self {
            CustomErrorCode::RequiredLsnGreaterThanManifestMaxLsn => 1000,
            CustomErrorCode::SlowDown => 1429,
        }
    }
}

impl TryFrom<u32> for CustomErrorCode {
    type Error = anyhow::Error;

    fn try_from(code: u32) -> Result<Self, Self::Error> {
        match code {
            1000 => Ok(CustomErrorCode::RequiredLsnGreaterThanManifestMaxLsn),
            1429 => Ok(CustomErrorCode::SlowDown),
            code => Err(anyhow::anyhow!("unknown internal error code: {code}")),
        }
    }
}

impl From<CustomError> for Status {
    fn from(error: CustomError) -> Self {
        let mut status = match error.code {
            CustomErrorCode::RequiredLsnGreaterThanManifestMaxLsn => {
                Status::failed_precondition(error.message)
            }
            CustomErrorCode::SlowDown => Status::resource_exhausted(error.message),
        };

        let error_code: u32 = error.code.into();
        status
            .metadata_mut()
            .insert("x-topk-error-code", error_code.into());

        status
    }
}
