"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const cloudformation_generated_1 = require("./cloudformation.generated");
class CustomResourceProvider {
    constructor(serviceToken) {
        this.serviceToken = serviceToken;
    }
    /**
     * The Lambda provider that implements this custom resource.
     *
     * We recommend using a lambda.SingletonFunction for this.
     */
    static lambda(handler) { return new CustomResourceProvider(handler.functionArn); }
    /**
     * The SNS Topic for the provider that implements this custom resource.
     */
    static topic(topic) { return new CustomResourceProvider(topic.topicArn); }
}
exports.CustomResourceProvider = CustomResourceProvider;
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 */
class CustomResource extends cdk_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        this.resource = new cdk_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.provider.serviceToken,
                ...uppercaseProperties(props.properties || {})
            }
        });
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return cloudformation_generated_1.CfnCustomResource.resourceTypeName;
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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