"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
class TestCustomResource extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const singletonLambda = new lambda.SingletonFunction(this, 'Lambda', {
            uuid: 'TestCustomResourceProvider',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.Python27,
            handler: 'index.hello',
            timeout: 300,
        });
        new lib_1.CustomResource(this, 'Resource', {
            provider: lib_1.CustomResourceProvider.lambda(singletonLambda)
        });
    }
}
module.exports = {
    'custom resource is added twice, lambda is added once'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new TestCustomResource(stack, 'Custom1');
        new TestCustomResource(stack, 'Custom2');
        // THEN
        assert_1.expect(stack).toMatch({
            "Resources": {
                "SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            { "Fn::Join": ["", [
                                        "arn:", { "Ref": "AWS::Partition" }, ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]] }
                        ]
                    }
                },
                "SingletonLambdaTestCustomResourceProviderA9255269": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "def hello(): pass"
                        },
                        "Handler": "index.hello",
                        "Role": {
                            "Fn::GetAtt": [
                                "SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C",
                                "Arn"
                            ]
                        },
                        "Runtime": "python2.7",
                        "Timeout": 300
                    },
                    "DependsOn": [
                        "SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C"
                    ]
                },
                "Custom1D319B237": {
                    "Type": "AWS::CloudFormation::CustomResource",
                    "Properties": {
                        "ServiceToken": {
                            "Fn::GetAtt": [
                                "SingletonLambdaTestCustomResourceProviderA9255269",
                                "Arn"
                            ]
                        }
                    }
                },
                "Custom2DD5FB44D": {
                    "Type": "AWS::CloudFormation::CustomResource",
                    "Properties": {
                        "ServiceToken": {
                            "Fn::GetAtt": [
                                "SingletonLambdaTestCustomResourceProviderA9255269",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'custom resources can specify a resource type that starts with Custom::'(test) {
        const stack = new cdk.Stack();
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            resourceType: 'Custom::MyCustomResourceType',
            provider: lib_1.CustomResourceProvider.topic(new sns.Topic(stack, 'Provider'))
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::MyCustomResourceType'));
        test.done();
    },
    'fails if custom resource type is invalid': {
        'does not start with "Custom::"'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'NoCustom::MyCustomResourceType',
                    provider: lib_1.CustomResourceProvider.topic(new sns.Topic(stack, 'Provider'))
                });
            }, /Custom resource type must begin with "Custom::"/);
            test.done();
        },
        'has invalid characters'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::My Custom?ResourceType',
                    provider: lib_1.CustomResourceProvider.topic(new sns.Topic(stack, 'Provider'))
                });
            }, /Custom resource type name can only include alphanumeric characters and/);
            test.done();
        },
        'is longer than 60 characters'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::0123456789012345678901234567890123456789012345678901234567891',
                    provider: lib_1.CustomResourceProvider.topic(new sns.Topic(stack, 'Provider'))
                });
            }, /Custom resource type length > 60/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5yZXNvdXJjZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QucmVzb3VyY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUF1RDtBQUN2RCw4Q0FBK0M7QUFDL0Msd0NBQXlDO0FBQ3pDLG9DQUFxQztBQUVyQyxnQ0FBZ0U7QUF5SWhFLE1BQU0sa0JBQW1CLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFDNUMsWUFBWSxLQUFvQixFQUFFLEVBQVU7UUFDMUMsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGVBQWUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ25FLElBQUksRUFBRSw0QkFBNEI7WUFDbEMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQztZQUNoRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRO1lBQ2hDLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLE9BQU8sRUFBRSxHQUFHO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxvQkFBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbkMsUUFBUSxFQUFFLDRCQUFzQixDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUM7U0FDekQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBckpELGlCQUFTO0lBQ1Asc0RBQXNELENBQUMsSUFBVTtRQUMvRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLElBQUksa0JBQWtCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3pDLElBQUksa0JBQWtCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCw4REFBOEQsRUFBRTtvQkFDaEUsTUFBTSxFQUFFLGdCQUFnQjtvQkFDeEIsWUFBWSxFQUFFO3dCQUNaLDBCQUEwQixFQUFFOzRCQUM1QixXQUFXLEVBQUU7Z0NBQ1g7b0NBQ0EsUUFBUSxFQUFFLGdCQUFnQjtvQ0FDMUIsUUFBUSxFQUFFLE9BQU87b0NBQ2pCLFdBQVcsRUFBRTt3Q0FDWCxTQUFTLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLEVBQUU7cUNBQ3hFO2lDQUNBOzZCQUNGOzRCQUNELFNBQVMsRUFBRSxZQUFZO3lCQUN0Qjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsRUFBRSxVQUFVLEVBQUUsQ0FBRSxFQUFFLEVBQUU7d0NBQ2xCLE1BQU0sRUFBRSxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLDJEQUEyRDtxQ0FBRSxDQUFFLEVBQUM7eUJBQ3hHO3FCQUNGO2lCQUNBO2dCQUNELG1EQUFtRCxFQUFFO29CQUNyRCxNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixZQUFZLEVBQUU7d0JBQ1osTUFBTSxFQUFFOzRCQUNSLFNBQVMsRUFBRSxtQkFBbUI7eUJBQzdCO3dCQUNELFNBQVMsRUFBRSxhQUFhO3dCQUN4QixNQUFNLEVBQUU7NEJBQ1IsWUFBWSxFQUFFO2dDQUNaLDhEQUE4RDtnQ0FDOUQsS0FBSzs2QkFDTjt5QkFDQTt3QkFDRCxTQUFTLEVBQUUsV0FBVzt3QkFDdEIsU0FBUyxFQUFFLEdBQUc7cUJBQ2Y7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLDhEQUE4RDtxQkFDL0Q7aUJBQ0E7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ25CLE1BQU0sRUFBRSxxQ0FBcUM7b0JBQzdDLFlBQVksRUFBRTt3QkFDWixjQUFjLEVBQUU7NEJBQ2hCLFlBQVksRUFBRTtnQ0FDWixtREFBbUQ7Z0NBQ25ELEtBQUs7NkJBQ047eUJBQ0E7cUJBQ0Y7aUJBQ0E7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ25CLE1BQU0sRUFBRSxxQ0FBcUM7b0JBQzdDLFlBQVksRUFBRTt3QkFDWixjQUFjLEVBQUU7NEJBQ2hCLFlBQVksRUFBRTtnQ0FDWixtREFBbUQ7Z0NBQ25ELEtBQUs7NkJBQ047eUJBQ0E7cUJBQ0Y7aUJBQ0E7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3RUFBd0UsQ0FBQyxJQUFVO1FBQ2pGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7WUFDNUMsWUFBWSxFQUFFLDhCQUE4QjtZQUM1QyxRQUFRLEVBQUUsNEJBQXNCLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7U0FDekUsQ0FBQyxDQUFDO1FBQ0gsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDhCQUE4QixDQUFDLENBQUMsQ0FBQztRQUMvRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMENBQTBDLEVBQUU7UUFDMUMsZ0NBQWdDLENBQUMsSUFBVTtZQUN6QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUM1QyxZQUFZLEVBQUUsZ0NBQWdDO29CQUM5QyxRQUFRLEVBQUUsNEJBQXNCLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7aUJBQ3pFLENBQUMsQ0FBQztZQUNMLENBQUMsRUFBRSxpREFBaUQsQ0FBQyxDQUFDO1lBRXRELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx3QkFBd0IsQ0FBQyxJQUFVO1lBQ2pDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQzVDLFlBQVksRUFBRSxnQ0FBZ0M7b0JBQzlDLFFBQVEsRUFBRSw0QkFBc0IsQ0FBQyxLQUFLLENBQUMsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztpQkFDekUsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxFQUFFLHdFQUF3RSxDQUFDLENBQUM7WUFFN0UsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDhCQUE4QixDQUFDLElBQVU7WUFDdkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2YsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtvQkFDNUMsWUFBWSxFQUFFLHVFQUF1RTtvQkFDckYsUUFBUSxFQUFFLDRCQUFzQixDQUFDLEtBQUssQ0FBQyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO2lCQUN6RSxDQUFDLENBQUM7WUFDTCxDQUFDLEVBQUUsa0NBQWtDLENBQUMsQ0FBQztZQUV2QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBRUY7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IGxhbWJkYSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbmltcG9ydCBzbnMgPSByZXF1aXJlKCdAYXdzLWNkay9hd3Mtc25zJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQ3VzdG9tUmVzb3VyY2UsIEN1c3RvbVJlc291cmNlUHJvdmlkZXIgfSBmcm9tICcuLi9saWInO1xuXG4vLyB0c2xpbnQ6ZGlzYWJsZTpvYmplY3QtbGl0ZXJhbC1rZXktcXVvdGVzXG5cbmV4cG9ydCA9IHtcbiAgJ2N1c3RvbSByZXNvdXJjZSBpcyBhZGRlZCB0d2ljZSwgbGFtYmRhIGlzIGFkZGVkIG9uY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBUZXN0Q3VzdG9tUmVzb3VyY2Uoc3RhY2ssICdDdXN0b20xJyk7XG4gICAgbmV3IFRlc3RDdXN0b21SZXNvdXJjZShzdGFjaywgJ0N1c3RvbTInKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIlNpbmdsZXRvbkxhbWJkYVRlc3RDdXN0b21SZXNvdXJjZVByb3ZpZGVyU2VydmljZVJvbGU4MUZFQUI1Q1wiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpSb2xlXCIsXG4gICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgXCJBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3RzOkFzc3VtZVJvbGVcIixcbiAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IHtcbiAgICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wibGFtYmRhLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTWFuYWdlZFBvbGljeUFybnNcIjogW1xuICAgICAgICAgICAgeyBcIkZuOjpKb2luXCI6IFsgXCJcIiwgW1xuICAgICAgICAgICAgICBcImFybjpcIiwgeyBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCIgfSwgXCI6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcIiBdIF19XG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiU2luZ2xldG9uTGFtYmRhVGVzdEN1c3RvbVJlc291cmNlUHJvdmlkZXJBOTI1NTI2OVwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLFxuICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiQ29kZVwiOiB7XG4gICAgICAgICAgXCJaaXBGaWxlXCI6IFwiZGVmIGhlbGxvKCk6IHBhc3NcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJIYW5kbGVyXCI6IFwiaW5kZXguaGVsbG9cIixcbiAgICAgICAgICBcIlJvbGVcIjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlNpbmdsZXRvbkxhbWJkYVRlc3RDdXN0b21SZXNvdXJjZVByb3ZpZGVyU2VydmljZVJvbGU4MUZFQUI1Q1wiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUnVudGltZVwiOiBcInB5dGhvbjIuN1wiLFxuICAgICAgICAgIFwiVGltZW91dFwiOiAzMDBcbiAgICAgICAgfSxcbiAgICAgICAgXCJEZXBlbmRzT25cIjogW1xuICAgICAgICAgIFwiU2luZ2xldG9uTGFtYmRhVGVzdEN1c3RvbVJlc291cmNlUHJvdmlkZXJTZXJ2aWNlUm9sZTgxRkVBQjVDXCJcbiAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBcIkN1c3RvbTFEMzE5QjIzN1wiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6Q2xvdWRGb3JtYXRpb246OkN1c3RvbVJlc291cmNlXCIsXG4gICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgXCJTZXJ2aWNlVG9rZW5cIjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlNpbmdsZXRvbkxhbWJkYVRlc3RDdXN0b21SZXNvdXJjZVByb3ZpZGVyQTkyNTUyNjlcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiQ3VzdG9tMkRENUZCNDREXCI6IHtcbiAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpDbG91ZEZvcm1hdGlvbjo6Q3VzdG9tUmVzb3VyY2VcIixcbiAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIlNlcnZpY2VUb2tlblwiOiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiU2luZ2xldG9uTGFtYmRhVGVzdEN1c3RvbVJlc291cmNlUHJvdmlkZXJBOTI1NTI2OVwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjdXN0b20gcmVzb3VyY2VzIGNhbiBzcGVjaWZ5IGEgcmVzb3VyY2UgdHlwZSB0aGF0IHN0YXJ0cyB3aXRoIEN1c3RvbTo6Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgbmV3IEN1c3RvbVJlc291cmNlKHN0YWNrLCAnTXlDdXN0b21SZXNvdXJjZScsIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6TXlDdXN0b21SZXNvdXJjZVR5cGUnLFxuICAgICAgcHJvdmlkZXI6IEN1c3RvbVJlc291cmNlUHJvdmlkZXIudG9waWMobmV3IHNucy5Ub3BpYyhzdGFjaywgJ1Byb3ZpZGVyJykpXG4gICAgfSk7XG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6TXlDdXN0b21SZXNvdXJjZVR5cGUnKSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIGlmIGN1c3RvbSByZXNvdXJjZSB0eXBlIGlzIGludmFsaWQnOiB7XG4gICAgJ2RvZXMgbm90IHN0YXJ0IHdpdGggXCJDdXN0b206OlwiJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgQ3VzdG9tUmVzb3VyY2Uoc3RhY2ssICdNeUN1c3RvbVJlc291cmNlJywge1xuICAgICAgICAgIHJlc291cmNlVHlwZTogJ05vQ3VzdG9tOjpNeUN1c3RvbVJlc291cmNlVHlwZScsXG4gICAgICAgICAgcHJvdmlkZXI6IEN1c3RvbVJlc291cmNlUHJvdmlkZXIudG9waWMobmV3IHNucy5Ub3BpYyhzdGFjaywgJ1Byb3ZpZGVyJykpXG4gICAgICAgIH0pO1xuICAgICAgfSwgL0N1c3RvbSByZXNvdXJjZSB0eXBlIG11c3QgYmVnaW4gd2l0aCBcIkN1c3RvbTo6XCIvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdoYXMgaW52YWxpZCBjaGFyYWN0ZXJzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgQ3VzdG9tUmVzb3VyY2Uoc3RhY2ssICdNeUN1c3RvbVJlc291cmNlJywge1xuICAgICAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6TXkgQ3VzdG9tP1Jlc291cmNlVHlwZScsXG4gICAgICAgICAgcHJvdmlkZXI6IEN1c3RvbVJlc291cmNlUHJvdmlkZXIudG9waWMobmV3IHNucy5Ub3BpYyhzdGFjaywgJ1Byb3ZpZGVyJykpXG4gICAgICAgIH0pO1xuICAgICAgfSwgL0N1c3RvbSByZXNvdXJjZSB0eXBlIG5hbWUgY2FuIG9ubHkgaW5jbHVkZSBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBhbmQvKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpcyBsb25nZXIgdGhhbiA2MCBjaGFyYWN0ZXJzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgQ3VzdG9tUmVzb3VyY2Uoc3RhY2ssICdNeUN1c3RvbVJlc291cmNlJywge1xuICAgICAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MDEyMzQ1Njc4OTAxMjM0NTY3ODkwMTIzNDU2Nzg5MScsXG4gICAgICAgICAgcHJvdmlkZXI6IEN1c3RvbVJlc291cmNlUHJvdmlkZXIudG9waWMobmV3IHNucy5Ub3BpYyhzdGFjaywgJ1Byb3ZpZGVyJykpXG4gICAgICAgIH0pO1xuICAgICAgfSwgL0N1c3RvbSByZXNvdXJjZSB0eXBlIGxlbmd0aCA+IDYwLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgfSxcbn07XG5cbmNsYXNzIFRlc3RDdXN0b21SZXNvdXJjZSBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBzaW5nbGV0b25MYW1iZGEgPSBuZXcgbGFtYmRhLlNpbmdsZXRvbkZ1bmN0aW9uKHRoaXMsICdMYW1iZGEnLCB7XG4gICAgICB1dWlkOiAnVGVzdEN1c3RvbVJlc291cmNlUHJvdmlkZXInLFxuICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdkZWYgaGVsbG8oKTogcGFzcycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUHl0aG9uMjcsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGVsbG8nLFxuICAgICAgdGltZW91dDogMzAwLFxuICAgIH0pO1xuXG4gICAgbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHByb3ZpZGVyOiBDdXN0b21SZXNvdXJjZVByb3ZpZGVyLmxhbWJkYShzaW5nbGV0b25MYW1iZGEpXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==