"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const AWS = require("aws-sdk");
let organizationsClient;
/**
 * The isComplete handler is repeatedly invoked checking CreateAccountStatus until SUCCEEDED or FAILED.
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#asynchronous-providers-iscomplete
 */
async function handler(event) {
    console.log(`Request of type ${event.RequestType} received`);
    if (!organizationsClient) {
        organizationsClient = new AWS.Organizations({ region: "us-east-1" });
    }
    console.log("Payload: %j", event);
    let accountId;
    if (event.RequestType == "Create" || isLegacyPhysicalResourceId(event)) {
        const response = await organizationsClient
            // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#describeCreateAccountStatus-property
            .describeCreateAccountStatus({
            CreateAccountRequestId: isLegacyPhysicalResourceId(event)
                ? event.PhysicalResourceId
                : event.Data?.CreateAccountStatusId,
        })
            .promise();
        if (response.CreateAccountStatus?.State == "IN_PROGRESS") {
            return { IsComplete: false, Data: {} };
        }
        if (response.CreateAccountStatus?.State == "FAILED" &&
            response.CreateAccountStatus?.FailureReason != "EMAIL_ALREADY_EXISTS") {
            throw new Error(`Failed ${event.RequestType} Account ${response.CreateAccountStatus?.AccountName}, reason: ${response.CreateAccountStatus?.FailureReason}`);
        }
        if (response.CreateAccountStatus?.FailureReason == "EMAIL_ALREADY_EXISTS" &&
            event.ResourceProperties.ImportOnDuplicate) {
            const account = await findAccountByEmail(organizationsClient, event.ResourceProperties.Email);
            if (!account) {
                throw new Error(`Failed ${event.RequestType} Account ${response.CreateAccountStatus?.AccountName}, reason: ${response.CreateAccountStatus?.FailureReason}; could not find account in organization.`);
            }
            accountId = account.Id;
        }
        else if (response.CreateAccountStatus?.FailureReason == "EMAIL_ALREADY_EXISTS" &&
            !event.ResourceProperties.ImportOnDuplicate) {
            throw new Error(`Failed ${event.RequestType} Account ${response.CreateAccountStatus?.AccountName}, reason: ${response.CreateAccountStatus?.FailureReason}.`);
        }
        else {
            // State == SUCCEEDED
            accountId = response.CreateAccountStatus?.AccountId;
        }
    }
    else {
        accountId = event.PhysicalResourceId;
    }
    const response = await organizationsClient
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#describeAccount-property
        .describeAccount({ AccountId: accountId })
        .promise();
    // On delete, update or create move account to destination parent
    await move(organizationsClient, accountId, event.ResourceProperties?.ParentId);
    // On delete close account
    if (event.RequestType == "Delete" && event.ResourceProperties?.RemovalPolicy == "destroy") {
        await close(organizationsClient, accountId);
    }
    return {
        IsComplete: true,
        PhysicalResourceId: accountId,
        Data: {
            ...event.ResourceProperties,
            ...event.Data,
            AccountId: accountId,
            AccountArn: response.Account?.Arn,
            AccountName: response.Account?.Name,
            Email: response.Account?.Email,
        },
    };
}
exports.handler = handler;
const findCurrentParent = async (client, id) => {
    const response = await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#listParents-property
        .listParents({
        ChildId: id,
    })
        .promise();
    if (response.Parents?.length) {
        return response.Parents[0];
    }
    throw new Error(`Could not find parent for id '${id}'`);
};
const move = async (client, accountId, destinationParentId) => {
    if (!destinationParentId) {
        return;
    }
    const currentParent = await findCurrentParent(organizationsClient, accountId);
    if (destinationParentId == currentParent.Id) {
        return;
    }
    await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#moveAccount-property
        .moveAccount({
        AccountId: accountId,
        SourceParentId: currentParent.Id,
        DestinationParentId: destinationParentId,
    })
        .promise();
};
/**
 * Before aws-cdk-lib 2.15.0 the physical resource was determined in the onEventHandler and therefor the physical resource id was the account's CreateAccountStatusId.
 */
const isLegacyPhysicalResourceId = (event) => {
    return /car-[a-z0-9]{8,32}/.test(event.PhysicalResourceId);
};
const findAccountByEmail = async (client, email) => {
    let response = await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#listAccounts-property
        .listAccounts()
        .promise();
    for (const account of response.Accounts ?? []) {
        if (account.Email == email) {
            return account;
        }
    }
    while (response.NextToken) {
        response = await client
            // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#listAccounts-property
            .listAccounts({ NextToken: response.NextToken })
            .promise();
        for (const account of response.Accounts ?? []) {
            if (account.Email == email) {
                return account;
            }
        }
    }
    return undefined;
};
const close = async (client, accountId) => {
    await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#closeAccount-property
        .closeAccount({
        AccountId: accountId,
    });
};
//# sourceMappingURL=data:application/json;base64,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