"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Account = exports.IamUserAccessToBilling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const pascal_case_1 = require("pascal-case");
const account_provider_1 = require("./account-provider");
const delegated_administrator_1 = require("./delegated-administrator");
const policy_attachment_1 = require("./policy-attachment");
const tag_resource_1 = require("./tag-resource");
const validators_1 = require("./validators");
/**
 * @see https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/control-access-billing.html#ControllingAccessWebsite-Activate
 */
var IamUserAccessToBilling;
(function (IamUserAccessToBilling) {
    /**
     * If set to ALLOW, the new account enables IAM users to access account billing information if they have the required permissions.
     */
    IamUserAccessToBilling["ALLOW"] = "ALLOW";
    /**
     * If set to DENY, only the root user of the new account can access account billing information.
     */
    IamUserAccessToBilling["DENY"] = "DENY";
})(IamUserAccessToBilling = exports.IamUserAccessToBilling || (exports.IamUserAccessToBilling = {}));
/**
 * Creates or imports an AWS account that is automatically a member of the organization whose credentials made the request. AWS Organizations automatically copies the information from the management account to the new member account
 */
class Account extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.KEY_VALUE, "Custom::Organizations_Account");
        this.scope = scope;
        const { email, accountName, roleName, iamUserAccessToBilling, parent, importOnDuplicate, removalPolicy } = props;
        if (!validators_1.Validators.of().email(email)) {
            aws_cdk_lib_1.Annotations.of(this).addError("The account's email must be of type string and between 6 and 64 characters long.");
        }
        if (!validators_1.Validators.of().accountName(accountName)) {
            aws_cdk_lib_1.Annotations.of(this).addError("The account's name must be of type string and between 1 and 50 characters long.");
        }
        const createAccountProvider = account_provider_1.AccountProvider.getOrCreate(this);
        const account = new aws_cdk_lib_1.CustomResource(this, "CreateAccount", {
            serviceToken: createAccountProvider.provider.serviceToken,
            resourceType: "Custom::Organizations_Account",
            properties: {
                Email: email,
                AccountName: accountName,
                RoleName: roleName ?? "OrganizationAccountAccessRole",
                IamUserAccessToBilling: iamUserAccessToBilling ?? IamUserAccessToBilling.ALLOW,
                ParentId: parent?.identifier(),
                ImportOnDuplicate: String(importOnDuplicate ?? true),
                RemovalPolicy: removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN,
            },
        });
        this.accountId = account.getAtt("AccountId").toString();
        this.accountArn = account.getAtt("AccountArn").toString();
        this.accountName = account.getAtt("AccountName").toString();
        this.email = account.getAtt("Email").toString();
        this.resource = account;
        const tagResource = new tag_resource_1.TagResource(this, "Tags", { resourceId: this.accountId, tags: this.tags.renderedTags });
        tagResource.node.addDependency(account);
    }
    identifier() {
        return this.accountId;
    }
    /**
     * Enables trusted access for the AWS service (trusted service) as <strong>Delegated Administrator</strong>, which performs tasks in your organization and its accounts on your behalf.
     *
     * @param {string} servicePrincipal The supported AWS service that you specify
     */
    delegateAdministrator(servicePrincipal) {
        const delegatedAdministrator = new delegated_administrator_1.DelegatedAdministrator(this.scope, `Delegate${pascal_case_1.pascalCase(servicePrincipal)}-${aws_cdk_lib_1.Names.nodeUniqueId(this.node)}`, {
            account: this,
            servicePrincipal: servicePrincipal,
        });
        delegatedAdministrator.node.addDependency(this.resource);
    }
    /**
     * Attach a policy. Before you can attach the policy, you must enable that policy type for use. You can use policies when you have all features enabled.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html
     */
    attachPolicy(policy) {
        const policyAttachment = new policy_attachment_1.PolicyAttachment(this.scope, `PolicyAttachment-${aws_cdk_lib_1.Names.nodeUniqueId(this.node)}-${aws_cdk_lib_1.Names.nodeUniqueId(policy.node)}`, {
            target: this,
            policy: policy,
        });
        policyAttachment.node.addDependency(this.resource, policy);
    }
}
exports.Account = Account;
_a = JSII_RTTI_SYMBOL_1;
Account[_a] = { fqn: "@pepperize/cdk-organizations.Account", version: "0.7.227" };
//# sourceMappingURL=data:application/json;base64,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