"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Root = exports.Organization = exports.FeatureSet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam = require("aws-cdk-lib/aws-iam");
const custom_resources = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const pascal_case_1 = require("pascal-case");
const dependency_chain_1 = require("./dependency-chain");
const enable_aws_service_access_1 = require("./enable-aws-service-access");
const enable_policy_type_1 = require("./enable-policy-type");
const organization_provider_1 = require("./organization-provider");
const policy_attachment_1 = require("./policy-attachment");
const tag_resource_1 = require("./tag-resource");
/**
 * Specifies the feature set supported by the new organization. Each feature set supports different levels of functionality.
 *
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set
 */
var FeatureSet;
(function (FeatureSet) {
    /**
     * All member accounts have their bills consolidated to and paid by the management account. For more information, see [Consolidated billing](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set-cb-only) in the AWS Organizations User Guide. The consolidated billing feature subset isn’t available for organizations in the AWS GovCloud (US) Region.
     */
    FeatureSet["CONSOLIDATED_BILLING"] = "CONSOLIDATED_BILLING";
    /**
     * In addition to all the features supported by the consolidated billing feature set, the management account can also apply any policy type to any member account in the organization. For more information, see [All features](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set-all) in the AWS Organizations User Guide.
     */
    FeatureSet["ALL"] = "ALL";
})(FeatureSet = exports.FeatureSet || (exports.FeatureSet = {}));
class Organization extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const featureSet = props.featureSet || FeatureSet.ALL;
        const organizationProvider = organization_provider_1.OrganizationProvider.getOrCreate(this);
        this.resource = new aws_cdk_lib_1.CustomResource(this, "Organization", {
            serviceToken: organizationProvider.provider.serviceToken,
            resourceType: "Custom::Organizations_Organization",
            properties: {
                FeatureSet: featureSet,
            },
        });
        this.organizationId = this.resource.getAtt("Id").toString();
        this.organizationArn = this.resource.getAtt("Arn").toString();
        this.featureSet = this.resource.getAtt("FeatureSet").toString();
        this.managementAccountArn = this.resource.getAtt("MasterAccountArn").toString();
        this.managementAccountId = this.resource.getAtt("MasterAccountId").toString();
        this.managementAccountEmail = this.resource.getAtt("MasterAccountEmail").toString();
        this.principal = new aws_iam.OrganizationPrincipal(this.organizationId);
        this.root = new Root(this, "Root");
        this.root.node.addDependency(this.resource);
    }
    /**
     * Describe the organization that the current account belongs to.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    static of(scope, id) {
        class Import extends constructs_1.Construct {
            constructor() {
                super(scope, id);
                const resource = new custom_resources.AwsCustomResource(scope, "CustomResource", {
                    resourceType: "Custom::Organizations_ImportOrganization",
                    onCreate: {
                        service: "Organizations",
                        action: "describeOrganization",
                        region: "us-east-1",
                        parameters: {},
                        physicalResourceId: custom_resources.PhysicalResourceId.fromResponse("Organization.Id"),
                    },
                    onUpdate: {
                        service: "Organizations",
                        action: "describeOrganization",
                        region: "us-east-1",
                        parameters: {},
                        physicalResourceId: custom_resources.PhysicalResourceId.fromResponse("Organization.Id"),
                    },
                    installLatestAwsSdk: false,
                    policy: custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                        resources: custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
                    }),
                });
                this.featureSet = resource.getResponseField("Organization.FeatureSet");
                this.managementAccountArn = resource.getResponseField("Organization.MasterAccountArn");
                this.managementAccountEmail = resource.getResponseField("Organization.MasterAccountEmail");
                this.managementAccountId = resource.getResponseField("Organization.MasterAccountId");
                this.organizationArn = resource.getResponseField("Organization.Arn");
                this.organizationId = resource.getResponseField("Organization.Id");
                this.principal = new aws_iam.OrganizationPrincipal(this.organizationId);
            }
        }
        return new Import();
    }
    /**
     * Enables trusted access for a supported AWS service (trusted service), which performs tasks in your organization and its accounts on your behalf.
     * @param servicePrincipal The supported AWS service that you specify
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services_list.html
     */
    enableAwsServiceAccess(servicePrincipal) {
        const enableAwsServiceAccess = new enable_aws_service_access_1.EnableAwsServiceAccess(this, `Enable${pascal_case_1.pascalCase(servicePrincipal)}`, {
            servicePrincipal: servicePrincipal,
        });
        enableAwsServiceAccess.node.addDependency(this.resource);
    }
    /**
     * Enables policy types in the following two broad categories: Authorization policies and Management policies.
     * @param policyType: the type of the policy that you specify
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html#orgs-policy-types
     */
    enablePolicyType(policyType) {
        this.root.enablePolicyType(policyType);
    }
    /**
     * Attach a policy. Before you can attach the policy, you must enable that policy type for use. You can use policies when you have all features enabled.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html
     */
    attachPolicy(policy) {
        this.root.attachPolicy(policy);
    }
}
exports.Organization = Organization;
_a = JSII_RTTI_SYMBOL_1;
Organization[_a] = { fqn: "@pepperize/cdk-organizations.Organization", version: "0.7.227" };
/**
 * The parent container for all the accounts for your organization. If you apply a policy to the root, it applies to all organizational units (OUs) and accounts in the organization.
 * <strong>Currently, you can have only one root. AWS Organizations automatically creates it for you when you create an organization.</strong>
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html
 */
class Root extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.KEY_VALUE, "Custom::Organizations_Root");
        this.scope = scope;
        this.resource = new custom_resources.AwsCustomResource(this, "RootCustomResource", {
            resourceType: "Custom::Organizations_Root",
            onCreate: {
                service: "Organizations",
                action: "listRoots",
                region: "us-east-1",
                physicalResourceId: custom_resources.PhysicalResourceId.fromResponse("Roots.0.Id"),
            },
            onUpdate: {
                service: "Organizations",
                action: "listRoots",
                region: "us-east-1",
                physicalResourceId: custom_resources.PhysicalResourceId.fromResponse("Roots.0.Id"),
            },
            onDelete: {
                service: "Organizations",
                action: "listRoots",
                region: "us-east-1",
            },
            installLatestAwsSdk: false,
            policy: custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        this.rootId = this.resource.getResponseField("Roots.0.Id"); // Returns first root id. It seems AWS Organizations doesn't contain multiple roots.
        const stack = aws_cdk_lib_1.Stack.of(this);
        aws_cdk_lib_1.Aspects.of(stack).add(new dependency_chain_1.DependencyChain()); // sequentially chain organization resources which can't be deployed in parallel
        const tagResource = new tag_resource_1.TagResource(this, "Tags", { resourceId: this.rootId, tags: this.tags.renderedTags });
        tagResource.node.addDependency(this.resource);
    }
    identifier() {
        return this.rootId;
    }
    /**
     * Attach a policy. Before you can attach the policy, you must enable that policy type for use. You can use policies when you have all features enabled.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html
     */
    attachPolicy(policy) {
        const policyAttachment = new policy_attachment_1.PolicyAttachment(this.scope, `PolicyAttachment-${aws_cdk_lib_1.Names.nodeUniqueId(this.node)}-${aws_cdk_lib_1.Names.nodeUniqueId(policy.node)}`, {
            target: this,
            policy: policy,
        });
        policyAttachment.node.addDependency(this.resource, policy);
    }
    /**
     * Enables and disables Enables a policy type. After you enable a policy type in a root, you can attach policies of that type to the root, any organizational unit (OU), or account in that root.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_enable-disable.html
     */
    enablePolicyType(policyType) {
        const enablePolicyType = new enable_policy_type_1.EnablePolicyType(this.scope, `Enable${pascal_case_1.pascalCase(policyType)}`, {
            root: this,
            policyType: policyType,
        });
        enablePolicyType.node.addDependency(this.resource);
    }
}
exports.Root = Root;
_b = JSII_RTTI_SYMBOL_1;
Root[_b] = { fqn: "@pepperize/cdk-organizations.Root", version: "0.7.227" };
//# sourceMappingURL=data:application/json;base64,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