"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_sdk_1 = require("aws-sdk");
let organizationsClient;
/**
 * The onEvent handler is invoked whenever a resource lifecycle event for an organizational unit occurs
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#handling-lifecycle-events-onevent
 */
exports.handler = async (event) => {
    console.log(`Request of type ${event.RequestType} received`);
    if (!organizationsClient) {
        organizationsClient = new aws_sdk_1.Organizations({ region: "us-east-1" });
    }
    console.log("Payload: %j", event);
    const { ParentId, Name, ImportOnDuplicate, RemovalPolicy } = event.ResourceProperties;
    if (event.RequestType == "Create") {
        try {
            const organizationalUnit = await createOrganizationalUnit(organizationsClient, ParentId, Name);
            return {
                PhysicalResourceId: organizationalUnit.Id,
                Data: {
                    ...organizationalUnit,
                },
            };
        }
        catch (e) {
            const error = e;
            console.log(error);
            // https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html#API_CreateOrganizationalUnit_Errors
            if (error.code == "DuplicateOrganizationalUnitException" && ImportOnDuplicate == "true") {
                console.log(`Organizational unit already created, trying to find existing one in parent.`);
                const organizationalUnit = await findOrganizationalUnitByParentAndName(organizationsClient, ParentId, Name);
                return {
                    PhysicalResourceId: organizationalUnit.Id,
                    Data: {
                        ...organizationalUnit,
                    },
                };
            }
            else {
                throw error;
            }
        }
    }
    if (event.RequestType == "Update") {
        const organizationalUnit = await updateOrganizationalUnit(organizationsClient, event.PhysicalResourceId, Name);
        return {
            PhysicalResourceId: organizationalUnit.Id,
            Data: {
                ...organizationalUnit,
            },
        };
    }
    if (event.RequestType == "Delete" && RemovalPolicy == "destroy") {
        await deleteOrganizationalUnit(organizationsClient, event.PhysicalResourceId);
    }
    return {
        PhysicalResourceId: event.PhysicalResourceId,
        Data: {
            ...event.ResourceProperties,
        },
    };
};
const findOrganizationalUnitByParentAndName = async (client, parentId, name) => {
    let response = await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#listOrganizationalUnitsForParent-property
        .listOrganizationalUnitsForParent({ ParentId: parentId })
        .promise();
    for (const organizationalUnit of response.OrganizationalUnits ?? []) {
        if (organizationalUnit.Name == name) {
            return organizationalUnit;
        }
    }
    while (response.NextToken) {
        response = await client
            .listOrganizationalUnitsForParent({ ParentId: parentId, NextToken: response.NextToken })
            .promise();
        for (const organizationalUnit of response.OrganizationalUnits ?? []) {
            if (organizationalUnit.Name == name) {
                return organizationalUnit;
            }
        }
    }
    throw new Error(`Organizational unit '${name}' not found in '${parentId}'`);
};
const createOrganizationalUnit = async (client, parentId, name) => {
    const response = await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#createOrganizationalUnit-property
        .createOrganizationalUnit({
        ParentId: parentId,
        Name: name,
    })
        .promise();
    console.log("Creating organizational unit: %j", response);
    if (!response.OrganizationalUnit) {
        throw new Error("Could not create organizational unit, reason: empty response");
    }
    return response.OrganizationalUnit;
};
const updateOrganizationalUnit = async (client, id, name) => {
    const response = await client
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#updateOrganizationalUnit-property
        .updateOrganizationalUnit({
        OrganizationalUnitId: id,
        Name: name,
    })
        .promise();
    console.log("Updating organizational unit: %j", response);
    if (!response.OrganizationalUnit) {
        throw new Error("Could not update organizational unit, reason: empty response");
    }
    return response.OrganizationalUnit;
};
const deleteOrganizationalUnit = async (client, id) => {
    // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#deleteOrganizationalUnit-property
    await client
        .deleteOrganizationalUnit({
        OrganizationalUnitId: id,
    })
        .promise();
};
//# sourceMappingURL=data:application/json;base64,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