"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationalUnitProvider = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const on_event_handler_function_1 = require("./on-event-handler-function");
/**
 * Creates a custom resource provider to create the organizational unit in AWS organization.
 *
 * <ul>
 *   <li><strong>If the organizational unit already exists, it will be imported if `ImportOnDuplicate` is true.</strong>
 *   <li><strong>Only an emptied organizational unit can be deleted!</strong></li>
 * </ul>
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#provider-framework
 */
class OrganizationalUnitProvider extends aws_cdk_lib_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.onEventHandler = new on_event_handler_function_1.OnEventHandlerFunction(this, "OnEventHandlerFunction", {
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "organizations:CreateOrganizationalUnit",
                        "organizations:DescribeOrganizationalUnit",
                        "organizations:UpdateOrganizationalUnit",
                        "organizations:DeleteOrganizationalUnit",
                        "organizations:ListOrganizationalUnitsForParent",
                    ],
                    resources: ["*"],
                }),
            ],
        });
        this.provider = new custom_resources_1.Provider(this, "Provider", {
            onEventHandler: this.onEventHandler,
        });
    }
    /**
     * Retrieve OrganizationalUnitProvider as stack singleton resource.
     *
     * @see https://github.com/aws/aws-cdk/issues/5023
     */
    static getOrCreate(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = "cdk-organizations.OrganizationalUnitProvider";
        const existing = stack.node.tryFindChild(id);
        return existing || new OrganizationalUnitProvider(stack, id, {});
    }
}
exports.OrganizationalUnitProvider = OrganizationalUnitProvider;
//# sourceMappingURL=data:application/json;base64,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