"""Hidden command that pre-generates recovery scripts for passthrough flows."""

from pathlib import Path

import click

from workstack.cli.core import discover_repo_context
from workstack.cli.shell_utils import render_cd_script, write_script_to_temp
from workstack.core.context import WorkstackContext


def generate_recovery_script(ctx: WorkstackContext) -> Path | None:
    """Create a recovery script that returns to the repo root if cwd vanishes.

    This helper intentionally guards against runtime cwd races:
    - Path.cwd() may raise FileNotFoundError if the directory vanished between invocations.
    - discover_repo_context() performs the authoritative repo lookup; probing earlier provides
      no additional safety and merely repeats the work.
    - Returning None signals that graceful degradation is preferred to exploding at the boundary.
    """
    try:
        current_dir = Path.cwd()
    except FileNotFoundError:
        return None

    if not current_dir.exists():
        return None

    try:
        repo = discover_repo_context(ctx, current_dir)
    except (FileNotFoundError, ValueError):
        return None

    script_content = render_cd_script(
        repo.root,
        comment="workstack passthrough recovery script",
        success_message="✓ Workstack detected a removed directory and returned to repo root.",
    )

    script_path = write_script_to_temp(
        script_content,
        command_name="prepare",
        comment="pre-generated by __prepare_cwd_recovery",
    )

    return script_path


@click.command(
    "__prepare_cwd_recovery",
    hidden=True,
    add_help_option=False,
)
@click.pass_obj
def prepare_cwd_recovery_cmd(ctx: WorkstackContext) -> None:
    """Emit a recovery script if we are inside a managed repository."""
    script_path = generate_recovery_script(ctx)
    if script_path is None:
        return

    click.echo(str(script_path), nl=False)
