"""Saf3AI SDK main entry point."""

from typing import Optional, Dict, Union, Any
from opentelemetry.sdk.trace import SpanProcessor
from opentelemetry.sdk.trace.export import SpanExporter

from saf3ai_sdk.config import Config
from saf3ai_sdk.core.tracer import tracer, TracingCore
from saf3ai_sdk.logging import logger

# Import decorators for easy access
from saf3ai_sdk.core.decorators import trace, agent, task, tool, workflow

# Import security scanning functionality (consolidated from adk-otel)
from saf3ai_sdk.callbacks import (
    register_security_callback,
    get_callback_manager,
    LLMSecurityCallback,
    LLMCallbackManager
)
from saf3ai_sdk.adk_callbacks import create_security_callback
from saf3ai_sdk.scanner import scan_prompt, scan_response, scan_prompt_and_response

__all__ = [
    "init",
    "tracer",
    "get_tracer",
    "trace",
    "agent",
    "task",
    "tool",
    "workflow",
    "reset_session",
    # Security scanning (from adk-otel, now consolidated)
    "register_security_callback",
    "get_callback_manager",
    "LLMSecurityCallback",
    "LLMCallbackManager",
    "create_security_callback",
    "scan_prompt",
    "scan_response",
    "scan_prompt_and_response",
]

def get_tracer(name: str = "saf3ai"):
    """
    Get a tracer from the Saf3AI SDK's TracerProvider.
    
    This ensures spans are processed by our span processors.
    
    Args:
        name: Name of the tracer
        
    Returns:
        Tracer instance from our TracerProvider
    """
    return tracer.get_tracer(name)

def init(
    service_name: Optional[str] = None,
    environment: Optional[str] = None,
    otlp_endpoint: Optional[str] = None,
    otlp_headers: Optional[Dict[str, str]] = None,
    log_level: Optional[Union[str, int]] = None,
    debug_mode: Optional[bool] = None,
    console_output: Optional[bool] = None,
    auto_instrument_adk: Optional[bool] = None,
    exporter: Optional[SpanExporter] = None,
    processor: Optional[SpanProcessor] = None,
    tags: Optional[Dict[str, str]] = None,
    **kwargs
) -> TracingCore:
    """
    Initialize the Saf3AI SDK.
    
    Configures the global tracer, sets up exporters, and applies
    automatic instrumentation.
    """
    
    config = Config()
    config.configure(
        service_name=service_name,
        environment=environment,
        otlp_endpoint=otlp_endpoint,
        otlp_headers=otlp_headers,
        log_level=log_level,
        debug_mode=debug_mode,
        console_output=console_output,
        auto_instrument_adk=auto_instrument_adk,
        exporter=exporter,
        processor=processor,
        tags=tags,
        **kwargs
    )
    
    # Initialize the core tracer
    tracer.initialize(config)
    
    # Apply auto-instrumentation if requested
    if auto_instrument_adk:
        from saf3ai_sdk.instrumentation import instrument_adk
        instrument_adk(tracer.get_tracer("saf3ai-adk"))
            
    return tracer

def reset_session():
    """
    Reset the current persistent session to create a new session.
    Call this when you want to start a new ADK web session.
    This will create a new persistent session ID and clean up old session data.
    """
    try:
        from saf3ai_sdk.instrumentation.adk_instrumentation import reset_persistent_session
        return reset_persistent_session()
    except ImportError:
        print("🔍 DEBUG: ADK instrumentation not available")
        return None

# Alias init_telemetry for compatibility (was init_telemetry in adk-otel)
init_telemetry = init