# Django-CFG RPC Client 📡

Lightweight synchronous RPC client for Django applications to communicate with **django-ipc** WebSocket servers via Redis.

**NEW**: ✨ Built-in RPC Dashboard for real-time monitoring! See [Dashboard README](./dashboard/README.md)

## 🎯 Key Features

- ✅ **100% Synchronous** - No async/await in Django, works with WSGI
- ✅ **Type-Safe** - Full Pydantic 2 support when django-ipc installed
- ✅ **Optional Dependency** - Works without django-ipc (dict-based fallback)
- ✅ **Redis IPC** - Reliable communication via Streams + Lists
- ✅ **Connection Pooling** - Automatic Redis connection management
- ✅ **Singleton Pattern** - Single client instance across Django app

---

## 📦 Installation

### Option 1: With django-ipc (Recommended)

```bash
pip install django-cfg django-ipc
```

### Option 2: Standalone (Basic)

```bash
pip install django-cfg redis
```

---

## ⚙️ Configuration

### Django Settings (via django-cfg)

```python
# config.py
from django_cfg import DjangoConfig
from django_cfg.modules.django_ipc_client.config import DjangoCfgRPCConfig

class MyProjectConfig(DjangoConfig):
    project_name = "My Project"

    # RPC Client Configuration
    django_ipc = DjangoCfgRPCConfig(
        enabled=True,
        redis_url="redis://localhost:6379/2",
        rpc_timeout=30,
    )

config = MyProjectConfig()

# settings.py
from config import config

# Settings are automatically generated by django-cfg orchestrator
# DJANGO_CFG_RPC settings will be available in Django settings
globals().update(config.get_all_settings())
```

**✨ Automatic Integration:**
- ✅ `DJANGO_CFG_RPC` settings are auto-generated
- ✅ Dashboard app auto-added to `INSTALLED_APPS`
- ✅ Dashboard URLs auto-registered at `/admin/rpc/`
- ✅ RPC tracked in `DJANGO_CFG_INTEGRATIONS` list

### Environment Variables

```bash
DJANGO_CFG_RPC__ENABLED=true
DJANGO_CFG_RPC__REDIS_URL=redis://localhost:6379/2
DJANGO_CFG_RPC__RPC_TIMEOUT=30
```

---

## 🚀 Usage

### With django-ipc Models (Type-Safe)

```python
# views.py
from django_cfg.modules.django_ipc_client import get_rpc_client
from django_ipc.models import NotificationRequest, NotificationResponse

rpc = get_rpc_client()

def notify_user(request, user_id):
    """Send notification to user via WebSocket."""

    result: NotificationResponse = rpc.call(
        method="send_notification",
        params=NotificationRequest(
            user_id=user_id,
            type="order_update",
            title="Order Confirmed",
            message="Your order #12345 has been confirmed",
            priority="high",
        ),
        result_model=NotificationResponse,
        timeout=10,
    )

    return JsonResponse({
        "delivered": result.delivered,
        "connections": result.connections,
    })
```

### Without django-ipc (Dict-Based)

```python
from django_cfg.modules.django_ipc_client import get_rpc_client

rpc = get_rpc_client()

def notify_user(request, user_id):
    """Send notification using dict params."""

    result = rpc.call_dict(
        method="send_notification",
        params={
            "user_id": user_id,
            "type": "order_update",
            "title": "Order Confirmed",
            "message": "Your order #12345 has been confirmed",
        },
        timeout=10,
    )

    return JsonResponse({
        "delivered": result.get("delivered", False),
    })
```

### Fire-and-Forget (No Response Wait)

```python
# For operations where you don't need the result
rpc.fire_and_forget(
    method="log_event",
    params={"event": "user_login", "user_id": "123"}
)
```

---

## 🔧 API Reference

### `get_rpc_client(force_new=False)`

Get singleton RPC client instance.

**Args:**
- `force_new` (bool): Force create new instance (for testing)

**Returns:**
- `DjangoCfgRPCClient` instance

---

### `DjangoCfgRPCClient.call(method, params, result_model=None, timeout=None)`

Make synchronous RPC call.

**Args:**
- `method` (str): RPC method name
- `params` (BaseModel | dict): Parameters (Pydantic model or dict)
- `result_model` (Type[BaseModel], optional): Expected result model class
- `timeout` (int, optional): Timeout override in seconds

**Returns:**
- Pydantic model instance (if `result_model` provided) or dict

**Raises:**
- `RPCTimeoutError`: If timeout exceeded
- `RPCRemoteError`: If remote execution failed

---

### `DjangoCfgRPCClient.call_dict(method, params, timeout=None)`

Make RPC call with dict params (no Pydantic).

**Args:**
- `method` (str): RPC method name
- `params` (dict): Dictionary with parameters
- `timeout` (int, optional): Timeout override in seconds

**Returns:**
- Dictionary with result

---

### `DjangoCfgRPCClient.fire_and_forget(method, params)`

Send RPC request without waiting for response.

**Args:**
- `method` (str): RPC method name
- `params` (BaseModel | dict): Parameters

**Returns:**
- str: Message ID from Redis Stream

---

### `DjangoCfgRPCClient.health_check(timeout=5)`

Check if RPC system is healthy.

**Args:**
- `timeout` (int): Health check timeout in seconds

**Returns:**
- bool: True if healthy, False otherwise

---

## ⚠️ Error Handling

### Timeout Handling

```python
from django_cfg.modules.django_ipc_client import get_rpc_client, RPCTimeoutError

rpc = get_rpc_client()

try:
    result = rpc.call(method="slow_operation", params={...}, timeout=5)
except RPCTimeoutError as e:
    logger.warning(f"RPC timeout: {e.method} after {e.timeout_seconds}s")
    # Handle timeout (retry, fallback, etc.)
```

### Remote Error Handling

```python
from django_cfg.modules.django_ipc_client import RPCRemoteError

try:
    result = rpc.call(method="...", params={...})
except RPCRemoteError as e:
    logger.error(f"Remote error [{e.error.code}]: {e.error.message}")

    if e.is_retryable:
        # Retry after delay
        import time
        time.sleep(e.retry_after or 5)
        result = rpc.call(...)  # Retry
    else:
        # Non-retryable error
        raise
```

---

## 🏗️ Architecture

```
┌─────────────────────────────────────────────┐
│       Django Application (WSGI/Sync)        │
│  ┌───────────────────────────────────────┐  │
│  │  django_ipc_client                │  │
│  │  - DjangoCfgRPCClient (sync)          │  │
│  │  - Redis Streams (requests)           │  │
│  │  - Redis Lists (responses)            │  │
│  └───────────────────────────────────────┘  │
└───────────────┬─────────────────────────────┘
                │
                │ Redis IPC
                │ (stream:requests → list:response:{cid})
                ▼
┌─────────────────────────────────────────────┐
│     django-ipc Server (Async)           │
│  - WebSocket Server                         │
│  - Connection Manager                       │
│  - RPC Handlers                             │
│  - Pydantic 2 Models                        │
└─────────────────────────────────────────────┘
```

---

## 🔗 Related

- **django-ipc** - WebSocket RPC server package
- **Old Module** - `django_rpc_old` (deprecated, kept for reference)

---

## 📝 Migration from django_rpc_old

```python
# Old (deprecated)
from django_cfg.modules.django_rpc import get_rpc_client

# New (current)
from django_cfg.modules.django_ipc_client import get_rpc_client
```

API remains mostly compatible. Main changes:
- Configuration key: `WEBSOCKET_RPC` → `DJANGO_CFG_RPC`
- Models now live in `django-ipc` package (optional dependency)
- Better error handling with specific exception types

---

**Status:** ✅ Production Ready
**Django-CFG Version:** 2.0+
**Python Version:** 3.10+


---

## 📊 RPC Dashboard (NEW!)

### Quick Start

Add dashboard to your Django URLs:

```python
# urls.py
from django.urls import path, include

urlpatterns = [
    # ... other URLs
    path('admin/rpc/', include('django_cfg.modules.django_ipc_client.dashboard.urls')),
]
```

Access at: **http://localhost:8000/admin/rpc/** (requires staff login)

### Features

- ✅ **Real-time monitoring** - Updates every 5 seconds
- ✅ **Request tracking** - View recent RPC calls
- ✅ **Notification stats** - Track sent notifications  
- ✅ **Method analytics** - See which methods are called
- ✅ **Health checks** - Monitor Redis and streams
- ✅ **Beautiful UI** - Tailwind CSS dark mode

See [Dashboard README](./dashboard/README.md) for full documentation.

