import unittest
import os
import threading
import uuid
from venv import create
import requests
import json
import sys
from decimal import Decimal # Import Decimal for consistent price/quantity handling in tests

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    TradingClientError,
    CreateOrderFailedError,
    NotLoggedInError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD
)

class TestCreateOrderE2E(unittest.TestCase):
    sdk_instance = None
    base_url = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SDK and retrieves the base URL for testing.
        Skips all tests in this class if initialization fails.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
            cls.base_url = os.environ.get('SPHERE_BACKEND_BASE_URL')
            if not cls.base_url:
                raise unittest.SkipTest("SPHERE_BACKEND_BASE_URL environment variable is not set.")
        except SDKInitializationError as e:
            raise unittest.SkipTest(f"SDK Initialization failed, skipping E2E tests: {e}.")
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup: {e}")

    def setUp(self):
        """
        Logs in before each test and prepares resources for handling async callbacks.
        """
        if self.sdk_instance and self.sdk_instance._is_logged_in:
            self.sdk_instance.logout()

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Login is a prerequisite for this test and it failed: {e}")
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be logged in for this test.")

        self.received_event = threading.Event()
        self.received_order_data = {}

    def tearDown(self):
        """
        Unsubscribes from events and logs out after each test to ensure isolation.
        """
        if self.sdk_instance:
            if hasattr(self.sdk_instance, '_user_order_callback') and self.sdk_instance._user_order_callback:
                try:
                    self.sdk_instance.unsubscribe_from_order_events()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during unsubscription in tearDown: {e}", file=sys.stderr)
            if self.sdk_instance._is_logged_in:
                try:
                    self.sdk_instance.logout()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during logout in tearDown: {e}", file=sys.stderr)

    def test_create_trader_flat_order_succeeds(self):
        """
        Tests that creating a flat order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        expiry = "Jun-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
            "externalId": f"order_{uuid.uuid4().hex}"
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=201" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlatOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            expiry=expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        try:
            order_result: sphere_sdk_types_pb2.OrderResponseDto = self.sdk_instance.create_trader_flat_order(order_request)
            
            self.assertIsNotNone(order_result)
            self.assertIsInstance(order_result.id, str)
            self.assertEqual(order_result.id, stubbed_create_order_response["externalId"])
            self.assertIsInstance(order_result.instance_id, str)
            self.assertTrue(order_result.instance_id)
            
        except CreateOrderFailedError as e:
            self.fail(f"SDK failed to create trader flat order: {e}")
        except Exception as e:
            self.fail(f"An unexpected error occurred during order creation: {e}")
        
        capture_url = f"{self.base_url}/_testing/price" 
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured order request from '{capture_url}': {e}")
            
        expected_captured_body = {"expiryIds": ['expiry-jun25-Id'], 'anonymous': False, "brokerCodes": [primary_broker_code, secondary_broker_code],
                              "priceType": 'Flat', "instrumentId": 'instrument-id',
                              "interestType": 'Live', "value": float(per_price_unit), "quantity": float(quantity_value),
                              "units": 'Kt', 'trackingTypes': [], "unitPeriod": 'Month',
                              "side": 'b', "goodUntil": 'endOfDay',
                              "preferredClearingCompanyId": 'ice-id',
                              "preferredClearingCompanyType": 'only'}
        
        self.maxDiff = None
        self.assertEqual(captured_body_json["requestBody"], expected_captured_body)

    def test_create_trader_flat_order_fails(self):
        """
        Tests that creating a flat order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        expiry = "Jun-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=500" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlatOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            expiry=expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_flat_order(order_request)
        
        self.assertEqual(
            "Create order failed: Failed to execute order.",
            str(cm.exception)
        )

    def test_create_trader_flat_invalid_instrument_fails(self):
        """
        Tests that creating a flat order for a trader succeeds and sends an invalid instrument
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        expiry = "Jun-25"
        instrument = "invalid"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlatOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            expiry=expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_flat_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Instrument.",
            str(cm.exception)
        )

    def test_create_trader_flat_invalid_expiry_fails(self):
        """
        Tests that creating a flat order for a trader succeeds and sends an invalid expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        expiry = "invalid-00"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlatOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            expiry=expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_flat_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_flat_invalid_broker_fails(self):
        """
        Tests that creating a flat order for a trader succeeds and sends an invalid broker
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        expiry = "Jun-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "INVALID"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlatOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            expiry=expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_flat_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Broker.",
            str(cm.exception)
        )

    def test_create_trader_spread_order_succeeds(self):
        """
        Tests that creating a spread order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
            "externalId": f"order_{uuid.uuid4().hex}"
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=201" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        try:
            order_result: sphere_sdk_types_pb2.OrderResponseDto = self.sdk_instance.create_trader_spread_order(order_request)
            
            self.assertIsNotNone(order_result)
            self.assertIsInstance(order_result.id, str)
            self.assertEqual(order_result.id, stubbed_create_order_response["externalId"])
            self.assertIsInstance(order_result.instance_id, str)
            self.assertTrue(order_result.instance_id)
            
        except CreateOrderFailedError as e:
            self.fail(f"SDK failed to create trader spread order: {e}")
        except Exception as e:
            self.fail(f"An unexpected error occurred during order creation: {e}")
        
        capture_url = f"{self.base_url}/_testing/price" 
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured order request from '{capture_url}': {e}")
            
        expected_captured_body = {"expiryIds": ['expiry-jun25-Id', 'expiry-aug25-Id'], 'anonymous': False, "brokerCodes": [primary_broker_code, secondary_broker_code],
                              "priceType": 'Spread', "instrumentId": 'instrument-id',
                              "interestType": 'Live', "value": float(per_price_unit), "quantity": float(quantity_value),
                              "units": 'Kt', 'trackingTypes': [], "unitPeriod": 'Month',
                              "side": 'b', "goodUntil": 'endOfDay',
                              "preferredClearingCompanyId": 'ice-id',
                              "preferredClearingCompanyType": 'only'}
        
        self.maxDiff = None
        self.assertEqual(captured_body_json["requestBody"], expected_captured_body)

    def test_create_trader_spread_order_fails(self):
        """
        Tests that creating a spread order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=500" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_spread_order(order_request)
        
        self.assertEqual(
            "Create order failed: Failed to execute order.",
            str(cm.exception)
        )

    def test_create_trader_spread_invalid_instrument_fails(self):
        """
        Tests that creating a spread order for a trader succeeds and sends an invalid instrument
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "invalid"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_spread_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Instrument.",
            str(cm.exception)
        )

    def test_create_trader_spread_invalid_front_expiry_fails(self):
        """
        Tests that creating a spread order for a trader succeeds and sends an invalid front expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "invalid-00"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_spread_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_spread_invalid_back_expiry_fails(self):
        """
        Tests that creating a spread order for a trader succeeds and sends an invalid back expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "invalid-00"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_spread_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_spread_invalid_broker_fails(self):
        """
        Tests that creating a spread order for a trader succeeds and sends an invalid broker
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "INVALID"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderSpreadOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_spread_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Broker.",
            str(cm.exception)
        )

    def test_create_trader_strip_order_succeeds(self):
        """
        Tests that creating a strip order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
            "externalId": f"order_{uuid.uuid4().hex}"
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=201" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        try:
            order_result: sphere_sdk_types_pb2.OrderResponseDto = self.sdk_instance.create_trader_strip_order(order_request)
            
            self.assertIsNotNone(order_result)
            self.assertIsInstance(order_result.id, str)
            self.assertEqual(order_result.id, stubbed_create_order_response["externalId"])
            self.assertIsInstance(order_result.instance_id, str)
            self.assertTrue(order_result.instance_id)
            
        except CreateOrderFailedError as e:
            self.fail(f"SDK failed to create trader strip order: {e}")
        except Exception as e:
            self.fail(f"An unexpected error occurred during order creation: {e}")
        
        capture_url = f"{self.base_url}/_testing/price" 
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured order request from '{capture_url}': {e}")
            
        expected_captured_body = {"expiryIds": ['expiry-jun25-Id', 'expiry-aug25-Id'], 'anonymous': False, "brokerCodes": [primary_broker_code, secondary_broker_code],
                              "priceType": 'Strip', "instrumentId": 'instrument-id',
                              "interestType": 'Live', "value": float(per_price_unit), "quantity": float(quantity_value),
                              "units": 'Kt', 'trackingTypes': [], "unitPeriod": 'Month',
                              "side": 'b', "goodUntil": 'endOfDay',
                              "preferredClearingCompanyId": 'ice-id',
                              "preferredClearingCompanyType": 'only'}
        
        self.maxDiff = None
        self.assertEqual(captured_body_json["requestBody"], expected_captured_body)

    def test_create_trader_strip_order_fails(self):
        """
        Tests that creating a strip order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=500" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_strip_order(order_request)
        
        self.assertEqual(
            "Create order failed: Failed to execute order.",
            str(cm.exception)
        )

    def test_create_trader_strip_invalid_instrument_fails(self):
        """
        Tests that creating a strip order for a trader succeeds and sends an invalid instrument
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "invalid"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_strip_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Instrument.",
            str(cm.exception)
        )

    def test_create_trader_strip_invalid_front_expiry_fails(self):
        """
        Tests that creating a strip order for a trader succeeds and sends an invalid front expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "invalid-00"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_strip_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_strip_invalid_back_expiry_fails(self):
        """
        Tests that creating a strip order for a trader succeeds and sends an invalid back expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "invalid-00"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_strip_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_strip_invalid_broker_fails(self):
        """
        Tests that creating a strip order for a trader succeeds and sends an invalid broker
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        front_expiry = "Jun-25"
        back_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "INVALID"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderStripOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            front_expiry=front_expiry,
            back_expiry=back_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_strip_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Broker.",
            str(cm.exception)
        )

    def test_create_trader_fly_order_succeeds(self):
        """
        Tests that creating a fly order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        first_expiry = "Jun-25"
        second_expiry = "Jul-25"
        third_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
            "externalId": f"order_{uuid.uuid4().hex}"
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=201" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        try:
            order_result: sphere_sdk_types_pb2.OrderResponseDto = self.sdk_instance.create_trader_fly_order(order_request)
            
            self.assertIsNotNone(order_result)
            self.assertIsInstance(order_result.id, str)
            self.assertEqual(order_result.id, stubbed_create_order_response["externalId"])
            self.assertIsInstance(order_result.instance_id, str)
            self.assertTrue(order_result.instance_id)
            
        except CreateOrderFailedError as e:
            self.fail(f"SDK failed to create trader fly order: {e}")
        except Exception as e:
            self.fail(f"An unexpected error occurred during order creation: {e}")
        
        capture_url = f"{self.base_url}/_testing/price" 
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured order request from '{capture_url}': {e}")
            
        expected_captured_body = {"expiryIds": ['expiry-jun25-Id','expiry-jul25-Id', 'expiry-aug25-Id'], 'anonymous': False, "brokerCodes": [primary_broker_code, secondary_broker_code],
                              "priceType": 'Fly', "instrumentId": 'instrument-id',
                              "interestType": 'Live', "value": float(per_price_unit), "quantity": float(quantity_value),
                              "units": 'Kt', 'trackingTypes': [], "unitPeriod": 'Month',
                              "side": 'b', "goodUntil": 'endOfDay',
                              "preferredClearingCompanyId": 'ice-id',
                              "preferredClearingCompanyType": 'only'}
        
        self.maxDiff = None
        self.assertEqual(captured_body_json["requestBody"], expected_captured_body)

    def test_create_trader_fly_order_fails(self):
        """
        Tests that creating a fly order for a trader succeeds and sends the correct
        payload to the backend, including all nested DTOs.
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"
                
        first_expiry = "Jun-25"
        second_expiry = "Jul-25"
        third_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        stubbed_create_order_response = {
        }

        stub_order_creation_url = f"{self.base_url}/_testing/price?statusCode=500" 
        try:
            response = requests.post(stub_order_creation_url, json=stubbed_create_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub order creation endpoint '{stub_order_creation_url}': {e}")
            
        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_fly_order(order_request)
        
        self.assertEqual(
            "Create order failed: Failed to execute order.",
            str(cm.exception)
        )

    def test_create_trader_fly_invalid_instrument_fails(self):
        """
        Tests that creating a fly order for a trader succeeds and sends an invalid instrument
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"
                
        first_expiry = "Jun-25"
        second_expiry = "Jul-25"
        third_expiry = "Aug-25"
        instrument = "invalid"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_fly_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Instrument.",
            str(cm.exception)
        )

    def test_create_trader_fly_invalid_front_expiry_fails(self):
        """
        Tests that creating a fly order for a trader succeeds and sends an invalid front expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        first_expiry = "invalid-00"
        second_expiry = "Jul-25"
        third_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_fly_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_fly_invalid_back_expiry_fails(self):
        """
        Tests that creating a fly order for a trader succeeds and sends an invalid back expiry
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        first_expiry = "Jun-25"
        second_expiry = "Jul-25"
        third_expiry = "invalid-00"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "BC1"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_fly_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Expiry.",
            str(cm.exception)
        )

    def test_create_trader_fly_invalid_broker_fails(self):
        """
        Tests that creating a fly order for a trader succeeds and sends an invalid broker
        """

        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"

        first_expiry = "Jun-25"
        second_expiry = "Jul-25"
        third_expiry = "Aug-25"
        instrument = "testInstrument"
        quantity_value = Decimal("100")
        per_price_unit = Decimal("5.25")
        primary_broker_code = "INVALID"
        secondary_broker_code = "BC2"
        clearing_option_code = "ice"

        price_dto = sphere_sdk_types_pb2.OrderRequestPriceDto(
            per_price_unit=str(per_price_unit),
            quantity=str(quantity_value),
            ordered_clearing_options=[
                sphere_sdk_types_pb2.OrderRequestClearingOptionDto(code=clearing_option_code)
            ]
        )

        primary_broker_dto = sphere_sdk_types_pb2.OrderRequestBrokerDto(
            code=primary_broker_code
        )

        secondary_brokers_dtos = [
            sphere_sdk_types_pb2.OrderRequestBrokerDto(code=secondary_broker_code)
        ]

        parties_dto = sphere_sdk_types_pb2.TraderOrderRequestPartiesDto(
            primary_broker=primary_broker_dto,
            secondary_brokers=secondary_brokers_dtos
        )

        order_request = sphere_sdk_types_pb2.TraderFlyOrderRequestDto(
            idempotency_key=idempotency_key,
            side=sphere_sdk_types_pb2.ORDER_SIDE_BID,
            first_expiry=first_expiry,
            second_expiry=second_expiry,
            third_expiry=third_expiry,
            instrument_name=instrument,
            price=price_dto,
            parties=parties_dto
        )

        with self.assertRaises(CreateOrderFailedError) as cm:
            self.sdk_instance.create_trader_fly_order(order_request)
        
        self.assertEqual(
            "Create order failed: Invalid Broker.",
            str(cm.exception)
        )

if __name__ == '__main__':
    unittest.main()