"""
End-to-end tests for Sphere SDK Login and Logout functionality.
"""
import unittest
import sys

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD,
    INVALID_PASSWORD,
    INVALID_USERNAME
)

class TestLoginE2E(unittest.TestCase):
    sdk_instance = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SphereTradingClientSDK once for all tests in this class.
        If SDK initialization fails, tests will be skipped.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
        except SDKInitializationError as e:
            raise unittest.SkipTest(
                f"SDK Initialization failed, skipping E2E login tests: {e}. "
                f"Ensure Client DLL/shared library is accessible."
            )
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup, skipping E2E tests: {e}")

    def setUp(self):
        """
        Ensures a clean state before each test by logging out if necessary.
        """
        if self.sdk_instance and self.sdk_instance._is_logged_in:
            try:
                self.sdk_instance.logout()
            except Exception as e:
                self.fail(f"Failed to logout during test setup: {e}")
        self.assertFalse(self.sdk_instance._is_logged_in, "SDK should be logged out at the start of each test.")

    @classmethod
    def tearDownClass(cls):
        """
        Cleans up after all tests, ensuring logout.
        """
        if cls.sdk_instance and cls.sdk_instance._is_logged_in:
            try:
                cls.sdk_instance.logout()
            except Exception as e:
                print(f"Error during tearDownClass logout: {e}", file=sys.stderr)
        cls.sdk_instance = None

    def test_successful_login_and_logout(self):
        """
        Tests a successful login with valid credentials and subsequent logout.
        """
        self.assertFalse(self.sdk_instance._is_logged_in, "SDK should initially be logged out.")

        self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)

        self.assertTrue(self.sdk_instance._is_logged_in,
                        "SDK internal state _is_logged_in should be True after successful login.")

        self.sdk_instance.logout()
        self.assertFalse(self.sdk_instance._is_logged_in,
                         "SDK internal state _is_logged_in should be False after logout.")

    def test_login_with_invalid_password(self):
        """
        Tests that a login attempt with an invalid password raises LoginFailedError.
        """
        with self.assertRaises(LoginFailedError, msg="Login should fail with an invalid password."):
            self.sdk_instance.login(VALID_USERNAME, INVALID_PASSWORD)
        self.assertFalse(self.sdk_instance._is_logged_in, "SDK internal state should remain False after failed login.")

    def test_login_with_invalid_username(self):
        """
        Tests that a login attempt with an invalid username raises LoginFailedError.
        """
        with self.assertRaises(LoginFailedError, msg="Login should fail with an invalid username."):
            self.sdk_instance.login(INVALID_USERNAME, VALID_PASSWORD)
        self.assertFalse(self.sdk_instance._is_logged_in, "SDK internal state should remain False after failed login.")

    def test_login_with_empty_credentials(self):
        """
        Tests that login attempts with empty credentials raise LoginFailedError.
        """
        with self.assertRaises(LoginFailedError, msg="Login should fail with an empty username."):
            self.sdk_instance.login("", VALID_PASSWORD)

        with self.assertRaises(LoginFailedError, msg="Login should fail with an empty password."):
            self.sdk_instance.login(VALID_USERNAME, "")

        with self.assertRaises(LoginFailedError, msg="Login should fail with empty username and password."):
            self.sdk_instance.login("", "")

        self.assertFalse(self.sdk_instance._is_logged_in, "SDK state should remain logged out.")

    def test_attempt_login_when_already_logged_in(self):
        """
        Tests that attempting to login again when already logged in succeeds without error.
        """
        self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be in logged-in state after initial login.")

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Second login attempt failed unexpectedly: {e}")

        self.assertTrue(self.sdk_instance._is_logged_in, "SDK should remain in a logged-in state after second login.")

    def test_logout_when_not_logged_in(self):
        """
        Tests calling logout when not logged in. It should not raise an error.
        """
        try:
            self.sdk_instance.logout()
        except Exception as e:
            self.fail(f"sdk_instance.logout() raised an unexpected exception when called while not logged in: {e}")
        self.assertFalse(self.sdk_instance._is_logged_in, "SDK should remain in a not-logged-in state.")


if __name__ == '__main__':
    unittest.main()