# test_order_subscription_e2e.py
"""
End-to-end tests for Sphere SDK Order Subscription functionality.
"""
import unittest
import os
import threading
import uuid
import requests
import time

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    TradingClientError,
    NotLoggedInError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD
)

class TestOrderSubscriptionE2E(unittest.TestCase):
    sdk_instance = None
    base_url = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SDK and retrieves the base URL for testing.
        Skips all tests in this class if initialization fails.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
            cls.base_url = os.environ.get('SPHERE_BACKEND_BASE_URL')
            if not cls.base_url:
                raise unittest.SkipTest("SPHERE_BACKEND_BASE_URL environment variable is not set.")
        except SDKInitializationError as e:
            raise unittest.SkipTest(f"SDK Initialization failed, skipping E2E tests: {e}.")
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup: {e}")

    def setUp(self):
        """
        Logs in before each test and prepares resources for handling async callbacks.
        """
        if self.sdk_instance._is_logged_in:
            self.sdk_instance.logout()

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Login is a prerequisite for this test and it failed: {e}")
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be logged in for this test.")

        self.received_event = threading.Event()
        self.received_order_data = {}

    def tearDown(self):
        """
        Unsubscribes from events and logs out after each test to ensure isolation.
        """
        if self.sdk_instance:
            if self.sdk_instance._user_order_callback:
                try:
                    self.sdk_instance.unsubscribe_from_order_events()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during unsubscription in tearDown: {e}", file=sys.stderr)
            if self.sdk_instance._is_logged_in:
                try:
                    self.sdk_instance.logout()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during logout in tearDown: {e}", file=sys.stderr)

    @classmethod
    def tearDownClass(cls):
        """
        Final cleanup of the SDK instance.
        """
        if cls.sdk_instance and cls.sdk_instance._is_logged_in:
            cls.sdk_instance.logout()
        cls.sdk_instance = None

    def test_subscribe_insert_and_receiving_live_order(self):
        """
        Tests subscribing to orders, injecting a new order via a test endpoint,
        and verifying it is received by the subscription callback.
        """
        order_id = f"order_{uuid.uuid4().hex}"
        instance_id = f"orderinstance_{uuid.uuid4().hex}"

        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            for stack in order_data.body:
                for order in stack.orders:
                    if order.id == order_id:
                        self.received_order_data['full_event'] = order_data
                        self.received_order_data['order'] = order
                        self.received_order_data['stack'] = stack
                        self.received_event.set()
                        return

        self.sdk_instance.subscribe_to_order_events(on_order_event_received)

        test_order_payload = { "prices": [{ "id": f"{uuid.uuid4()}", "externalId": order_id, "externalInstanceId": instance_id, "expiryId": "expiry-jun25-Id", "expiryName": "Jun 25", "instrumentId": "instrument-id", "instrumentName": "testInstrument", "side": "a", "value": 99.5, "quantity": 10, "priceType": "flat", "interestType": "Live", "time": "2025-06-23T23:12:00+00:00", "expiries": [{ "id": "expiry-jun25-Id", "shortName": "Jun-25", "tradingEndDate": "2026-05-31T23:59:00+00:00", "deliveryEndDate": "2026-06-30T23:59:00+00:00"}], "traderId": "test-id", "traderCompanyId": "test-company-id", "brokerCompanyId": "broker-company-id", "brokerCodes": ["BC1"], "clearingCompanyCodes": ["ICE"], "receiverType": "Trader" }] }
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"
        
        try:
            response = requests.post(inject_url, json=test_order_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test order via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test order (ID: {order_id}) within {timeout_seconds} seconds.")

        received_event = self.received_order_data['full_event']
        received_order = self.received_order_data['order']
        received_stack = self.received_order_data['stack']
        
        self.assertEqual(
            received_event.event_type,
            sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED,
            "The event type for the received order data should be SNAPSHOT AMENDED."
        )
        self.assertEqual(received_order.id, order_id)
        self.assertEqual(received_order.instance_id, instance_id)
        self.assertEqual(received_order.price.quantity, "10")
        self.assertEqual(received_order.price.per_price_unit, "99.5")
        self.assertEqual(received_order.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)
        self.assertEqual(received_order.tradability, sphere_sdk_types_pb2.TRADABILITY_TRADABLE)
        self.assertEqual(received_stack.contract.side, sphere_sdk_types_pb2.ORDER_SIDE_ASK)

    def test_subscribe_insert_and_receive_spread_order_bid_side(self):
        """
        Tests subscribing to orders, injecting a new SPREAD order on the BID side,
        and verifying it is received with correct leg information.
        """
        order_id = f"order_{uuid.uuid4().hex}"
        instance_id = f"orderinstance_{uuid.uuid4().hex}"

        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            for stack in order_data.body:
                for order in stack.orders:
                    if order.id == order_id:
                        self.received_order_data['full_event'] = order_data
                        self.received_order_data['order'] = order
                        self.received_order_data['stack'] = stack
                        self.received_event.set()
                        return

        self.sdk_instance.subscribe_to_order_events(on_order_event_received)

        test_order_payload = { "prices": [{ "id": f"{uuid.uuid4()}", "externalId": order_id, "externalInstanceId": instance_id, "traderId": "test-id", "traderCompanyId": "test-company-id", "brokerCompanyId": "broker-company-id", "expiryId": "expiry-jun25-Id", "expiryName": "Jun 25", "instrumentId": "instrument-id", "instrumentName": "testInstrumentSpreadPy", "side": 'b', "value": 1.5, "quantity": 5, "priceType": "spread", "interestType": "Live", "time": "2025-06-23T23:12:00+00:00", "expiries": [ { "id": "expiry-jun25-Id", "shortName": "Jun-25", "tradingEndDate": "2026-05-31T23:59:00+00:00", "deliveryEndDate": "2026-06-30T23:59:00+00:00" }, { "id": "expiry-jul25-Id", "shortName": "Jul-25", "tradingEndDate": "2025-06-30T23:59:00+00:00", "deliveryEndDate": "2025-07-31T23:59:00+00:00" } ] }] }
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"

        try:
            response = requests.post(inject_url, json=test_order_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test spread order via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test spread order (ID: {order_id}) within {timeout_seconds} seconds.")

        received_event = self.received_order_data['full_event']
        received_order = self.received_order_data['order']
        contract = self.received_order_data['stack'].contract

        self.assertEqual(
            received_event.event_type,
            sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED,
            "The event type for the received order data should be SNAPSHOT AMENDED."
        )
        self.assertEqual(received_order.id, order_id)
        self.assertEqual(received_order.instance_id, instance_id)
        self.assertEqual(received_order.price.per_price_unit, "1.5")
        self.assertEqual(received_order.tradability, sphere_sdk_types_pb2.TRADABILITY_UNTRADABLE)
        self.assertEqual(contract.side, sphere_sdk_types_pb2.ORDER_SIDE_BID)
        self.assertEqual(len(contract.legs), 2, "Spread contract should have 2 legs.")
        self.assertEqual(contract.legs[0].expiry, "Jul-25")
        self.assertEqual(contract.legs[1].expiry, "Jun-25")
        self.assertEqual(contract.legs[0].spread_side, sphere_sdk_types_pb2.SPREAD_SIDE_TYPE_SELL)
        self.assertEqual(contract.legs[1].spread_side, sphere_sdk_types_pb2.SPREAD_SIDE_TYPE_BUY)
        
    def test_subscribe_insert_and_cancel_live_order(self): 
        """
        Tests subscribing to orders, injecting a new order via a test endpoint,
        cancelling that order and verifying by the subscription callback.
        """
        order_id = f"order_{uuid.uuid4().hex}"
        instance_id = f"orderinstance_{uuid.uuid4().hex}"
        db_id = f"{uuid.uuid4()}"

        self.messages_processed = 0

        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            if order_data.event_type != sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT:     
                
                self.messages_processed += 1

                for stack in order_data.body:                    
                    self.received_order_data['full_event'] = order_data  
                    self.received_order_data['stack'] = stack

                    for order in stack.orders:
                        if order.id == order_id:              
                            self.received_order_data['order'] = order
                 
                    self.received_event.set()   

        self.sdk_instance.subscribe_to_order_events(on_order_event_received)

        test_order_payload = { "prices": [{ "id": db_id, "externalId": order_id, "externalInstanceId": instance_id, "expiryId": "expiry-jun25-Id", "expiryName": "Jun 25", "instrumentId": "instrument-id", "instrumentName": "testInstrument", "side": "a", "value": 99.5, "quantity": 10, "traderId": "test-id", "traderCompanyId": "test-company-id", "brokerCompanyId": "broker-company-id", "priceType": "flat", "interestType": "Live", "time": "2025-06-23T23:12:00+00:00", "expiries": [{ "id": "expiry-jun25-Id", "shortName": "Jun-25", "tradingEndDate": "2026-05-31T23:59:00+00:00", "deliveryEndDate": "2026-06-30T23:59:00+00:00"}], "brokerCodes": ["BC1"], "clearingCompanyCodes": ["ICE"], "receiverType": "Trader" }] }
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"
        
        try:
            response = requests.post(inject_url, json=test_order_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test order via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test order (ID: {order_id}) within {timeout_seconds} seconds.")

        received_event = self.received_order_data['full_event']
        received_order = self.received_order_data['order']
        received_stack = self.received_order_data['stack']
        
        self.assertEqual(
            received_event.event_type,
            sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED,
            "The event type for the received order data should be SNAPSHOT AMENEDED."
        )

        self.assertEqual(received_order.id, order_id)
        self.assertEqual(received_order.instance_id, instance_id)
        self.assertEqual(received_order.price.quantity, "10")
        self.assertEqual(received_order.price.per_price_unit, "99.5")
        self.assertEqual(received_order.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)
        self.assertEqual(received_order.tradability, sphere_sdk_types_pb2.TRADABILITY_TRADABLE)
        self.assertEqual(received_stack.contract.side, sphere_sdk_types_pb2.ORDER_SIDE_ASK)
        
        self.received_event.clear()

        test_cancel_order_payload = { "prices": [{ "id": db_id, "externalId": order_id }] }
        inject_cancel_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=CANCELLED"
        
        try:
            cancelResponse = requests.post(inject_cancel_url, json=test_cancel_order_payload, timeout=5)
            cancelResponse.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test cancel order via HTTP endpoint '{inject_cancel_url}': {e}")

        expected_messages = 2 
        start_time = time.time()

        while self.messages_processed < expected_messages and time.time() - start_time < timeout_seconds:
            self.received_event.wait(timeout=timeout_seconds)
            self.received_event.clear()

        if self.messages_processed < expected_messages:
            self.fail(f"Timeout: Expected at least {expected_messages} messages, but only processed {self.messages_processed}.")

        received_event = self.received_order_data['full_event']
        received_stack = self.received_order_data['stack']

        self.assertEqual(
            received_event.event_type,
            sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED,
            "The event type for the received order data should be SNAPSHOT_AMENDED."
        )

        self.assertEqual(len(received_stack.orders), 0, "The stack should be empty")
        self.assertEqual(received_stack.contract.side, sphere_sdk_types_pb2.ORDER_SIDE_ASK)

    def test_subscribe_insert_and_update_live_order(self): 
        """
        Tests subscribing to orders, injecting a new order via a test endpoint,
        updating that order and verifying by the subscription callback.
        """
        order_id = f"order_{uuid.uuid4().hex}"
        initial_instance_id = f"orderinstance_{uuid.uuid4().hex}"
        updated_instance_id = f"orderinstance_{uuid.uuid4().hex}"
        add_db_id = "0101"
        initial_quantity = "5"
        updated_quantity = "10"

        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            self.received_event_type = order_data.event_type
            for stack in order_data.body:
                for order in stack.orders:             
                    self.received_order_data['order'] = order
                    self.received_event.set()
                    return  

        self.sdk_instance.subscribe_to_order_events(on_order_event_received)

        add_order_payload = { "prices": [{ "id": add_db_id, "externalId": order_id, "externalInstanceId": initial_instance_id, "expiryId": "expiry-jun25-Id",
                                          "expiryName": "Jun 25", "instrumentId": "instrument-id", "instrumentName": "testInstrument",
                                          "side": "a", "value": 99.5, "quantity": 5, "priceType": "flat", "interestType": "Live",
                                          "time": "2025-06-23T23:12:00+00:00", "traderId": "test-id", "traderCompanyId": "test-company-id", "brokerCompanyId": "broker-company-id",
                                          "expiries": [{ "id": "expiry-jun25-Id", "shortName": "Jun-25",
                                                        "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                                        "deliveryEndDate": "2026-06-30T23:59:00+00:00"}],
                                          "brokerCodes": ["BC1"], "clearingCompanyCodes": ["ICE"], "receiverType": "Trader" }] }

        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"
        
        try:
            response = requests.post(inject_url, json=add_order_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test order via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test order (ID: {order_id}) within {timeout_seconds} seconds.")

        initial_order = self.received_order_data['order']
        initial_event_type = self.received_event_type
        
        self.assertEqual(
            initial_event_type,
            sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED,
            "The event type for the received order data should be SNAPSHOT AMENEDED."
        )

        self.assertEqual(initial_order.id, order_id)
        self.assertEqual(initial_order.instance_id, initial_instance_id)
        self.assertEqual(initial_order.price.quantity, "5")
        self.assertEqual(initial_order.price.per_price_unit, "99.5")
        self.assertEqual(initial_order.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)
        self.assertEqual(initial_order.tradability, sphere_sdk_types_pb2.TRADABILITY_TRADABLE)
        
        self.received_event.clear()

        update_db_id = "1010"
        test_update_order_payload = { "prices": [{ "id": update_db_id, "externalId": order_id, "externalInstanceId": updated_instance_id, "replacedPriceId": add_db_id, "expiryId": "expiry-jun25-Id",
                                                  "expiryName": "Jun 25", "instrumentId": "instrument-id", "instrumentName": "testInstrument",
                                                  "side": "a", "value": 99.5, "quantity": 10, "priceType": "flat", "interestType": "Live",
                                                  "time": "2025-06-23T23:12:00+00:00", "traderId": "test-id", "traderCompanyId": "test-company-id", "brokerCompanyId": "broker-company-id",
                                                  "expiries": [{ "id": "expiry-jun25-Id", "shortName": "Jun-25",
                                                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"}],
                                                  "brokerCodes": ["BC1"], "clearingCompanyCodes": ["ICE"], "receiverType": "Trader" }] }
        

        inject_update_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=UPDATED"
        
        try:
            updatedResponse = requests.post(inject_update_url, json=test_update_order_payload, timeout=5)
            updatedResponse.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test update order via HTTP endpoint '{inject_update_url}': {e}")

        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the updated order (ID: {update_db_id}) within {timeout_seconds} seconds.")

        updated_event_type = self.received_event_type
        self.assertEqual(updated_event_type, sphere_sdk_types_pb2.ORDER_STACKS_EVENT_TYPE_SNAPSHOT_AMENDED)

        updated_order = self.received_order_data['order']
        self.assertEqual(updated_order.id, order_id)
        self.assertEqual(updated_order.instance_id, updated_instance_id)
        self.assertEqual(updated_order.price.quantity, updated_quantity, "Order quantity was not updated correctly.")


if __name__ == '__main__':
    unittest.main()