import unittest
import os
import sys
import uuid

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    NotLoggedInError,
    TradingClientError,
    GetInstrumentsFailedError,
    GetExpiriesFailedError,
    GetBrokersFailedError,
    GetClearingOptionsFailedError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD
)

class TestStaticDataE2E(unittest.TestCase):
    """
    End-to-end tests for retrieving static data like instruments, expiries, brokers and clearing options.
    These tests interact with a live (or test) backend but do not require
    data injection or endpoint stubbing.
    """
    sdk_instance = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SDK once for all tests in this class.
        Skips all tests if initialization fails.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
        except SDKInitializationError as e:
            raise unittest.SkipTest(f"SDK Initialization failed, skipping E2E tests: {e}")
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup: {e}")

    def setUp(self):
        """
        Logs in before each test to ensure a clean, authenticated state.
        """
        if self.sdk_instance and self.sdk_instance._is_logged_in:
            self.sdk_instance.logout()

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Login is a prerequisite for this test and it failed: {e}")
        
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be logged in for this test.")

    def tearDown(self):
        """
        Logs out after each test to ensure isolation.
        """
        if self.sdk_instance and self.sdk_instance._is_logged_in:
            try:
                self.sdk_instance.logout()
            except (TradingClientError, NotLoggedInError) as e:
                # Suppress errors during teardown as they are not critical to test results.
                print(f"WARNING: Harmless error during logout in tearDown: {e}", file=sys.stderr)

    def test_get_instruments_succeeds_and_returns_data(self):
        """
        Tests that get_instruments() returns a list of instruments.
        """
        instruments_container = self.sdk_instance.get_instruments()

        instruments_list = list(instruments_container)

        self.assertIsInstance(
            instruments_list,
            list,
            "The result of get_instruments should be convertible to a list."
        )

        self.assertTrue(
            instruments_list,
            "Expected to receive at least one instrument, but the list was empty."
        )
    
        first_instrument = instruments_list[0]
        self.assertIsInstance(
            first_instrument,
            sphere_sdk_types_pb2.InstrumentDto,
            "The items in the list should be of type InstrumentDto."
        )
    
        self.assertTrue(first_instrument.name, "Instrument should have a non-empty name.")
        print(f"\n[INFO] Successfully retrieved {len(instruments_list)} instruments. First instrument: '{first_instrument.name}'")

    def test_get_expiries_for_valid_instrument_succeeds(self):
        """
        Tests that get_expiries_by_instrument_name() succeeds for a valid instrument name.
        """
        instruments_container = self.sdk_instance.get_instruments()
        instruments_list = list(instruments_container)
    
        self.assertGreater(
            len(instruments_list), 0,
            "Prerequisite failed: No instruments found to test expiries with."
        )
        valid_instrument_name = instruments_list[0].name

        expiries_container = self.sdk_instance.get_expiries_by_instrument_name(valid_instrument_name)
        expiries_list = list(expiries_container)

        self.assertIsInstance(
            expiries_list,
            list,
            "The result of get_expiries_by_instrument_name should be convertible to a list."
        )
    
        self.assertTrue(
            expiries_list,
            f"Expected to receive at least one expiry for instrument '{valid_instrument_name}', but the list was empty."
        )
    
        first_expiry = expiries_list[0]
        self.assertIsInstance(
            first_expiry,
            sphere_sdk_types_pb2.ExpiryDto,
            "The items in the list should be of type ExpiryDto."
        )
        self.assertTrue(first_expiry.name, "Expiry should have a non-empty name.")
        print(f"\n[INFO] Successfully retrieved {len(expiries_list)} expiries for '{valid_instrument_name}'. First expiry: '{first_expiry.name}'")

    def test_get_brokers_succeeds_and_returns_data(self):
        """
        Tests that get_brokers() returns a list of brokers.
        """
        brokers_container = self.sdk_instance.get_brokers()
        brokers_list = list(brokers_container)

        self.assertIsInstance(
            brokers_list,
            list,
            "The result of get_brokers should be convertible to a list."
        )
        
        self.assertTrue(
            brokers_list,
            "Expected to receive at least one broker, but the list was empty."
        )

        first_broker = brokers_list[0]
        self.assertIsInstance(
            first_broker,
            sphere_sdk_types_pb2.BrokerDto,
            "The items in the list should be of type BrokerDto."
        )

        self.assertTrue(first_broker.name, "Broker should have a non-empty name.")
        print(f"\n[INFO] Successfully retrieved {len(brokers_list)} brokers. First broker: '{first_broker.name}'")

    def test_get_clearing_options_succeeds_and_returns_data(self):
        """
        Tests that get_clearing_options() returns a list of clearing options.
        """
        clearing_options_container = self.sdk_instance.get_clearing_options()
        clearing_options_list = list(clearing_options_container)
    
        self.assertIsInstance(
            clearing_options_list,
            list,
            "The result of get_clearing_options should be convertible to a list."
        )

        self.assertTrue(
            clearing_options_list,
            "Expected to receive at least one clearing option, but the list was empty."
        )
    
        first_clearing_option = clearing_options_list[0]
        self.assertIsInstance(
            first_clearing_option,
            sphere_sdk_types_pb2.ClearingOptionDto,
            "The items in the list should be of type ClearingOptionDto."
        )
    
        self.assertTrue(first_clearing_option.code, "Clearing Option should have a non-empty code.")
        print(f"\n[INFO] Successfully retrieved {len(clearing_options_list)} Clearing Options. First clearing option: '{first_clearing_option.code}'")
    
    def test_get_expiries_for_invalid_instrument_returns_error(self):
        """
        Tests that get_expiries_by_instrument_name() raises an exception
        for an instrument name that does not exist.
        """
        invalid_instrument_name = f"non_existent_instrument_{uuid.uuid4().hex}"

        with self.assertRaises(GetExpiriesFailedError) as cm:
            self.sdk_instance.get_expiries_by_instrument_name(invalid_instrument_name)
    
        self.assertTrue(
            str(cm.exception).startswith("Get Expiries by Instrument Name failed:")
        )
        print(f"\n[INFO] Correctly received expected exception: {cm.exception}")


if __name__ == '__main__':
    unittest.main()