import unittest
import os
import threading
import uuid
import requests
import json
import sys

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    TradingClientError,
    TradeOrderFailedError,
    NotLoggedInError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD
)

class TestTradeOrderE2E(unittest.TestCase):
    sdk_instance = None
    base_url = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SDK and retrieves the base URL for testing.
        Skips all tests in this class if initialization fails.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
            cls.base_url = os.environ.get('SPHERE_BACKEND_BASE_URL')
            if not cls.base_url:
                raise unittest.SkipTest("SPHERE_BACKEND_BASE_URL environment variable is not set.")
        except SDKInitializationError as e:
            raise unittest.SkipTest(f"SDK Initialization failed, skipping E2E tests: {e}.")
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup: {e}")

    def setUp(self):
        """
        Logs in before each test and prepares resources for handling async callbacks.
        """
        if self.sdk_instance and self.sdk_instance._is_logged_in:
            self.sdk_instance.logout()

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Login is a prerequisite for this test and it failed: {e}")
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be logged in for this test.")

        self.received_event = threading.Event()
        self.received_order_data = {}

    def tearDown(self):
        """
        Unsubscribes from events and logs out after each test to ensure isolation.
        """
        if self.sdk_instance:
            if self.sdk_instance._user_order_callback:
                try:
                    self.sdk_instance.unsubscribe_from_order_events()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during unsubscription in tearDown: {e}", file=sys.stderr)
            if self.sdk_instance._is_logged_in:
                try:
                    self.sdk_instance.logout()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during logout in tearDown: {e}", file=sys.stderr)

    def test_trade_full_quantity_succeeds(self):
        """
        Tests that trading the full quantity of an order succeeds and sends the correct
        payload to the backend.
        """
        # 1. Define unique IDs for the test entities to ensure isolation.
        order_id = f"order_{uuid.uuid4().hex}"
        price_id = f"price_{uuid.uuid4().hex}"

        # 2. Define the price/order to be injected into the system.
        # This is equivalent to pre-loading the order into the client's cache.
        test_price_payload = {
            "prices": [{
                "id": price_id,
                "externalId": order_id,
                "expiryName": "Jun 25",
                "instrumentId": "instrument-id",
                "instrumentName": "testInstrumentPy",
                "side": "b",
                "value": 99.0,
                "quantity": 50,
                "priceType": "flat",
                "interestType": "Live",
                "time": "2025-06-23T23:12:00+00:00",
                "expiries": [{
                    "id": "expiry-jun25-Id",
                    "shortName": "Jun-25",
                    "tradingEndDate": "2025-05-31T23:59:00+00:00",
                    "deliveryEndDate": "2025-06-30T23:59:00+00:00"
                }],
                "receiverType": "trader",
                "brokerCodes": ["BC1"],
                "clearingCompanyCodes": ["ICE"]
            }]
        }

        # 3. Stub the backend's trade execution API endpoint to return a successful response.
        stubbed_trade_order_response = {
            "id": f"trade_{uuid.uuid4().hex}",
            "createdAt": "2025-08-01T11:17:30.1667766+00:00",
            "tradeType": "sphereLive",
            "priceId": price_id,
            "quantity": 50
        }
        stub_trade_url = f"{self.base_url}/_testing/price/trade?statusCode=201"
        try:
            response = requests.post(stub_trade_url, json=stubbed_trade_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub trade endpoint '{stub_trade_url}': {e}")
            
        # 4. Set up a subscription to wait for the injected price to arrive in the client.
        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            for stack in order_data.body:
                for order in stack.orders:
                    if order.id == order_id:
                        self.received_order_data['order'] = order
                        self.received_event.set()
                        return
        
        self.sdk_instance.subscribe_to_order_events(on_order_event_received)
        
        # 5. Inject the price using the test SignalR endpoint.
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"
        try:
            response = requests.post(inject_url, json=test_price_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test order via HTTP endpoint '{inject_url}': {e}")
            
        # 6. Wait until the SDK has processed the new order.
        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive test order (ID: {order_id}) within {timeout_seconds}s.")

        # 7. Execute the trade for the full quantity (by not specifying a quantity).
        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"
        trade_request = sphere_sdk_types_pb2.TradeOrderRequestDto(order_instance_id=self.received_order_data['order'].instance_id, idempotency_key=idempotency_key)
        trade_response = self.sdk_instance.trade_order(trade_request)
        
        # 8. Assert that the trade call was successful.
        self.assertEqual(trade_response.code, sphere_sdk_types_pb2.ErrorCode.ERROR_CODE_NONE)

        # 9. Retrieve the captured request from the test endpoint and verify its content.
        capture_url = f"{self.base_url}/_testing/price/trade"
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured trade request from '{capture_url}': {e}")
            
        expected_body = {
            "priceId": price_id,
            "requestBody": {
                "quantity": 50,
                "dealtByMe": True
            },
            "idempotencyKey": idempotency_key
        }
        self.assertEqual(captured_body_json, expected_body)

    def test_trade_partial_quantity_succeeds(self):
        """
        Tests that trading a partial quantity of an order succeeds and sends the correct
        payload to the backend. 
        """
        # 1. Define unique IDs and quantities for the test.
        order_id = f"order_{uuid.uuid4().hex}"
        price_id = f"price_{uuid.uuid4().hex}"
        initial_quantity = 50
        partial_trade_quantity = 20

        # 2. Define the price/order to be injected into the system.
        test_price_payload = {
            "prices": [{
                "id": price_id,
                "externalId": order_id,
                "expiryName": "Jun 25",
                "instrumentId": "instrument-id",
                "instrumentName": "testInstrumentPy",
                "side": "b",
                "value": 99.0,
                "quantity": initial_quantity,
                "priceType": "flat",
                "interestType": "Live",
                "time": "2025-06-23T23:12:00+00:00",
                "expiries": [{
                    "id": "expiry-jun25-Id",
                    "shortName": "Jun-25",
                    "tradingEndDate": "2025-05-31T23:59:00+00:00",
                    "deliveryEndDate": "2025-06-30T23:59:00+00:00"
                }],
                "receiverType": "trader",
                "brokerCodes": ["BC1"],
                "clearingCompanyCodes": ["ICE"]
            }]
        }

        # 3. Stub the backend's trade execution API endpoint for the partial trade.
        stubbed_trade_order_response = {
            "id": f"trade_{uuid.uuid4().hex}",
            "createdAt": "2025-08-01T11:17:30.1667766+00:00",
            "tradeType": "sphereLive",
            "priceId": price_id,
            "quantity": partial_trade_quantity
        }
        stub_trade_url = f"{self.base_url}/_testing/price/trade?statusCode=201"
        try:
            response = requests.post(stub_trade_url, json=stubbed_trade_order_response, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to stub trade endpoint '{stub_trade_url}': {e}")

        # 4. Set up a subscription to wait for the injected price to arrive.
        def on_order_event_received(order_data: sphere_sdk_types_pb2.OrderStacksDto):
            for stack in order_data.body:
                for order in stack.orders:
                    if order.id == order_id:
                        self.received_order_data['order'] = order
                        self.received_event.set()
                        return

        self.sdk_instance.subscribe_to_order_events(on_order_event_received)

        # 5. Inject the price using the test SignalR endpoint.
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=LIVE_PRICE&arg2=ADDED"
        try:
            response = requests.post(inject_url, json=test_price_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test order via HTTP endpoint '{inject_url}': {e}")

        # 6. Wait until the SDK has processed the new order.
        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive test order (ID: {order_id}) within {timeout_seconds}s.")

        # 7. Execute the trade for the partial quantity.
        idempotency_key = f"idempotency_key_{uuid.uuid4().hex}"
        trade_request = sphere_sdk_types_pb2.TradeOrderRequestDto(
            order_instance_id=self.received_order_data['order'].instance_id,
            quantity=str(partial_trade_quantity),
            idempotency_key=idempotency_key
        )
        trade_response = self.sdk_instance.trade_order(trade_request)

        # 8. Assert that the trade call was successful.
        self.assertEqual(trade_response.code, sphere_sdk_types_pb2.ErrorCode.ERROR_CODE_NONE)

        # 9. Retrieve the captured request from the test endpoint and verify its content.
        capture_url = f"{self.base_url}/_testing/price/trade"
        try:
            captured_response = requests.get(capture_url, timeout=5)
            captured_response.raise_for_status()
            captured_body_json = captured_response.json()
        except (requests.exceptions.RequestException, json.JSONDecodeError) as e:
            self.fail(f"Failed to retrieve or parse captured trade request from '{capture_url}': {e}")

        expected_body = {
            "priceId": price_id,
            "requestBody": {
                "quantity": partial_trade_quantity,
                "dealtByMe": True
            },
            "idempotencyKey": idempotency_key
        }
        self.assertEqual(captured_body_json, expected_body)

    def test_trade_full_quantity_fails(self):
        """
        Tests that attempting to trade a non-existent order raises a TradeOrderFailedError.
        """
        # 1. Define an ID for an order that has not been injected into the client.
        order_id = f"orderinstance_{uuid.uuid4().hex}"
        trade_request = sphere_sdk_types_pb2.TradeOrderRequestDto(order_instance_id=order_id)
        
        # 2. Assert that calling trade_order with this non-existent ID raises the expected exception.
        with self.assertRaises(TradeOrderFailedError) as cm:
            self.sdk_instance.trade_order(trade_request)
        
        # 3. Assert that the exception message is correct, confirming why the trade failed.
        self.assertEqual(
            "Trade order failed: Order not found.",
            str(cm.exception)
        )

if __name__ == '__main__':
    # This allows the test to be run directly from the command line.
    unittest.main()