"""
End-to-end tests for Sphere SDK Trade Subscription functionality.
"""
import unittest
import os
import threading
import uuid
import requests
import time

from test_common import (
    SphereTradingClientSDK,
    SDKInitializationError,
    LoginFailedError,
    TradingClientError,
    NotLoggedInError,
    sphere_sdk_types_pb2,
    VALID_USERNAME,
    VALID_PASSWORD
)

class TestTradeSubscriptionE2E(unittest.TestCase):
    sdk_instance = None
    base_url = None

    @classmethod
    def setUpClass(cls):
        """
        Initializes the SDK and retrieves the base URL for testing.
        Skips all tests in this class if initialization fails.
        """
        try:
            cls.sdk_instance = SphereTradingClientSDK()
            cls.base_url = os.environ.get('SPHERE_BACKEND_BASE_URL')
            if not cls.base_url:
                raise unittest.SkipTest("SPHERE_BACKEND_BASE_URL environment variable is not set.")
        except SDKInitializationError as e:
            raise unittest.SkipTest(f"SDK Initialization failed, skipping E2E tests: {e}.")
        except Exception as e:
            raise unittest.SkipTest(f"An unexpected error occurred during SDK setup: {e}")

    def setUp(self):
        """
        Logs in before each test and prepares resources for handling async callbacks.
        """
        if self.sdk_instance._is_logged_in:
            self.sdk_instance.logout()

        try:
            self.sdk_instance.login(VALID_USERNAME, VALID_PASSWORD)
        except LoginFailedError as e:
            self.fail(f"Login is a prerequisite for this test and it failed: {e}")
        self.assertTrue(self.sdk_instance._is_logged_in, "SDK must be logged in for this test.")

        self.received_event = threading.Event()
        self.received_trade_data = {}
        self.received_event_type = None

    def tearDown(self):
        """
        Unsubscribes from events and logs out after each test to ensure isolation.
        """
        if self.sdk_instance:
            
            self.received_event = threading.Event()
            self.received_trade_data = {}
            self.received_event_type = None

            if self.sdk_instance._user_trade_callback:
                try:
                    self.sdk_instance.unsubscribe_from_trade_events()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during unsubscription in tearDown: {e}", file=sys.stderr)
            if self.sdk_instance._is_logged_in:
                try:
                    self.sdk_instance.logout()
                except (TradingClientError, NotLoggedInError) as e:
                    print(f"WARNING: Harmless error during logout in tearDown: {e}", file=sys.stderr)

    @classmethod
    def tearDownClass(cls):
        """
        Final cleanup of the SDK instance.
        """
        if cls.sdk_instance and cls.sdk_instance._is_logged_in:
            cls.sdk_instance.logout()
        cls.sdk_instance = None

    def test_subscribe_insert_and_receiving_live_trade(self):
        """
        Tests subscribing to trades, injecting a new trade via a test endpoint,
        and verifying it is received by the subscription callback.
        """
        trade_id = f"trade_{uuid.uuid4().hex}"

        def on_trade_event_received(trade_data: sphere_sdk_types_pb2.TradeMessageDto):
            self.received_event_type = trade_data.event_type
            for trade in trade_data.body:
                if trade.id == trade_id:
                    self.received_trade_data['trade'] = trade
                    self.received_event.set()
                    return

        self.sdk_instance.subscribe_to_trade_events(on_trade_event_received)

        test_trade_payload = {
            "trades": [
                {
                    "id": "111",
                    "externalId": trade_id,
                    "dealtPriceId": "111",
                    "trade": {
                        "id": "111",
                        "externalId": trade_id,
                        "expiryId": "expiry-jun25-Id",
                        "instrumentId": "instrument-id",
                        "instrumentName": "testInstrumentSpreadPy",
                        "side": 'b',
                        "value": 1.5,
                        "quantity": 5,
                        "priceType": "flat",
                        "tradeType": "sphereLive",
                        "time": "2025-06-23T23:12:00+00:00",
                        "expiries": [
                            {
                                "id": "expiry-jun25-Id",
                                "shortName": "Jun-25",
                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"
                            }
                        ]
                    }
                }
            ]
        }
                
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=ADDED"
        
        try:
            response = requests.post(inject_url, json=test_trade_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test trade via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test trade (ID: {trade_id}) within {timeout_seconds} seconds.")

        self.assertEqual(self.received_event_type, sphere_sdk_types_pb2.TRADE_EVENT_TYPE_EXECUTED)
        received_trade = self.received_trade_data['trade']
        
        self.assertEqual(received_trade.id, trade_id)
        self.assertEqual(received_trade.price.quantity, "5")
        self.assertEqual(received_trade.price.per_price_unit, "1.5")
        self.assertEqual(received_trade.contract.expiry, "Jun-25")
        self.assertEqual(received_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)
        
    def test_subscribe_insert_and_receiving_indicative_trade(self):
        """
        Tests subscribing to trades, injecting a new IM trade via a test endpoint,
        and verifying it is received by the subscription callback.
        """
        trade_id = f"trade_{uuid.uuid4().hex}"

        def on_trade_event_received(trade_data: sphere_sdk_types_pb2.TradeMessageDto):
            self.received_event_type = trade_data.event_type
            for trade in trade_data.body:
                if trade.id == trade_id:
                    self.received_trade_data['trade'] = trade
                    self.received_event.set()
                    return

        self.sdk_instance.subscribe_to_trade_events(on_trade_event_received)

        test_trade_payload = {
            "trades": [
                {
                    "id": "111",
                    "externalId": trade_id,
                    "dealtPriceId": "111",
                    "trade": {
                        "id": "111",
                        "externalId": trade_id,
                        "expiryId": "expiry-jun25-Id",
                        "instrumentId": "instrument-id",
                        "instrumentName": "testInstrumentSpreadPy",
                        "side": 'b',
                        "value": 1.5,
                        "quantity": 5,
                        "priceType": "flat",
                        "tradeType": "IM",
                        "time": "2025-06-23T23:12:00+00:00",
                        "expiries": [
                            {
                                "id": "expiry-jun25-Id",
                                "shortName": "Jun-25",
                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"
                            }
                        ]
                    }
                }
            ]
        }
                
        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=IM_TRADE&arg2=ADDED"
        
        try:
            response = requests.post(inject_url, json=test_trade_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test trade via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test trade (ID: {trade_id}) within {timeout_seconds} seconds.")

        self.assertEqual(self.received_event_type, sphere_sdk_types_pb2.TRADE_EVENT_TYPE_EXECUTED)
        received_trade = self.received_trade_data['trade']
        
        self.assertEqual(received_trade.id, trade_id)
        self.assertEqual(received_trade.price.quantity, "5")
        self.assertEqual(received_trade.price.per_price_unit, "1.5")
        self.assertEqual(received_trade.contract.expiry, "Jun-25")
        self.assertEqual(received_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_INDICATIVE)

    def test_subscribe_insert_and_receive_spread_trade_bid_side(self):
        """
        Tests subscribing to trades, injecting a new SPREAD trade on the BID side,
        and verifying it is received with correct leg information.
        """
        trade_id = f"trade_{uuid.uuid4().hex}"

        def on_trade_event_received(trade_data: sphere_sdk_types_pb2.TradeMessageDto):
            self.received_event_type = trade_data.event_type
            for trade in trade_data.body:
                self.received_trade_data['trade'] = trade
                self.received_event.set()
                return

        self.sdk_instance.subscribe_to_trade_events(on_trade_event_received)

        test_trade_payload = {
            "trades": [
                {
                    "id": "222",
                    "externalId": trade_id,
                    "dealtPriceId": "111",
                    "trade": {
                        "id": "222",
                        "externalId": trade_id,
                        "expiryId": "expiry-jun25-Id",
                        "expiryName": "Jun 25",
                        "instrumentId": "instrument-id",
                        "instrumentName": "testInstrumentSpreadPy",
                        "side": 'b',
                        "value": 1.5,
                        "quantity": 5,
                        "priceType": "spread",
                        "tradeType": "sphereLive",
                        "time": "2025-06-23T23:12:00+00:00",
                        "expiries": [
                            {
                                "id": "expiry-jun25-Id",
                                "shortName": "Jun-25",
                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"
                            },
                            {
                                "id": "expiry-jul25-Id",
                                "shortName": "Jul-25",
                                "tradingEndDate": "2025-06-30T23:59:00+00:00",
                                "deliveryEndDate": "2025-07-31T23:59:00+00:00"
                            }
                        ]
                    }
                }
            ]
        }

        inject_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=ADDED"

        try:
            response = requests.post(inject_url, json=test_trade_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject test spread trade via HTTP endpoint '{inject_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the test spread trade (ID: {trade_id}) within {timeout_seconds} seconds.")

        self.assertEqual(self.received_event_type, sphere_sdk_types_pb2.TRADE_EVENT_TYPE_EXECUTED)
        received_trade = self.received_trade_data['trade']

        self.assertEqual(received_trade.id, trade_id)
        self.assertEqual(received_trade.price.per_price_unit, "1.5")
        self.assertEqual(len(received_trade.contract.legs), 2, "Spread contract should have 2 legs.")
        self.assertEqual(received_trade.contract.legs[0].expiry, "Jul-25")
        self.assertEqual(received_trade.contract.legs[1].expiry, "Jun-25")
        self.assertEqual(received_trade.contract.legs[0].spread_side, sphere_sdk_types_pb2.SPREAD_SIDE_TYPE_SELL)
        self.assertEqual(received_trade.contract.legs[1].spread_side, sphere_sdk_types_pb2.SPREAD_SIDE_TYPE_BUY)
        self.assertEqual(received_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)

    def test_subscribe_insert_then_update_trade(self):
        """
        Tests subscribing to trades, injecting a new trade,
        verifying it is received, then injecting an update to that trade
        with a new quantity and timestamp, and verifying the update.
        """
        
        def on_trade_event_received(trade_data: sphere_sdk_types_pb2.TradeMessageDto):
            self.received_event_type = trade_data.event_type
            for trade in trade_data.body:
                self.received_trade_data['trade'] = trade
                self.received_event.set()
                return

        self.sdk_instance.subscribe_to_trade_events(on_trade_event_received)

        external_id = f"trade_{uuid.uuid4().hex}"
        addedTrade_id = "333"
        initial_quantity = "5"
        updated_quantity = "10"
        
        trade_time = "2025-06-23T23:12:00.0000000+00:00"

        test_trade_payload = {
            "trades": [
                {
                    "id": addedTrade_id,
                    "externalId": external_id,
                    "dealtPriceId": "111",
                    "trade": {
                        "id": addedTrade_id,
                        "externalId": external_id,
                        "expiryId": "expiry-jun25-Id",
                        "expiryName": "Jun 25",
                        "instrumentId": "instrument-id",
                        "instrumentName": "testInstrumentSpreadPy",
                        "side": 'b',
                        "value": 1.5,
                        "quantity": 5,
                        "priceType": "spread",
                        "tradeType": "sphereLive",
                        "time": trade_time,
                        "expiries": [
                            {
                                "id": "expiry-jun25-Id",
                                "shortName": "Jun-25",
                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"
                            },
                            {
                                "id": "expiry-jul25-Id",
                                "shortName": "Jul-25",
                                "tradingEndDate": "2025-06-30T23:59:00+00:00",
                                "deliveryEndDate": "2025-07-31T23:59:00+00:00"
                            }
                        ]
                    }
                }
            ]
        }

        inject_add_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=ADDED"

        try:
            response = requests.post(inject_add_url, json=test_trade_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject initial test trade via HTTP endpoint '{inject_add_url}': {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the initial trade (ID: {addedTrade_id}) within {timeout_seconds} seconds.")

        self.assertEqual(self.received_event_type, sphere_sdk_types_pb2.TRADE_EVENT_TYPE_EXECUTED)
        initial_trade = self.received_trade_data['trade']
        self.assertEqual(initial_trade.id, external_id)
        self.assertEqual(initial_trade.price.quantity, initial_quantity)
        self.assertEqual(initial_trade.price.per_price_unit, "1.5")
        self.assertEqual(initial_trade.created_time, trade_time, "Initial created_time does not match.")
        self.assertEqual(len(initial_trade.contract.legs), 2, "Spread contract should have 2 legs.")
        self.assertEqual(initial_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)

        self.received_event.clear()

        updatedTrade_id = "444"

        updated_trade_payload = {
            "trades": [
                {
                    "id": updatedTrade_id,
                    "externalId": external_id,
                    "dealtPriceId": "111",
                    "trade": {
                        "id": updatedTrade_id,
                        "externalId": external_id,
                        "replacedTradeId": addedTrade_id,
                        "expiryId": "expiry-jun25-Id",
                        "expiryName": "Jun 25",
                        "instrumentId": "instrument-id",
                        "instrumentName": "testInstrumentSpreadPy",
                        "side": 'b',
                        "value": 1.5,
                        "quantity": 10,
                        "priceType": "spread",
                        "tradeType": "sphereLive",
                        "time": trade_time,
                        "expiries": [
                            {
                                "id": "expiry-jun25-Id",
                                "shortName": "Jun-25",
                                "tradingEndDate": "2026-05-31T23:59:00+00:00",
                                "deliveryEndDate": "2026-06-30T23:59:00+00:00"
                            },
                            {
                                "id": "expiry-jul25-Id",
                                "shortName": "Jul-25",
                                "tradingEndDate": "2025-06-30T23:59:00+00:00",
                                "deliveryEndDate": "2025-07-31T23:59:00+00:00"
                            }
                        ]
                    }
                }
            ]
        }

        inject_update_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=UPDATED"

        try:
            response = requests.post(inject_update_url, json=updated_trade_payload, timeout=5)
            response.raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject updated test trade via HTTP endpoint '{inject_update_url}': {e}")

        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the updated trade (ID: {updatedTrade_id}) within {timeout_seconds} seconds.")
        
        self.assertEqual(self.received_event_type, sphere_sdk_types_pb2.TRADE_EVENT_TYPE_AMENDED)
        updated_trade = self.received_trade_data['trade']

        self.assertEqual(updated_trade.id, external_id)
        self.assertEqual(updated_trade.price.quantity, updated_quantity, "Trade quantity was not updated correctly.")
        self.assertEqual(updated_trade.created_time, trade_time, "Trade created_time was not set correctly.")
        self.assertEqual(updated_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)

    def test_subscribe_insert_then_cancel_trade(self):
        """
        Tests subscribing to trades, injecting two new trades, then injecting a cancel message,
        and finally verifying that only one trade remains.
        """
    
        trade_to_keep_external_id = f"trade_{uuid.uuid4().hex}"
        trade_to_keep_id = "555"
        trade_to_cancel_external_id = f"trade_{uuid.uuid4().hex}"
        trade_to_cancel_id = "666"

        self.messages_processed = 0
        self.received_event_types = []

        def on_trade_event_received(trade_data: sphere_sdk_types_pb2.TradeMessageDto):
            if (len(trade_data.body) > 0):
                self.messages_processed += 1
                self.received_event_types.append(trade_data.event_type)        
                new_trades_snapshot = {trade.id: trade for trade in trade_data.body}        
                self.received_trade_data = new_trades_snapshot
                self.received_event.set()

        self.sdk_instance.subscribe_to_trade_events(on_trade_event_received)

        add_trades_payload = { "trades": [
                {
                    "id": trade_to_keep_id, "externalId": trade_to_keep_external_id, "dealtPriceId": "111",
                    "trade": { "id": trade_to_keep_id, "externalId": trade_to_keep_external_id, "expiryId": "expiry-jun25-Id", "instrumentId": "instrument-id", "instrumentName": "testInstrumentSpreadPy", "side": 'b', "value": 1.5, "quantity": 5, "priceType": "spread", "tradeType": "sphereLive", "time": "2025-06-23T23:12:00+00:00",
                        "expiries": [ {"id": "expiry-jun25-Id", "shortName": "Jun-25", "tradingEndDate": "2026-05-31T23:59:00+00:00", "deliveryEndDate": "2026-06-30T23:59:00+00:00"}, {"id": "expiry-jul25-Id", "shortName": "Jul-25", "tradingEndDate": "2025-06-30T23:59:00+00:00", "deliveryEndDate": "2025-07-31T23:59:00+00:00"} ]
                    }
                },
                {
                    "id": trade_to_cancel_id, "externalId": trade_to_cancel_external_id, "dealtPriceId": "222",
                    "trade": { "id": trade_to_cancel_id, "externalId": trade_to_cancel_external_id, "expiryId": "expiry-jun25-Id", "instrumentId": "instrument-id", "instrumentName": "testInstrumentSpreadPy", "side": 'b', "value": 2, "quantity": 5, "priceType": "spread", "tradeType": "sphereLive", "time": "2025-06-23T23:12:00+00:00",
                        "expiries": [ {"id": "expiry-jun25-Id", "shortName": "Jun-25", "tradingEndDate": "2026-05-31T23:59:00+00:00", "deliveryEndDate": "2026-06-30T23:59:00+00:00"}, {"id": "expiry-jul25-Id", "shortName": "Jul-25", "tradingEndDate": "2025-06-30T23:59:00+00:00", "deliveryEndDate": "2025-07-31T23:59:00+00:00"} ]
                    }
                }
            ]
        }

        inject_add_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=ADDED"
        try:
            requests.post(inject_add_url, json=add_trades_payload, timeout=5).raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject ADDED trades: {e}")

        timeout_seconds = 5
        event_was_set = self.received_event.wait(timeout=timeout_seconds)
        self.assertTrue(event_was_set, f"Timeout: Did not receive the initial trades within {timeout_seconds} seconds.")
        self.assertEqual(self.received_event_types[0], sphere_sdk_types_pb2.TRADE_EVENT_TYPE_EXECUTED)
        self.assertEqual(len(self.received_trade_data), 2, "There should be exactly two trades initially.")

        self.received_event.clear()

        cancel_trade_payload = { "trades": [{ "id": trade_to_cancel_id, "externalId": trade_to_cancel_external_id }] }
        inject_cancel_url = f"{self.base_url}/_testing/signalr/inject?method=ReceiveDataMessage&arg1=SPHERE_LIVE_TRADE&arg2=CANCELLED"
        try:
            requests.post(inject_cancel_url, json=cancel_trade_payload, timeout=5).raise_for_status()
        except requests.exceptions.RequestException as e:
            self.fail(f"Failed to inject CANCELLED trade: {e}")
        
        expected_messages = 2 
        start_time = time.time()

        while self.messages_processed < expected_messages and time.time() - start_time < timeout_seconds:
            self.received_event.wait(timeout=timeout_seconds)
            self.received_event.clear()

        if self.messages_processed < expected_messages:
            self.fail(f"Timeout: Expected at least {expected_messages} messages, but only processed {self.messages_processed}.")

        self.assertEqual(self.received_event_types[-1], sphere_sdk_types_pb2.TRADE_EVENT_TYPE_VOIDED)
        self.assertEqual(len(self.received_trade_data), 1, "There should be exactly one trade showing voided trade.")

        voided_trade = self.received_trade_data[trade_to_cancel_external_id]
        self.assertEqual(voided_trade.id, trade_to_cancel_external_id)
        self.assertEqual(voided_trade.interest_type, sphere_sdk_types_pb2.INTEREST_TYPE_LIVE)

if __name__ == '__main__':
    unittest.main()