import json
import os
from typing import List, Dict, Any, Iterable
from ToonEscape.Constants.Constants import Constants

class ToonParser:
    def __init__(self):
        self.data = None

    def _load_json(self, json_input):
        if isinstance(json_input, str) and os.path.isfile(json_input):
            with open(json_input, "r", encoding=Constants.ENCODING) as f:
                content = f.read().strip()

            if not content:
                raise ValueError(f"JSON file '{json_input}' is empty.")

            try:
                self.data = json.loads(content)
            except json.JSONDecodeError as e:
                raise json.JSONDecodeError(
                    f"Failed to parse JSON file '{json_input}': {e.msg}", e.doc, e.pos
                ) from e
            return self.data

        if isinstance(json_input, dict):
            if not json_input:
                raise ValueError("Provided JSON dict is empty.")
            self.data = json_input
            return self.data

        if isinstance(json_input, str):
            content = json_input.strip()
            if not content:
                raise ValueError("Provided JSON string is empty.")

            try:
                self.data = json.loads(content)
                return self.data
            except json.JSONDecodeError:
                raise json.JSONDecodeError(
                    "Provided string is not valid JSON and no file was found at the path.",
                    json_input,
                    0,
                )
        raise TypeError("Input must be: file path, dict, or JSON string.")

    def _is_scalar(self, v: Any) -> bool:
        return isinstance(v, (int, float, bool, str, type(None)))

    def _is_list_of_scalars(self, lst: Iterable) -> bool:
        if not isinstance(lst, list):
            return False
        return all(self._is_scalar(x) for x in lst)

    def _is_list_of_dicts(self, lst: Iterable) -> bool:
        if not isinstance(lst, list) or not lst:
            return False
        return all(isinstance(x, dict) for x in lst)

    def _dicts_have_uniform_keys(self, lst: List[Dict]) -> bool:
        """Return True if every dict has the same set of keys."""
        if not lst:
            return True
        keyset = set(lst[0].keys())
        return all(set(d.keys()) == keyset for d in lst)

    def _all_dict_values_scalar(self, lst: List[Dict]) -> bool:
        """Return True if every dict's values are scalar (no nested dict/list)."""
        for d in lst:
            for v in d.values():
                if not self._is_scalar(v):
                    return False
        return True

    def _format_scalar(self, v: Any) -> str:
        if v is None:
            return Constants.NONE_STR
        if isinstance(v, bool):
            return Constants.TRUE_STR if v else Constants.FALSE_STR
        return str(v)

    def _handle_single_value(self, key: str, value: Any, indent: int = Constants.INDENT) -> str:
        pad = " " * indent
        return f"{pad}{key}:{self._format_scalar(value)}\n"

    def _handle_dict_value(self, dict_value: Dict[str, Any], indent: int = Constants.INDENT) -> str:
        out = ""
        pad = " " * indent
        for key, val in dict_value.items():
            if isinstance(val, dict):
                out += f"{pad}{key}:\n"
                out += self._handle_dict_value(val, indent + Constants.INDENT)
            elif isinstance(val, list):
                out += self._handle_list_value(key, val, indent + Constants.INDENT)
            else:
                out += self._handle_single_value(key, val, indent)
        return out

    def _handle_list_of_scalars(self, list_name: str, list_values: List[Any], indent: int) -> str:
        pad = " " * indent
        out = f"{pad}{list_name}[{len(list_values)}]:\n"
        item_pad = " " * (indent + Constants.INDENT)
        for item in list_values:
            out += f"{item_pad}- {self._format_scalar(item)}\n"
        return out

    def _handle_list_of_onekey_dicts_as_bullets(self, list_name: str, list_values: List[Dict], indent: int) -> str:
        """
        When each dict has exactly one key and values are scalar:
        teams[1]:
          - name: Team Alpha
          - name: Team Beta
        """
        pad = " " * indent
        out = f"{pad}{list_name}[{len(list_values)}]:\n"
        item_pad = " " * (indent + Constants.INDENT)
        for d in list_values:
            (k, v), = d.items()
            out += f"{item_pad}- {k}: {self._format_scalar(v)}\n"
        return out

    def _handle_list_as_csv(self, list_name: str, list_values: List[Dict], indent: int) -> str:
        """
        members[2]{id,name}:
            1,Alice
            2,Bob
        """
        pad = " " * indent
        keys = list(list_values[0].keys())
        keys_str = ",".join(keys)
        out = f"{pad}{list_name}[{len(list_values)}]{{{keys_str}}}:\n"
        row_pad = " " * (indent + Constants.INDENT)
        for d in list_values:
            row = ",".join(self._format_scalar(d[k]) for k in keys)
            out += f"{row_pad}{row}\n"
        return out

    def _handle_list_nonuniform_or_nested(self, list_name: str, list_values: List[Any], indent: int) -> str:
        """
        Generic recursive handling for lists with nested structures / mixed items.
        We print the count and then recursively render each item with a leading dash.
        """
        pad = " " * indent
        out = f"{pad}{list_name}[{len(list_values)}]:"
        item_pad = " " * (indent + Constants.INDENT)
        for item in list_values:
            if self._is_scalar(item):
                out += f"{item_pad}- {self._format_scalar(item)}\n"
            elif isinstance(item, dict):
                out += f"{item_pad}\n"
                out += self._handle_dict_value(item, indent + 2 * Constants.INDENT)
            elif isinstance(item, list):
                out += f"{item_pad}-\n"
                out += self._handle_list_value(list_name="item", list_values=item, indent=indent + 2 * Constants.INDENT)
            else:
                out += f"{item_pad}- {self._format_scalar(item)}\n"
        return out

    def _handle_list_value(self, list_name: str, list_values: List[Any], indent: int = Constants.INDENT) -> str:
        """
        Decide how to render a list depending on its contents:
        - scalar list -> bullets
        - list of dicts with uniform keys and scalar values -> CSV
        - list of dicts each with 1 key -> bullets with key:value
        - otherwise recursive nested representation
        """
        if not list_values:
            pad = " " * indent
            return f"{pad}{list_name}[0]:\n"

        if self._is_list_of_scalars(list_values):
            return self._handle_list_of_scalars(list_name, list_values, indent)

        if self._is_list_of_dicts(list_values):
            if all(len(d.keys()) == 1 and self._all_dict_values_scalar([d]) for d in list_values):
                return self._handle_list_of_onekey_dicts_as_bullets(list_name, list_values, indent)

            if self._dicts_have_uniform_keys(list_values) and self._all_dict_values_scalar(list_values):
                return self._handle_list_as_csv(list_name, list_values, indent)

            return self._handle_list_nonuniform_or_nested(list_name, list_values, indent)

        return self._handle_list_nonuniform_or_nested(list_name, list_values, indent)


    def _parse_value(self, key: str, value: Any, indent: int = 0) -> str:
        if isinstance(value, dict):
            return f"{' ' * indent}{key}:\n" + self._handle_dict_value(value, indent + Constants.INDENT)
        elif isinstance(value, list):
            return self._handle_list_value(key, value, indent)
        else:
            return f"{' ' * indent}{key}:{self._format_scalar(value)}\n"
        
    def parseToon(self, json_data = None) -> str:
        """
        Accepts:
        - A file path to a JSON file
        - A Python dict (already parsed JSON)
        - A JSON string
        """
        if json_data is None:
            raise ValueError("No JSON Data Found")
        self.data = self._load_json(json_data)

        if self.data is None:
            raise ValueError("No JSON data loaded.")

        def recurse(obj, indent: int = 0):
            result = ""
            if isinstance(obj, dict):
                for key, value in obj.items():
                    result += self._parse_value(key, value, indent)
            elif isinstance(obj, list):
                result += self._handle_list_value(Constants.ROOT_NAME, obj, indent)
            else:
                result += " " * indent + str(obj) + "\n"
            return result

        return recurse(self.data).rstrip()