
# Database Migrations with Alembic

This document outlines the essential commands for managing database schema changes using Alembic. Use it as a quick reference for creating new migration scripts and applying them to the database.

Run any commands from the database directory, where the
`alembic.ini` file is located.
```bash
cd localstack-pro-core/localstack/pro/core/eventstudio/database
```


## 1. Running Migrations (Upgrading the Database)

To apply all pending migrations and bring the database schema to the latest version, run:

```bash
alembic upgrade head
```

This command checks the current database version and applies all new migration scripts in order.

To upgrade to a specific version, use its revision ID:
```bash
alembic upgrade <revision_id>
```

## 2. Creating a New Migration

Follow this three-step process whenever you change your SQLAlchemy models (e.g., add a column or a new table).

#### Step 1: Generate the Migration Script

After changing your models, run the `revision` command with the `--autogenerate` flag. This detects your model changes and generates a new migration script.

```bash
# Use a short, descriptive message for the -m flag
alembic revision --autogenerate -m "Add arn column to spans table"
```

#### Step 2: Review and Edit the Generated Script

**This is the most important step.** Alembic's autogeneration is not perfect.

1.  **Open the new file** located in the `alembic/versions/` directory.
2.  **Verify the `upgrade()` function** correctly reflects your intended changes.
3.  **Verify the `downgrade()` function** correctly reverses those changes.
4.  **Manually adjust the script** if anything is missing or incorrect. Complex constraints, server defaults, or new enum types often require manual adjustment.

```python
"""Add arn column to spans table

Revision ID: 1a2b3c4d5e
Revises: 9f8e7d6c5b
Create Date: 2025-07-25 10:30:00.123456

"""
from alembic import op
import sqlalchemy as sa

# revision identifiers, used by Alembic.
revision = '1a2b3c4d5e'
down_revision = '9f8e7d6c5b'
branch_labels = None
depends_on = None

def upgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    op.add_column('spans', sa.Column('arn', sa.String(), nullable=False))
    # ### end Alembic commands ###

def downgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    op.drop_column('spans', 'arn')
    # ### end Alembic commands ###
```

#### Step 3: Commit the Script
Once the script is correct, commit the new migration file to your version control system (e.g., Git). It is a part of your application's source code.


## 3. Downgrading and Resetting the Database

Downgrading reverts a migration. **Use with caution**, as this can lead to data loss.

*   **Downgrade by one revision:**
    ```bash
    alembic downgrade -1
    ```

*   **Revert all migrations to an empty schema:**
    ```bash
    alembic downgrade base
    ```

## 4. Checking Database Status

These commands are read-only and are safe to run at any time.

*   **Show the current revision of the database:**
    ```bash
    alembic current
    ```

*   **Show the full history of all revisions:**
    ```bash
    alembic history
    ```

*   **Show only the migrations that have not yet been applied:**
    ```bash
    alembic history --rev-range $(alembic current):head
    ```
