from os import makedirs
from os.path import join
import sys
import argparse

BASE_MOONFILE="""#https://github.com/ranon-rat/moonmake
import moonmake as mmake
import platform
import sys

dir_path = mmake.get_dir(__file__)
def join (*dir, separator="/"): return f"{separator}".join(dir)
def get_raylib_url():
    \"""Determines the Raylib download URL based on the operating system.\"""
    system = platform.system()
    BASE_URL = "https://github.com/raysan5/raylib/releases/download/5.5"
    
    if system == "Windows":
        return f"{BASE_URL}/raylib-5.5_win64_mingw-w64.zip"
    elif system == "Darwin":  # macOS
        return f"{BASE_URL}/raylib-5.5_macos.tar.gz"
    elif system == "Linux":
        return f"{BASE_URL}/raylib-5.5_linux_amd64.tar.gz"
    else:
        raise Exception(f"Unsupported system: {system}")

def install():
    \"""Downloads and installs the necessary dependencies.\"""
    raylib_url = get_raylib_url()
    mmake.download_dependency(
        raylib_url, 
        "raylib", 
        ".moonmake/dependencies", 
        headers=["include"]
    )

def execute():
    \"""Configures and executes the build process.\"""
    # Directories and configuration
    MOONMAKE_DIR = ".moonmake"
    PROJECT_NAME = "msrc"
    CPP_VERSION = "2b"
    EXTENSION = mmake.get_extension()
    
    # Important paths
    include_paths = [
        join(".", MOONMAKE_DIR, "dependencies", "headers"),
        join(".", dir_path, "src", "include")
    ]
    
    lib_paths = [
        join(MOONMAKE_DIR, "dependencies", "lib"),
        join( MOONMAKE_DIR, "lib")
    ]
    
 
    # Static libraries
    static_a_files = mmake.discover(join(".", MOONMAKE_DIR, "dependencies", "lib"), ".a")
    static_libs = [f"-l{mmake.strip_lib_prefix(a).replace('.a', '')}" for a in static_a_files]
    
    if platform.system() == "Windows":
        static_libs.extend(["-lgdi32", "-lwinmm"])
    
    # Compilation flags
    INCLUDE_FLAGS = mmake.join_with_flag(include_paths, "-I")
    LINK_FLAGS = mmake.join_with_flag(lib_paths, "-L")
    STATIC_LIBRARY = " ".join(static_libs)
    COMPILER_FLAGS = f"-Wall -Wextra -std=c++{CPP_VERSION} -Werror -O2"
    IGNORE_FLAGS = "-Wno-unused-parameter -Wno-return-type"  
    OBJ_FLAGS= "-MMD -MP"
    # Files to watch for changes
    static_watch_files = [join(MOONMAKE_DIR, "dependencies", "lib", a) for a in static_a_files]
    
    # Library files
    lib_files = [f for f in mmake.discover(join(dir_path, "src", "lib"), ".cpp")]
    lib_obj = mmake.change_extension(
        lib_files, 
        join(dir_path, MOONMAKE_DIR, "obj", "lib"), 
        old=".cpp", 
        new=".o"
    )
    lib_static = join(dir_path, MOONMAKE_DIR, "lib", f"lib{PROJECT_NAME}.a")
    
    # Target files (binaries)
    target_files = [f for f in mmake.discover(join(dir_path, "src", "target"), ".cpp")]
    print(target_files)
    target_obj = mmake.change_extension(
        target_files, 
        join(dir_path, MOONMAKE_DIR, "obj", "target"), 
        old=".cpp", 
        new=".o"
    )
    target_bin = mmake.change_extension(
        target_files, 
        join(dir_path, MOONMAKE_DIR, "bin"), 
        old=".cpp", 
        new=EXTENSION
    )
    
    # Configure the builder
    builder = mmake.Builder()
    
    # Rule to generate the final binaries
    builder.watch(
        target_bin, 
        target_obj, 
        f"g++ $< -o $@ {COMPILER_FLAGS} {LINK_FLAGS} {STATIC_LIBRARY} -l{PROJECT_NAME}",
    )
    
    # Rule to compile target object files
    builder.watch(
        target_obj, 
        [join(".", "src", "target", f) for f in target_files],
        f"g++ -c $< -o $@ {COMPILER_FLAGS} {INCLUDE_FLAGS} {IGNORE_FLAGS}",
        dependency_file=True
    )
    
    # Rule to create the static library
    builder.watch(
        [lib_static], 
        lib_obj, 
        "ar rcs $@ $^"
    )
    
    # Rule to compile library object files
    builder.watch(
        lib_obj, 
        [join(".", "src", "lib", f) for f in lib_files],
        f"g++ {COMPILER_FLAGS} {IGNORE_FLAGS} -c $< -o $@ {INCLUDE_FLAGS} {OBJ_FLAGS}",
        dependency_file=True
    )
    
    # Execute all build rules
    builder.compile_all()

if __name__ == "__main__":
    mmake.arguments_cmd(sys.argv, execute, install)


"""
def build_folder(route:str,readme:str):
    makedirs(route,exist_ok=True)
    with open(join(route,"README.md"),"w") as f:
        f.write(readme+"\n")
def create_new(name:str):
    global BASE_MOONFILE
    moonmake=".moonmake"# its better to keep this thing hidden :)
    routes=[
        (join(".",name,moonmake,"obj"),"compiled object files"),#a,o,.exe
        (join(".",name,moonmake,"bin"),"binary files"),#a,o,.exe
        (join(".",name,moonmake,"lib"),"library files generated by the program"),#a,o,.exe
        (join(".",name,moonmake,"dependencies","headers"),"headers from the dependencies"),
        (join(".",name,moonmake,"dependencies","libs"),"libraries to link"),
        (join(".",name,"src","include"),"place here the header files and only initialize your functions"),
        (join(".",name,"src","target"),"here it goes the target files, every file that you place here will have its own binary"),
        (join(".",name,"src","lib"),"define your functions here"),
    ]
    for (p,r) in routes:build_folder(p,r)
    with open(join(".",name,"build.py"),"w") as f:
        f.write(BASE_MOONFILE)
def __main():
    parser=argparse.ArgumentParser(description="create a new moonmake project")
    parser.add_argument("-c","--compiler",help="select the compiler that you want to use",default="g++")
    parser.add_argument("-n","--name",help="the name of the project that you want to create",required=True)
    args=vars(parser.parse_args())
    name=args["name"]
    create_new(name)
if __name__=="__main__":
    __main()