import gurobipy as gp
from gurobipy import GRB


class Problem:
    """
    Defines a problem instance.

    Parameters:
    - instance (dict): Generated by higher level functions, constains the instance data.
    """
    def __init__(self, instance = None):
        self.instance = instance
        self.model = gp.Model("Problem")
        self.define_solver_parameters()
        self.root_bound = None
    
    def set_instance(self, instance):
        self.instance = instance
    
    def unpack_instance(self):

        self.S = self.instance["S"]
        self.J = self.instance["J"]
        self.T = self.instance["T"]
        self.Jc = self.instance["Jc"]
        self.Js = self.instance["Js"]
        self.power_resourse = self.instance["power_resource"]

        self.sat_priority = self.instance["sat_priority"]
        self.sat_power_consumption = self.instance["sat_power_consumption"]
        self.sat_min_startup = self.instance["sat_min_startup"]
        self.sat_max_startup = self.instance["sat_max_startup"]
        self.sat_min_cpu_time = self.instance["sat_min_cpu_time"]
        self.sat_max_cpu_time = self.instance["sat_max_cpu_time"]
        self.sat_min_period_job = self.instance["sat_min_period_job"]
        self.sat_max_period_job = self.instance["sat_max_period_job"]
        self.sat_win_min = self.instance["sat_win_min"]
        self.sat_win_max = self.instance["sat_win_max"]

        self.const_priority = self.instance["const_priority"]
        self.const_power_consumption = self.instance["const_power_consumption"]
        self.const_min_startup = self.instance["const_min_startup"]
        self.const_max_startup = self.instance["const_max_startup"]
        self.const_min_cpu_time = self.instance["const_min_cpu_time"]
        self.const_max_cpu_time = self.instance["const_max_cpu_time"]
        self.const_min_period_job = self.instance["const_min_period_job"]
        self.const_max_period_job = self.instance["const_max_period_job"]
        self.const_win_min = self.instance["const_win_min"]
        self.const_win_max = self.instance["const_win_max"]
        self.const_concurrency_max = self.instance["const_concurrency_max"]
        self.const_concurrency_min = self.instance["const_concurrency_min"] 
        
        self.soc_inicial = self.instance["soc_inicial"]
        self.limite_inferior = self.instance["limite_inferior"]
        self.ef = self.instance["ef"]
        self.v_bat = self.instance["v_bat"]
        self.q = self.instance["q"]
        self.bat_usage = self.instance["bat_usage"]


    def build_and_solve(self):
        self.unpack_instance()
        self.build_model()
        self.model.optimize()
        
        if self.model.status == GRB.OPTIMAL:
            return {
                'model': self.model,  # Make sure to include the model
                'x': self.x,
                'phi': self.phi,
                'soc': self.soc,
                'b': self.b,
                'i': self.i
            }
        return None

    def write_solution(self):
        with open('solution_output.txt', 'w') as file:
            if self.model.status == GRB.Status.OPTIMAL:
                file.write("Optimal Solution Found:\n")
                for s in range(self.S):
                    file.write(f"--- Satellite {s} ---\n")
                    for t in range(self.T):
                        active_jobs = [j for j in range(self.J) if self.x[s, j, t].x > 0.5]
                        startups = [j for j in range(self.J) if self.phi[s, j, t].x > 0.5]
                        file.write(f"Time {t}: Active Jobs: {active_jobs}, Startups: {startups}\n")
            else:
                file.write("No optimal solution found.\n")

    def build_model(self):
        """
        Builds the Gurobi model for the instance.

        """
        self.build_variables()
        self.set_objective()
        self.add_constraints()
    
    def define_solver_parameters(self):
        #self.model.setParam('MIPFocus', 1)
        hour = 3600
        self.model.setParam('TimeLimit', hour * 1)
        #self.model.setParam("SolutionLimit", 1)  # Stop as soon as a feasible solution is found
        #self.model.setParam('OutputFlag', 0)  # Disable output (stdout
        pass

    def build_variables(self):
        # Remove alpha variable as it's not used in the mathematical formulation
        self.x = {}
        self.phi = {}
        self.soc = {}
        self.i = {}
        self.b = {}
        for s in range(self.S):
            for j in range(self.J):
                for t in range(self.T):
                    self.x[s,j,t] = self.model.addVar(name="x(%s,%s,%s)" % (s, j, t), lb=0, ub=1, vtype="BINARY")
                for t in range(self.T):
                    self.phi[s,j,t] = self.model.addVar(name="phi(%s,%s,%s)" % (s, j, t), lb=0, ub=1, vtype="BINARY")
            for t in range(self.T):
                self.soc[s, t] = self.model.addVar(
                    vtype=GRB.CONTINUOUS, 
                    name="soc(%s,%s)" % (s, t),
                    lb=self.limite_inferior,
                    ub=1
                )
                self.i[s, t] = self.model.addVar(
                    lb=-GRB.INFINITY, 
                    vtype=GRB.CONTINUOUS, 
                    name="i(%s,%s)" % (s, t)
                )
                self.b[s, t] = self.model.addVar(
                    lb=-GRB.INFINITY, 
                    vtype=GRB.CONTINUOUS, 
                    name="b(%s,%s)" % (s, t)
                )

    def set_objective(self):
        self.model.setObjective(gp.quicksum(self.sat_priority[s][j] * self.x[s, j + self.Jc, t] for s in range(self.S) for j in range(self.Js) for t in range(self.T))
                                + gp.quicksum(self.const_priority[j] * self.x[s, j , t] for s in range(self.S) for j in range(self.Jc) for t in range(self.T))
                                , GRB.MAXIMIZE)
        pass

    def add_constraints(self):
        self.add_power_constraints()
        self.add_phi_constraints()
        self.add_job_constraints()

    def add_power_constraints(self):
        for s in range(self.S):
            for t in range(self.T):
                # Total power consumption
                total_power = (
                    gp.quicksum(self.const_power_consumption[j] * self.x[s, j, t] for j in range(self.Jc)) +
                    gp.quicksum(self.sat_power_consumption[s][j] * self.x[s, j + self.Jc, t] for j in range(self.Js))
                )
                
                # Energy balance constraint (eq:milp_energy_balance)
                # b_{s,t} = r_{s,t} - power_consumed
                self.model.addConstr(
                    self.b[s, t] == self.power_resourse[s][t] - total_power
                )
                
                # Total current constraint (eq:milp_total_current)
                self.model.addConstr(
                    self.i[s, t] == self.b[s, t] / self.v_bat
                )
                
                # Battery usage constraint (eq:gamma)
                self.model.addConstr(
                    total_power <= self.power_resourse[s][t] + self.bat_usage * self.v_bat
                )
            
            # SOC constraints (eq:soc-cst)
            for t in range(self.T):
                if t == 0:
                    # Initial SOC (eq:soc_zero)
                    self.model.addConstr(
                        self.soc[s, t] == self.soc_inicial[s] + (self.ef / self.q) * (self.i[s, t] / 60)
                    )
                else:
                    # SOC update (eq:soc_delta)
                    self.model.addConstr(
                        self.soc[s, t] == self.soc[s, t-1] + (self.ef / self.q) * (self.i[s, t] / 60)
                    )
                
                # SOC bounds (eq:soc_min and eq:soc_max)
                # These are already handled by variable bounds in build_variables
                # but we can add explicit constraints for clarity
                self.model.addConstr(self.soc[s, t] >= self.limite_inferior)
                self.model.addConstr(self.soc[s, t] <= 1)

    def add_phi_constraints(self):
        for s in range(self.S):
            for t in range(self.T):
                for j in range(self.J):
                    if t == 0:
                        self.model.addConstr(self.phi[s, j, t] >= self.x[s, j, t] - 0)
                    else:
                        self.model.addConstr(self.phi[s, j, t] >= self.x[s, j, t] - self.x[s, j, t - 1])

                    self.model.addConstr(self.phi[s, j, t] <= self.x[s, j, t])

                    if t == 0:
                        self.model.addConstr(self.phi[s, j, t] <= 2 - self.x[s, j, t] - 0)
                    else:
                        self.model.addConstr(self.phi[s, j, t] <= 2 - self.x[s, j, t] - self.x[s, j, t - 1])
    
    def add_job_constraints(self):
        #Constellation Constraints
        for j in range(self.Jc):
                self.model.addConstr(
                    gp.quicksum(self.phi[s, j, t] for s in range(self.S) for t in range(self.T)) <= self.const_max_startup[j]
                    , name = f"constellation_max_startup_{j}"
                )
                self.model.addConstr(
                    gp.quicksum(self.phi[s, j, t] for s in range(self.S) for t in range(self.T)) >= self.const_min_startup[j]
                    , name = f"global_startup_min_{j}"
                )
                for t in range(self.T):
                          self.model.addConstr(gp.quicksum(self.x[s,j,t] for s in range(self.S)) <= self.const_concurrency_max[j])
                          self.model.addConstr(gp.quicksum(self.x[s,j,t] for s in range(self.S)) >= self.const_concurrency_min[j])   
                
                for t in range(0, self.T - self.const_min_period_job[j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.phi[s, j, t_] for t_ in range(t, t + self.const_min_period_job[j]) for s in range(self.S))
                        <= 1,
                        name = f"global_min_period_job_{j,t}"
                    )
                for t in range(0, self.T - self.const_max_period_job[j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.phi[s, j, t_] for t_ in range(t, t + self.const_max_period_job[j]) for s in range(self.S))
                        >= 1,
                        name = f"global_max_period_job_{j,t}"
                    )


        for s in range(self.S):
            for j in range(self.Js):
                self.model.addConstr(gp.quicksum(self.phi[s, j + self.Jc, t] for t in range(self.T)) >= self.sat_min_startup[s][j], name = f"local_min_startup_{s,j}")
                self.model.addConstr(gp.quicksum(self.phi[s, j + self.Jc, t] for t in range(self.T)) <= self.sat_max_startup[s][j], name = f"local_startup_max_{s,j}")

                self.model.addConstr(gp.quicksum(self.x[s, j + self.Jc, t] for t in range(self.sat_win_min[s][j])) == 0)
                self.model.addConstr(gp.quicksum(self.x[s, j + self.Jc, t] for t in range(self.sat_win_max[s][j]+1, self.T)) == 0)

            for j in range(self.Js):
                for t in range(0, self.T - self.sat_min_period_job[s][j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.phi[s, j + self.Jc, t_] for t_ in range(t, t + self.sat_min_period_job[s][j]))
                        <= 1,
                        name = f"local_min_period_job_{s,j,t}"
                    )
                for t in range(0, self.T - self.sat_max_period_job[s][j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.phi[s, j + self.Jc, t_] for t_ in range(t, t + self.sat_max_period_job[s][j]))
                        >= 1,
                        name = f"local_max_period_job_{s,j,t}"
                    )
                for t in range(0, self.T - self.sat_min_cpu_time[s][j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j + self.Jc, t_] for t_ in range(t, t + self.sat_min_cpu_time[s][j]))
                        >= self.sat_min_cpu_time[s][j] * self.phi[s, j + self.Jc, t]
                    )
                for t in range(0, self.T - self.sat_max_cpu_time[s][j]):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j + self.Jc, t_] for t_ in range(t, t + self.sat_max_cpu_time[s][j] + 1))
                        <= self.sat_max_cpu_time[s][j]
                    )
                for t in range(self.T - self.sat_min_cpu_time[s][j] + 1, self.T):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j + self.Jc, t_] for t_ in range(t, self.T)) >= (self.T - t) * self.phi[s, j + self.Jc, t]
                    )
                for t in range(self.T):
                        self.model.addConstr(gp.quicksum(self.phi[s,j + self.Jc,t] for t in range(t,min(self.T, t+self.sat_min_cpu_time[s][j]+1))) <= 1, name = f"VI_min_CPU_TIME_phi({j},{t})")
                        self.model.addConstr(gp.quicksum(self.x[s,j + self.Jc,t] for t in range(self.T)) <= self.sat_max_cpu_time[s][j]*gp.quicksum(self.phi[s,j + self.Jc,t] for t in range(self.T)),name = f"VI_max_cpu_time_2({j})")

                for t in range(self.T - self.sat_max_cpu_time[s][j]):
                        self.model.addConstr(gp.quicksum(self.x[s,j + self.Jc,t] for t in range(t, t + self.sat_max_cpu_time[s][j])) <= self.sat_max_cpu_time[s][j]*gp.quicksum(self.phi[s,j + self.Jc,t] for t in range(max(t - self.sat_max_cpu_time[s][j] + 1,0),t + self.sat_max_cpu_time[s][j], 1)), name = f"VI_max_cpu_time_3({j},{t})")

                for t in range(self.T - self.sat_min_period_job[s][j]+1):
                        self.model.addConstr(gp.quicksum(self.x[s,j + self.Jc,t] for t in range(t, t + self.sat_min_period_job[s][j])) <= self.sat_min_period_job[s][j], name = f"VI_min_period_btw_jobs_2({j},{t})")

                if self.sat_max_cpu_time[s][j] < self.sat_max_period_job[s][j] - self.sat_min_cpu_time[s][j]:
                        for t in range(self.T - self.sat_max_cpu_time[s][j]):
                                self.model.addConstr(self.phi[s,j + self.Jc,t] + self.x[s,j + self.Jc,t+self.sat_max_cpu_time[s][j]]<=1)

            for j in range(self.Jc):
                for t in range(0, self.T - self.const_min_cpu_time[j] + 1):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j, t_] for t_ in range(t, t + self.const_min_cpu_time[j]))
                        >= self.const_min_cpu_time[j] * self.phi[s, j, t]
                    )
                for t in range(0, self.T - self.const_max_cpu_time[j]):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j, t_] for t_ in range(t, t + self.const_max_cpu_time[j] + 1))
                        <= self.const_max_cpu_time[j]
                    )
                for t in range(self.T - self.const_min_cpu_time[j] + 1, self.T):
                    self.model.addConstr(
                        gp.quicksum(self.x[s, j, t_] for t_ in range(t, self.T)) >= (self.T - t) * self.phi[s, j, t]
                    )
                for t in range(self.T):
                        self.model.addConstr(gp.quicksum(self.phi[s,j,t] for t in range(t,min(self.T, t+self.const_min_cpu_time[j]+1))) <= 1, name = f"VI_min_CPU_TIME_phi_const({j})")
                        self.model.addConstr(gp.quicksum(self.x[s,j,t] for t in range(self.T)) <= self.const_max_cpu_time[j]*gp.quicksum(self.phi[s,j,t] for t in range(self.T)),name = f"VI_max_cpu_time_2_const({j})")

                for t in range(self.T - self.const_max_cpu_time[j]):
                        self.model.addConstr(gp.quicksum(self.x[s,j,t] for t in range(t, t + self.const_max_cpu_time[j])) <= self.const_max_cpu_time[j]*gp.quicksum(self.phi[s,j,t] for t in range(max(t - self.const_max_cpu_time[j] + 1,0),t + self.const_max_cpu_time[j], 1)), name = f"VI_max_cpu_time_3_const({j},{t})")

                self.model.addConstr(gp.quicksum(self.x[s, j, t] for t in range(self.const_win_min[j])) == 0)
                self.model.addConstr(gp.quicksum(self.x[s, j, t] for t in range(self.const_win_max[j]+1, self.T)) == 0)

                

import numpy as np


class InstanceGenerator:
    #crete init with data hints
    def __init__(self, S: int, Js: int, T: int, Jc: int):
        self.S = S
        self.Jc = Jc
        self.Js = Js
        self.J = self.Jc + self.Js
        self.T = T
        self.initialize_instance()
        
    def initialize_instance(self):
        
        self.power_resource = np.zeros((self.S, self.T))

        self.sat_power_consumption = np.zeros((self.S, self.Js))
        self.sat_priority = np.zeros((self.S, self.Js))
        self.sat_min_startup = np.zeros((self.S, self.Js))
        self.sat_max_startup = np.zeros((self.S, self.Js))
        self.sat_min_cpu_time = np.zeros((self.S, self.Js))
        self.sat_max_cpu_time = np.zeros((self.S, self.Js))
        self.sat_min_period_job = np.zeros((self.S, self.Js))
        self.sat_max_period_job = np.zeros((self.S, self.Js))
        self.sat_win_min = np.zeros((self.S, self.Js))
        self.sat_win_max = np.zeros((self.S, self.Js))

        self.const_power_consumption = np.zeros(self.Jc)
        self.const_priority = np.zeros(self.Jc)
        self.const_min_startup = np.zeros(self.Jc)
        self.const_max_startup = np.zeros(self.Jc)
        self.const_min_cpu_time = np.zeros(self.Jc)
        self.const_max_cpu_time = np.zeros(self.Jc)
        self.const_min_period_job = np.zeros(self.Jc)
        self.const_max_period_job = np.zeros(self.Jc)
        self.const_win_min = np.zeros(self.Jc)
        self.const_win_max = np.zeros(self.Jc)
        self.const_concurrency_min = np.zeros(self.Jc)
        self.const_concurrency_max = np.zeros(self.Jc)

        self.soc_inicial = np.zeros(self.S)
        self.limite_inferior = 0.3
        self.ef = 0.9
        self.v_bat = 3.6
        self.q = 5
        self.bat_usage = 5    

    def generate_instance(self):
        # Parameters
        min_power = 0.3
        max_power = 4.0
        orbit_power = np.loadtxt('resource.csv')
        self.const_priority = np.random.choice(range(1, self.J + 1), self.Jc, replace=False)
        remaining_priorities = [x for x in range(1, self.J + 1) if x not in self.const_priority]

        ## define constellation tasks parameters

        for j in range(self.Jc):

            self.const_power_consumption[j] = np.random.uniform(min_power, max_power)

            max_limit_min_startup = max(1, np.ceil(self.T /45))
            self.const_min_startup[j] = np.random.randint(0, max_limit_min_startup + 1)

            min_limit_max_startup = np.ceil(self.T / 45)
            max_limit_max_startup = np.ceil(self.T /15)
            min_bound = max(self.const_min_startup[j] + 1, np.ceil(self.T / 45) + 1)
            self.const_max_startup[j] = np.random.randint(min_bound, max_limit_max_startup + 1 )
                                                            
            self.const_min_cpu_time[j] = np.random.randint(1, np.ceil(self.T / 10))
            min_bound = max(self.const_min_cpu_time[j]+1, np.ceil(self.T / 20)+1)
            self.const_max_cpu_time[j] = np.random.randint(min_bound, np.ceil(self.T / 4))

            self.const_min_period_job[j] = np.random.randint(self.const_min_cpu_time[j] + 1, np.ceil(self.T / 4) + 1)
            max_periodo_min_bound = np.ceil(self.T / 4)
            max_periodo_min_bound = 0
            min_bound = max(self.const_min_period_job[j]+1, np.ceil(self.T / 5)+1)
            self.const_max_period_job[j] = np.random.randint(min_bound, self.T + 1)
            self.const_win_min[j] = 0
            self.const_win_max[j] = self.T + 1

            self.const_concurrency_min[j] = 0
            self.const_concurrency_max[j] = 1

        for s in range(self.S):

            self.soc_inicial[s] = np.random.uniform(0.4, 0.9)
            # Random start for each satellite
            orbit_start = np.random.randint(0, 600)  
            # Uniform distribution of power usage between [min_power, max_power]
            self.power_resource[s] = orbit_power[orbit_start:orbit_start+self.T]
            self.sat_priority[s, :self.Jc] = self.const_priority
            self.sat_priority[s] = np.random.choice(remaining_priorities, self.Js, replace=False)

            for j in range(self.Js):
                
                self.sat_power_consumption[s][j] = np.random.uniform(min_power, max_power)
                # Random startup time between [1, T/45]
                min_limit = 0
                max_limit_min_startup = max(1, np.ceil(self.T /45))
                self.sat_min_startup[s][j] = np.random.randint(min_limit, max_limit_min_startup + 1)
                # Random startup time between [min_startup, T/15]
                max_limit_max_startup = np.ceil(self.T /15)
                min_limit_max_startup = 2
                min_bound = max(self.sat_min_startup[s][j]+1  , np.ceil(self.T / 45) + 1)
                self.sat_max_startup[s][j] = np.random.randint(min_bound, max_limit_max_startup + 1)
                # # Random min
                self.sat_min_cpu_time[s][j] = np.random.randint(1, np.ceil(self.T / 10))
                self.sat_max_cpu_time[s][j] = np.random.randint(self.sat_min_cpu_time[s][j]+1, np.ceil(self.T / 4)+1)
                # self.min_cpu_time[s][j] = np.random.randint(1, 11)
                # self.max_cpu_time[s][j] = np.random.randint(3,26)
                # Random min period between [min_cpu_time, T/4] and max period between [min_period, T]
                self.sat_min_period_job[s][j] = np.random.randint(self.sat_min_cpu_time[s][j] + 1, np.floor(self.T / 4) + 1)
                #max_periodo_min_bound = np.ceil(self.T / 3.5)
                max_periodo_min_bound = np.ceil(self.T / 5)
                min_bound = max(self.sat_min_period_job[s][j]+1, max_periodo_min_bound)
                self.sat_max_period_job[s][j] = np.random.randint(min_bound,  self.T + 1)
                # self.min_period_job[s][j] = np.random.randint(9,25)
                # self.max_period_job[s][j] = np.random.randint(np.ceil(self.T * 0.7), self.T)
                
                self.sat_win_min[s][j] = np.random.randint(0, np.ceil(self.T/20)+1)
                self.sat_win_min[s][j] = 0
                self.sat_win_max[s][j] = np.random.randint(self.T - np.ceil(self.T/20), self.T+1)
                self.sat_win_max[s][j] = self.T + 10



        return {
            "S": self.S,
            "J": self.J,
            "Js": self.Js,
            "Jc": self.Jc,
            "T": self.T,
            "power_resource": self.power_resource.tolist(),
            "sat_power_consumption": self.sat_power_consumption.tolist(),  
            "sat_min_cpu_time": self.sat_min_cpu_time.astype(int).tolist(),
            "sat_max_cpu_time": self.sat_max_cpu_time.astype(int).tolist(),
            "sat_min_period_job": self.sat_min_period_job.astype(int).tolist(),
            "sat_max_period_job": self.sat_max_period_job.astype(int).tolist(),
            "sat_min_startup": self.sat_min_startup.astype(int).tolist(),
            "sat_max_startup": self.sat_max_startup.astype(int).tolist(),
            "sat_priority": self.sat_priority.tolist(),
            "sat_win_min": self.sat_win_min.astype(int).tolist(),
            "sat_win_max": self.sat_win_max.astype(int).tolist(),
            "const_power_consumption": self.const_power_consumption.tolist(),
            "const_min_cpu_time": self.const_min_cpu_time.astype(int).tolist(),
            "const_max_cpu_time": self.const_max_cpu_time.astype(int).tolist(),
            "const_min_period_job": self.const_min_period_job.astype(int).tolist(),
            "const_max_period_job": self.const_max_period_job.astype(int).tolist(),
            "const_min_startup": self.const_min_startup.astype(int).tolist(),
            "const_max_startup": self.const_max_startup.astype(int).tolist(),
            "const_priority": self.const_priority.tolist(),
            "const_win_min": self.const_win_min.astype(int).tolist(),
            "const_win_max": self.const_win_max.astype(int).tolist(),
            "const_concurrency_min": self.const_concurrency_min.astype(int).tolist(),
            "const_concurrency_max": self.const_concurrency_max.astype(int).tolist(),
            'soc_inicial': self.soc_inicial.tolist(),
            'limite_inferior': self.limite_inferior,
            'ef': self.ef,
            'v_bat': self.v_bat,
            'q': self.q,
            'bat_usage': self.bat_usage

        }   
    

# File: solution_storage.py
import json

import numpy as np


class SolutionStorage:
    """
    Stores optimization results for later analysis and graphing.
    
    Parameters:
    - problem: The solved Problem instance
    - result: The result dictionary returned by Problem.build_and_solve()
    """
    def __init__(self, problem, result):
        self.instance = problem.instance
        self.status = problem.model.status
        self.objective_value = problem.model.ObjVal if self.status == 2 else None  # 2 = GRB.OPTIMAL
        self.runtime = problem.model.Runtime
        self.mip_gap = problem.model.MIPGap
        
        # Store solution if available
        self.solution = {}
        if result:
            self._store_solution(result, problem)
    
    def _store_solution(self, result, problem):
        """Extract and store solution variables"""
        # Basic dimensions
        S, J, T = problem.S, problem.J, problem.T
        
        # Store variable values
        self.solution['x'] = np.zeros((S, J, T), dtype=int)
        self.solution['phi'] = np.zeros((S, J, T), dtype=int)
        self.solution['soc'] = np.zeros((S, T))
        self.solution['b'] = np.zeros((S, T))
        self.solution['i'] = np.zeros((S, T))
        
        for s in range(S):
            for t in range(T):
                self.solution['soc'][s, t] = result['soc'][s, t].X
                self.solution['b'][s, t] = result['b'][s, t].X
                self.solution['i'][s, t] = result['i'][s, t].X
                
                for j in range(J):
                    self.solution['x'][s, j, t] = result['x'][s, j, t].X
                    self.solution['phi'][s, j, t] = result['phi'][s, j, t].X
        
        # Calculate derived metrics
        self._calculate_derived_metrics(problem)
    
    def _calculate_derived_metrics(self, problem):
        """Calculate additional metrics for analysis"""
        S, J, T = problem.S, problem.J, problem.T
        Jc, Js = problem.Jc, problem.Js
        
        # Job execution metrics
        self.metrics = {
            'total_jobs_executed': int(np.sum(self.solution['phi'])),
            'satellite_jobs': np.zeros(S),
            'constellation_jobs': np.zeros(Jc),
            'power_usage': np.zeros((S, T)),
            'battery_usage': np.zeros((S, T)),
        }
        
        for s in range(S):
            self.metrics['satellite_jobs'][s] = np.sum(
                self.solution['phi'][s, Jc:Jc+Js, :]
            )
            
            for t in range(T):
                # Power usage breakdown
                constellation_power = sum(
                    problem.const_power_consumption[j] * self.solution['x'][s, j, t]
                    for j in range(Jc)
                )
                satellite_power = sum(
                    problem.sat_power_consumption[s][j] * self.solution['x'][s, j+Jc, t]
                    for j in range(Js)
                )
                self.metrics['power_usage'][s, t] = constellation_power + satellite_power
                self.metrics['battery_usage'][s, t] = self.solution['b'][s, t]
    
    def to_dict(self):
        """Serialize solution to dictionary for storage"""
        return {
            'instance': self.instance,
            'status': self.status,
            'objective_value': self.objective_value,
            'runtime': self.runtime,
            'mip_gap': self.mip_gap,
            'solution': {
                'x': self.solution['x'].tolist(),
                'phi': self.solution['phi'].tolist(),
                'soc': self.solution['soc'].tolist(),
                'b': self.solution['b'].tolist(),
                'i': self.solution['i'].tolist(),
            },
            'metrics': {
                'total_jobs_executed': self.metrics['total_jobs_executed'],
                'satellite_jobs': self.metrics['satellite_jobs'].tolist(),
                'constellation_jobs': self.metrics['constellation_jobs'].tolist(),
                'power_usage': self.metrics['power_usage'].tolist(),
                'battery_usage': self.metrics['battery_usage'].tolist(),
            }
        }
    
    def save(self, filename):
        """Save solution to JSON file"""
        with open(filename, 'w') as f:
            json.dump(self.to_dict(), f, indent=2)
    
    @classmethod
    def load(cls, filename):
        """Load solution from JSON file"""
        with open(filename, 'r') as f:
            data = json.load(f)
        
        # Create dummy object
        storage = cls.__new__(cls)
        storage.instance = data['instance']
        storage.status = data['status']
        storage.objective_value = data['objective_value']
        storage.runtime = data['runtime']
        storage.mip_gap = data['mip_gap']
        
        # Recreate numpy arrays
        storage.solution = {
            'x': np.array(data['solution']['x']),
            'phi': np.array(data['solution']['phi']),
            'soc': np.array(data['solution']['soc']),
            'b': np.array(data['solution']['b']),
            'i': np.array(data['solution']['i']),
        }
        
        storage.metrics = {
            'total_jobs_executed': data['metrics']['total_jobs_executed'],
            'satellite_jobs': np.array(data['metrics']['satellite_jobs']),
            'constellation_jobs': np.array(data['metrics']['constellation_jobs']),
            'power_usage': np.array(data['metrics']['power_usage']),
            'battery_usage': np.array(data['metrics']['battery_usage']),
        }
        
        return storage