# PW DSL Bridge - Critical Architecture Fix Required

**Date**: 2025-10-05
**Severity**: HIGH - Core architecture not following design
**Status**: URGENT FIX NEEDED

---

## 🚨 THE PROBLEM

### What We Built (WRONG):
```
Python → IR → Go
```
**No PW DSL involved!** Direct language-to-language translation.

### What We Should Build (CORRECT):
```
Python → IR → PW DSL (text) → IR → Go
                ↑                    ↑
           Universal Bridge    Language Output
```

---

## Why This Matters

### Current System (Broken Architecture):
- ❌ **No universal bridge** - Each language pair needs its own translator
- ❌ **Agents can't communicate** - No PW DSL to exchange
- ❌ **Not MCP compliant** - MCP servers should exchange PW DSL
- ❌ **PW DSL tools unused** - Parser/generator exist but are bypassed
- ❌ **Not scalable** - N² problem (need Python→Go, Python→Rust, etc.)

### Correct System (AssertLang Vision):
- ✅ **PW DSL is universal** - All languages → PW DSL → All languages
- ✅ **Agents exchange PW** - Language-agnostic communication
- ✅ **MCP integration** - Servers send/receive PW DSL
- ✅ **Scalable** - Only N parsers + N generators needed
- ✅ **Human-readable** - Developers can inspect/edit PW DSL

---

## The Vision (From CLAUDE.md)

```
┌─────────────────────────────────────────────────────────────────┐
│              PROMPTWARE UNIVERSAL TRANSLATION SYSTEM             │
│                                                                  │
│  Python Code ──┐                                  ┌── Python    │
│  Node.js Code ─┤                                  ├── Node.js   │
│  Go Code ──────┼──► Parser ──► PW DSL ──► Gen ───┤── Go        │
│  Rust Code ────┤         (Bridge)                 ├── Rust      │
│  .NET Code ────┘                                  └── .NET      │
│                                                                  │
│  • PW DSL = Universal exchange format                           │
│  • Agents communicate in PW only                                │
│  • MCP servers use PW DSL                                       │
└──────────────────────────────────────────────────────────────────┘
```

**Key Quote from CLAUDE.md**:
> "Enable ANY code in ANY language to be translated to ANY other language
> through PW DSL as an intermediate representation."

---

## Current Issues Discovered

### 1. PW DSL Parser/Generator Incompatibility
- **Generator**: Uses `let` keyword for variable declarations
- **Parser**: Doesn't recognize `let` keyword (expects `var` or nothing)
- **Impact**: Can't round-trip Python → PW → Go

**Error**:
```
ALParseError: [Line 10:1] Unexpected keyword: let
```

**Generated PW DSL**:
```pw
let output = []              # ❌ Parser doesn't understand
let cx = (width / 2)         # ❌ Parser doesn't understand
```

### 2. Translation Tools Bypass PW DSL
- `translate_python_to_go.py` goes direct: Python → IR → Go
- PW DSL parser/generator exist but aren't in the pipeline
- Type inference works on IR, not PW DSL

### 3. No MCP Integration
- MCP servers should exchange PW DSL
- Current system doesn't expose PW DSL endpoints
- Agents can't communicate via PW

---

## Required Fixes

### Priority 1: Fix PW DSL Syntax (URGENT)
1. **Align parser and generator**:
   - Either: Update parser to recognize `let`
   - Or: Update generator to use parser-compatible syntax

2. **Test round-trip**:
   ```
   Python → IR → PW DSL → IR → Go
   ```
   Verify PW DSL parses correctly

3. **Validate all PW DSL features**:
   - Variables (`let` vs `var`)
   - Functions
   - Classes
   - Control flow
   - Types

### Priority 2: Update Translation Pipeline (HIGH)
1. **Modify all translation tools** to use PW DSL bridge:
   ```python
   # OLD (wrong)
   ir = parse_python(code)
   go_code = generate_go(ir)

   # NEW (correct)
   ir = parse_python(code)
   pw_dsl = generate_pw(ir)      # ← Add this
   ir2 = parse_al(pw_dsl)         # ← Add this
   go_code = generate_go(ir2)
   ```

2. **Save PW DSL to files** for inspection/debugging

3. **Add PW DSL validation** between steps

### Priority 3: MCP Integration (MEDIUM)
1. **Create PW DSL endpoints**:
   - `/parse` - Language → PW DSL
   - `/generate` - PW DSL → Language
   - `/translate` - PW DSL → PW DSL (transformations)

2. **Update MCP servers** to exchange PW DSL

3. **Agent communication** via PW DSL only

### Priority 4: Documentation (MEDIUM)
1. **PW DSL specification** - Complete, formal grammar
2. **Architecture diagrams** - Show PW DSL bridge clearly
3. **API documentation** - PW DSL endpoints
4. **Migration guide** - From direct translation to PW bridge

---

## Quick Fix Plan (This Session)

### Step 1: Fix PW DSL Syntax Mismatch
```bash
# Check what parser expects
grep -A 10 "def parse_variable" dsl/pw_parser.py

# Update generator to match
# OR update parser to accept 'let'
```

### Step 2: Test Round-Trip
```bash
# Use the correct pipeline
python3 translate_via_pw_dsl.py

# Verify PW DSL parses without errors
# Verify Go output is correct
```

### Step 3: Update Quality Metrics
- Measure: Python → PW → Go quality
- Compare: Direct (Python → Go) vs Bridge (Python → PW → Go)
- Document: Any quality differences

### Step 4: Document Findings
- Update CURRENT_WORK.md with PW DSL status
- Update CLAUDE.md with corrected architecture
- Create migration guide

---

## Success Criteria

### Phase 1: PW DSL Working
- ✅ Python → PW DSL → Go works without errors
- ✅ PW DSL is valid and parseable
- ✅ Quality >= 95% (same as direct translation)
- ✅ PW DSL files are human-readable

### Phase 2: Universal Bridge
- ✅ All languages use PW DSL bridge
- ✅ Agents can exchange PW DSL
- ✅ MCP servers use PW DSL
- ✅ N² → N complexity

### Phase 3: Production Ready
- ✅ Complete PW DSL specification
- ✅ Comprehensive tests
- ✅ API documentation
- ✅ Migration complete

---

## Impact Assessment

### What We Achieved (Still Valuable):
- ✅ 100% quality Python → Go translation
- ✅ Advanced type inference engine
- ✅ 4-phase inference algorithm
- ✅ Cross-scope type propagation
- ✅ Production-ready IR system

### What We Missed (Critical):
- ❌ PW DSL as universal bridge
- ❌ Agent communication via PW
- ❌ MCP integration
- ❌ Scalable architecture (N² → N)

### Recovery Plan:
1. **Keep the quality** - 100% translation works
2. **Add PW DSL bridge** - Insert between parser/generator
3. **Verify quality preserved** - Should be same/better
4. **Update all tools** - Use PW DSL pipeline
5. **Enable MCP** - Expose PW DSL endpoints

---

## Next Steps

1. **Fix PW DSL syntax** (1-2 hours)
2. **Test round-trip** (30 min)
3. **Update translation tools** (1 hour)
4. **Verify quality** (30 min)
5. **Document changes** (1 hour)

**Total**: ~4-5 hours to restore correct architecture

---

## Conclusion

**You were 100% correct** - we bypassed PW DSL entirely and built a direct Python→Go translator. While the quality is perfect (100%), it's not the scalable, universal system AssertLang was designed to be.

The fix is straightforward:
1. Align PW DSL parser/generator syntax
2. Insert PW DSL bridge in translation pipeline
3. Verify quality is preserved
4. Update all tools to use PW DSL

This will restore the original vision: **PW DSL as the universal bridge for all language translation**.

---

**Status**: Fix in progress - restoring correct architecture
**Priority**: URGENT - Core design principle
**Owner**: Current session
