import random
import unittest

import jax
import jax.numpy as jnp
import numpy as np
from jax.sharding import AxisType, Mesh, NamedSharding
from jax.sharding import PartitionSpec as P

from sgl_jax.srt.mem_cache.memory_pool import update_kv_cache
from sgl_jax.srt.utils.mesh_utils import create_device_mesh

mesh = create_device_mesh(ici_parallelism=[1, -1, 1, 1], dcn_parallelism=[1, 1, 1, 1])
jax.sharding.set_mesh(mesh)


class TestKVCache(unittest.TestCase):
    """Test cases for the KV Cache update functions."""

    def setUp(self):
        if not jax.devices():
            self.skipTest("JAX not available")

        self.max_seq_len = 16
        self.num_heads = 8
        self.head_dim = 128
        self.batch_size = 2
        self.layer_num = 2

    def generate_test_data(self, total_tokens: int, add_padding: bool = False):
        """Generate test data for KV cache update.

        Args:
            total_tokens: Total number of tokens (including padding)
            add_padding: Whether to add padding tokens (loc=-1)
        """
        # Create KV cache buffers
        cache_size = total_tokens + 100  # Add extra space for cache
        k_cache = jnp.zeros(
            (cache_size, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )
        v_cache = jnp.zeros(
            (cache_size, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )

        # Generate K and V tensors
        k = jax.random.uniform(
            jax.random.PRNGKey(42),
            (total_tokens, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )
        v = jax.random.uniform(
            jax.random.PRNGKey(43),
            (total_tokens, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )

        # Generate location indices
        if add_padding:
            # Create some padding tokens (-1) and some valid tokens
            num_padding = total_tokens // 4  # 25% padding
            valid_locs = (
                jnp.arange(total_tokens - num_padding, dtype=jnp.int32) + 10
            )  # Start from index 10
            padding_locs = jnp.full((num_padding,), -1, dtype=jnp.int32)

            # Shuffle to mix padding and valid tokens
            all_locs = jnp.concatenate([valid_locs, padding_locs])
            # Simple shuffle by reversing and interleaving
            loc = jnp.zeros(total_tokens, dtype=jnp.int32)
            loc = loc.at[::2].set(all_locs[: total_tokens // 2])
            loc = loc.at[1::2].set(
                all_locs[
                    total_tokens // 2 : total_tokens // 2
                    + (total_tokens - total_tokens // 2)
                ]
            )
        else:
            # All valid tokens
            loc = jnp.arange(total_tokens, dtype=jnp.int32) + 10  # Start from index 10

        k = jax.device_put(k, P(None, "tensor", None))
        v = jax.device_put(v, P(None, "tensor", None))
        k_cache = jax.device_put(k_cache, P(None, "tensor", None))
        v_cache = jax.device_put(v_cache, P(None, "tensor", None))
        loc = jax.device_put(loc, P(None))

        return k, v, loc, k_cache, v_cache

    def expected_update_kv_cache(self, k, v, loc, k_cache, v_cache):
        """Expected result using simple JAX operations."""
        expected_k_cache = k_cache.copy()
        expected_v_cache = v_cache.copy()

        # Update cache only for valid tokens (where loc != -1)
        for i in range(loc.shape[0]):
            if loc[i] != -1:
                expected_k_cache = expected_k_cache.at[loc[i]].set(k[i])
                expected_v_cache = expected_v_cache.at[loc[i]].set(v[i])

        return expected_k_cache, expected_v_cache

    def test_kv_cache_update_page_size_1(self):
        """Test KV cache update with page_size=1."""
        total_tokens = 16
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        updated_k_cache, updated_v_cache = update_kv_cache(
            k, v, loc, k_cache, v_cache, page_size=1
        )

        # Expected result
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))

    def test_kv_cache_update_page_size_1_with_padding(self):
        """Test KV cache update with page_size=1 and padding tokens."""
        total_tokens = 12
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=True
        )

        updated_k_cache, updated_v_cache = update_kv_cache(
            k, v, loc, k_cache, v_cache, page_size=1
        )

        # Expected result (should ignore padding tokens where loc == -1)
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))
        # Verify that padding tokens didn't affect the cache
        padding_mask = loc == -1
        if jnp.any(padding_mask):
            # Check that original cache values at padding positions are unchanged
            original_k_cache = jnp.zeros_like(k_cache)  # Original was all zeros
            original_v_cache = jnp.zeros_like(v_cache)

            # For positions that should be ignored (padding), cache should remain unchanged
            for i in range(total_tokens):
                if loc[i] == -1:
                    # Cache at this position should remain as original (zeros in this case)
                    continue  # We don't update cache for padding tokens

    def test_kv_cache_update_page_size_4(self):
        """Test KV cache update with page_size=4."""
        total_tokens = 16
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        # Test with page_size=4
        updated_k_cache, updated_v_cache = update_kv_cache(
            k, v, loc, k_cache, v_cache, page_size=4
        )

        # Expected result
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )
        print("updated_k_cache", updated_k_cache[loc])
        print("expected_k_cache", expected_k_cache[loc])
        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))

    def test_kv_cache_update_page_size_4_with_padding(self):
        """Test KV cache update with page_size=4 and padding tokens."""
        total_tokens = 12
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=True
        )

        # Test with page_size=4
        updated_k_cache, updated_v_cache = update_kv_cache(
            k, v, loc, k_cache, v_cache, page_size=4
        )

        # Expected result (should ignore padding tokens where loc == -1)
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))

    def test_kv_cache_update_page_size_8_contiguous(self):
        """Test KV cache update with page_size=8 and contiguous locations."""
        total_tokens = 16
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        # Test with page_size=8
        updated_k_cache, updated_v_cache = update_kv_cache(
            k, v, loc, k_cache, v_cache, page_size=8
        )

        # Expected result
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))

    def test_all_padding_tokens(self):
        """Test case where all tokens are padding tokens."""
        total_tokens = 4
        k, v, _, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        # Make all tokens padding
        loc = jnp.full((total_tokens,), -1, dtype=jnp.int32)

        # Store original cache
        original_k_cache = k_cache.copy()
        original_v_cache = v_cache.copy()

        # Test both approaches
        updated_k_cache, updated_v_cache = update_kv_cache(k, v, loc, k_cache, v_cache)

        # Cache should remain unchanged since all tokens are padding
        self.assertTrue(jnp.allclose(updated_k_cache, original_k_cache))
        self.assertTrue(jnp.allclose(updated_v_cache, original_v_cache))

    def test_update_kv_cache_logic_page_size_1(self):
        """Test KV cache update logic with page_size=1 using optimization functions."""
        from sgl_jax.srt.mem_cache.memory_pool import _optimize_contiguous_updates

        total_tokens = 8
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        # Test the optimization logic for page_size=1
        page_size = 1
        if page_size > 1:
            kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
                _optimize_contiguous_updates(loc, page_size)
            )
        else:
            # Use original logic for page_size = 1: one slice per token
            kv_cache_locs = jnp.where(loc == -1, 0, loc).astype(jnp.int32)
            new_kv_locs = jnp.arange(total_tokens, dtype=jnp.int32)
            slice_lens = jnp.where(loc == -1, 0, 1).astype(jnp.int32)
            num_slices = total_tokens

        # Verify the slice logic
        self.assertEqual(num_slices, total_tokens)
        # For page_size=1, each token should have its own slice
        non_padding_count = jnp.sum(loc != -1)
        expected_valid_slices = jnp.sum(slice_lens > 0)
        self.assertEqual(expected_valid_slices, non_padding_count)

        # Manual update using the slice logic
        updated_k_cache = k_cache.copy()
        updated_v_cache = v_cache.copy()

        for i in range(num_slices):
            if slice_lens[i] > 0:  # Valid slice
                cache_start = kv_cache_locs[i]
                new_start = new_kv_locs[i]
                length = slice_lens[i]

                # Update cache
                for j in range(length):
                    updated_k_cache = updated_k_cache.at[cache_start + j].set(
                        k[new_start + j]
                    )
                    updated_v_cache = updated_v_cache.at[cache_start + j].set(
                        v[new_start + j]
                    )

        # Expected result
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache, rtol=1e-5))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache, rtol=1e-5))

    def test_update_kv_cache_logic_page_size_4(self):
        """Test KV cache update logic with page_size=4 using optimization functions."""
        from sgl_jax.srt.mem_cache.memory_pool import _optimize_contiguous_updates

        total_tokens = 16
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=False
        )

        # Test the optimization logic for page_size=4
        page_size = 4
        kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
            _optimize_contiguous_updates(loc, page_size)
        )

        # Verify the slice logic makes sense
        # num_slices always equals total_tokens for array size consistency,
        # but only some positions will have non-zero slice_lens
        self.assertEqual(num_slices, total_tokens)

        # Count total tokens that should be processed
        total_processed = jnp.sum(slice_lens)
        non_padding_count = jnp.sum(loc != -1)
        # After fix: all non-padding tokens should be processed
        self.assertEqual(total_processed, non_padding_count)

        # For contiguous tokens with page_size=4, expect 4 slices of length 4 each
        actual_slices = [
            (i, slice_lens[i]) for i in range(num_slices) if slice_lens[i] > 0
        ]
        expected_slices = [(0, 4), (4, 4), (8, 4), (12, 4)]
        self.assertEqual(len(actual_slices), len(expected_slices))
        for (actual_i, actual_len), (expected_i, expected_len) in zip(
            actual_slices, expected_slices
        ):
            self.assertEqual(actual_i, expected_i)
            self.assertEqual(actual_len, expected_len)

        # Manual update using the slice logic
        updated_k_cache = k_cache.copy()
        updated_v_cache = v_cache.copy()

        for i in range(num_slices):
            if slice_lens[i] > 0:  # Valid slice
                cache_start = kv_cache_locs[i]
                new_start = new_kv_locs[i]
                length = slice_lens[i]

                # Update cache
                for j in range(length):
                    updated_k_cache = updated_k_cache.at[cache_start + j].set(
                        k[new_start + j]
                    )
                    updated_v_cache = updated_v_cache.at[cache_start + j].set(
                        v[new_start + j]
                    )

        # For this test, we'll verify that the processed tokens are correctly updated
        # rather than expecting all tokens to be processed due to the optimization bug
        for i in range(num_slices):
            if slice_lens[i] > 0:  # Valid slice that was processed
                cache_start = kv_cache_locs[i]
                new_start = new_kv_locs[i]
                length = slice_lens[i]

                # Verify that these tokens were updated correctly
                for j in range(length):
                    expected_k_val = k[new_start + j]
                    expected_v_val = v[new_start + j]
                    actual_k_val = updated_k_cache[cache_start + j]
                    actual_v_val = updated_v_cache[cache_start + j]

                    self.assertTrue(
                        jnp.allclose(actual_k_val, expected_k_val, rtol=1e-5)
                    )
                    self.assertTrue(
                        jnp.allclose(actual_v_val, expected_v_val, rtol=1e-5)
                    )

    def test_update_kv_cache_logic_page_size_8_with_padding(self):
        """Test KV cache update logic with page_size=8 and padding using optimization functions."""
        from sgl_jax.srt.mem_cache.memory_pool import _optimize_contiguous_updates

        total_tokens = 20
        k, v, loc, k_cache, v_cache = self.generate_test_data(
            total_tokens, add_padding=True
        )

        # Test the optimization logic for page_size=8 with padding
        page_size = 8
        kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
            _optimize_contiguous_updates(loc, page_size)
        )

        # Verify the slice logic handles padding correctly
        self.assertEqual(num_slices, total_tokens)

        # Count total tokens that should be processed (excluding padding)
        total_processed = jnp.sum(slice_lens)
        non_padding_count = jnp.sum(loc != -1)
        self.assertEqual(total_processed, non_padding_count)

        # Manual update using the slice logic
        updated_k_cache = k_cache.copy()
        updated_v_cache = v_cache.copy()

        for i in range(num_slices):
            if slice_lens[i] > 0:  # Valid slice
                cache_start = kv_cache_locs[i]
                new_start = new_kv_locs[i]
                length = slice_lens[i]

                # Update cache
                for j in range(length):
                    updated_k_cache = updated_k_cache.at[cache_start + j].set(
                        k[new_start + j]
                    )
                    updated_v_cache = updated_v_cache.at[cache_start + j].set(
                        v[new_start + j]
                    )

        # Expected result (should ignore padding tokens where loc == -1)
        expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
            k, v, loc, k_cache, v_cache
        )

        self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache, rtol=1e-5))
        self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache, rtol=1e-5))

    def test_kv_cache_update_multiple_segments_with_padding(self):
        """Test KV cache update with multiple contiguous segments of different lengths and padding."""
        # Corner case: multiple segments with varying lengths and padding
        # Segments: [11-17] (7 tokens), [22-25] (4 tokens), [30-39] (10 tokens), then padding
        total_tokens = 25

        # Create location array with multiple segments and padding
        loc = jnp.full((total_tokens,), -1, dtype=jnp.int32)

        # Segment 1: positions 0-6 -> cache locations 11-17 (7 tokens)
        loc = loc.at[0:7].set(jnp.arange(11, 18))

        # Segment 2: positions 7-10 -> cache locations 22-25 (4 tokens)
        loc = loc.at[7:11].set(jnp.arange(22, 26))

        # Segment 3: positions 11-20 -> cache locations 30-39 (10 tokens)
        loc = loc.at[11:21].set(jnp.arange(30, 40))

        # Positions 21-24 remain as padding (-1)

        print(f"Corner case loc = {loc}")

        # Generate test data
        k = jax.random.uniform(
            jax.random.PRNGKey(42),
            (total_tokens, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )
        v = jax.random.uniform(
            jax.random.PRNGKey(43),
            (total_tokens, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )

        cache_size = total_tokens + 50
        k_cache = jnp.zeros(
            (cache_size, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )
        v_cache = jnp.zeros(
            (cache_size, self.num_heads, self.head_dim),
            dtype=jnp.bfloat16,
        )

        # Test with different page sizes
        for page_size in [1, 2, 4, 8]:
            with self.subTest(page_size=page_size):
                print(f"\nTesting page_size={page_size}")

                updated_k_cache, updated_v_cache = update_kv_cache(
                    k, v, loc, k_cache, v_cache, page_size=page_size
                )

                # Expected result
                expected_k_cache, expected_v_cache = self.expected_update_kv_cache(
                    k, v, loc, k_cache, v_cache
                )

                self.assertTrue(jnp.allclose(updated_k_cache, expected_k_cache))
                self.assertTrue(jnp.allclose(updated_v_cache, expected_v_cache))

                # Verify specific segments are updated correctly
                # Segment 1: cache locations 11-17
                for i in range(7):
                    cache_pos = 11 + i
                    input_pos = i
                    self.assertTrue(
                        jnp.allclose(
                            updated_k_cache[cache_pos], k[input_pos], rtol=1e-5
                        )
                    )

                # Segment 2: cache locations 22-25
                for i in range(4):
                    cache_pos = 22 + i
                    input_pos = 7 + i
                    self.assertTrue(
                        jnp.allclose(
                            updated_k_cache[cache_pos], k[input_pos], rtol=1e-5
                        )
                    )

                # Segment 3: cache locations 30-39
                for i in range(10):
                    cache_pos = 30 + i
                    input_pos = 11 + i
                    self.assertTrue(
                        jnp.allclose(
                            updated_k_cache[cache_pos], k[input_pos], rtol=1e-5
                        )
                    )

                print(f"  ✓ page_size={page_size} passed")

    def test_optimize_contiguous_updates_corner_cases(self):
        """Test _optimize_contiguous_updates with various corner cases."""
        from sgl_jax.srt.mem_cache.memory_pool import _optimize_contiguous_updates

        # Test case 1: Multiple segments with different lengths
        print("\nTest case 1: Multiple segments")
        total_tokens = 25
        loc = jnp.full((total_tokens,), -1, dtype=jnp.int32)

        # Segment 1: [11-17] (7 tokens)
        loc = loc.at[0:7].set(jnp.arange(11, 18))
        # Segment 2: [22-25] (4 tokens)
        loc = loc.at[7:11].set(jnp.arange(22, 26))
        # Segment 3: [30-39] (10 tokens)
        loc = loc.at[11:21].set(jnp.arange(30, 40))
        # Padding: positions 21-24 are -1

        print(f"loc = {loc}")

        for page_size in [1, 2, 4, 8]:
            with self.subTest(case=1, page_size=page_size):
                kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
                    _optimize_contiguous_updates(loc, page_size)
                )

                # Verify all valid tokens are processed
                total_processed = jnp.sum(slice_lens)
                valid_tokens = jnp.sum(loc != -1)
                self.assertEqual(
                    total_processed,
                    valid_tokens,
                    f"page_size={page_size}: processed {total_processed}, expected {valid_tokens}",
                )

                # Verify slice starts and lengths make sense
                slices = [
                    (i, kv_cache_locs[i], new_kv_locs[i], slice_lens[i])
                    for i in range(num_slices)
                    if slice_lens[i] > 0
                ]

                print(
                    f"  page_size={page_size}: {len(slices)} slices, {total_processed} tokens processed"
                )
                for idx, cache_start, new_start, length in slices:
                    print(
                        f"    Slice at pos {idx}: cache[{cache_start}:{cache_start+length}] <- input[{new_start}:{new_start+length}]"
                    )

                    # Verify slice doesn't exceed page_size
                    self.assertLessEqual(length, page_size)

                    # Verify cache locations are contiguous
                    for j in range(length):
                        self.assertEqual(loc[new_start + j], cache_start + j)

        # Test case 2: Single token segments with gaps
        print("\nTest case 2: Single token segments with gaps")
        loc2 = jnp.array([10, -1, -1, 15, -1, 20, 21, -1, -1, 25], dtype=jnp.int32)
        print(f"loc = {loc2}")

        for page_size in [1, 2, 4]:
            with self.subTest(case=2, page_size=page_size):
                kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
                    _optimize_contiguous_updates(loc2, page_size)
                )

                total_processed = jnp.sum(slice_lens)
                valid_tokens = jnp.sum(loc2 != -1)
                self.assertEqual(total_processed, valid_tokens)

                slices = [
                    (i, kv_cache_locs[i], new_kv_locs[i], slice_lens[i])
                    for i in range(num_slices)
                    if slice_lens[i] > 0
                ]
                print(
                    f"  page_size={page_size}: {len(slices)} slices, {total_processed} tokens processed"
                )

        # Test case 3: All padding
        print("\nTest case 3: All padding tokens")
        loc3 = jnp.full((10,), -1, dtype=jnp.int32)

        for page_size in [1, 4]:
            with self.subTest(case=3, page_size=page_size):
                kv_cache_locs, new_kv_locs, slice_lens, num_slices = (
                    _optimize_contiguous_updates(loc3, page_size)
                )

                total_processed = jnp.sum(slice_lens)
                self.assertEqual(
                    total_processed, 0, "Should process 0 tokens when all are padding"
                )

                # No slices should be created
                slices = [i for i in range(num_slices) if slice_lens[i] > 0]
                self.assertEqual(
                    len(slices), 0, "No slices should be created for all-padding input"
                )


if __name__ == "__main__":
    unittest.main()
