#!/usr/bin/env bash
# scripts/prune_legacy.sh
# Remove legacy Graph/Subgraph duplication and error branches.
set -euo pipefail

echo "==> Pruning legacy Graph/Subgraph duplication"

# 1) Remove GraphTable's 'EnhancedSubgraph needs graph reference' branches (if any remain)
if rg -n "EnhancedSubgraph needs graph reference" -g "python-groggy/**" >/dev/null 2>&1; then
  rg -l "EnhancedSubgraph needs graph reference" -g "python-groggy/**" | xargs -I{} sed -i.bak "/EnhancedSubgraph needs graph reference/d" {}
  echo "Removed legacy EnhancedSubgraph error branches in python-groggy/"
fi

# 2) Remove Graph-only column accessors (if present)
if rg -n "fn\s+_get_node_attribute_column" -g "python-groggy/**" >/dev/null 2>&1; then
  rg -l "fn\s+_get_node_attribute_column" -g "python-groggy/**" | while read -r f; do
    sed -i.bak "/fn\s\+_get_node_attribute_column/,/^[}]\s*$/d" "$f" || true
    echo "Pruned _get_node_attribute_column in $f"
  done
fi
if rg -n "fn\s+_get_edge_attribute_column" -g "python-groggy/**" >/dev/null 2>&1; then
  rg -l "fn\s+_get_edge_attribute_column" -g "python-groggy/**" | while read -r f; do
    sed -i.bak "/fn\s\+_get_edge_attribute_column/,/^[}]\s*$/d" "$f" || true
    echo "Pruned _get_edge_attribute_column in $f"
  done
fi

# 3) Remove Graph-only traversal/analytics implementations (keep forwards only)
if rg -n "impl\s+PyGraph[\s\S]*?(bfs|dfs|connected_components|filter_nodes|filter_edges)" -g "python-groggy/**" >/dev/null 2>&1; then
  echo "NOTE: Review PyGraph methods for traversal/analytics and replace with forwards to view()."
  rg -n "impl\s+PyGraph[\s\S]*?(bfs|dfs|connected_components|filter_nodes|filter_edges)" -g "python-groggy/**" || true
fi

# 4) Remove legacy subgraph constructors without a graph reference
if rg -n "EnhancedSubgraph|legacy|new\(nodes,.*edges" -g "python-groggy/**" >/dev/null 2>&1; then
  rg -l "EnhancedSubgraph|legacy|new\(nodes,.*edges" -g "python-groggy/**" | while read -r f; do
    echo "Inspect legacy constructor in $f"
  done
fi

# 5) Clean backup files generated by sed
find . -name "*.bak" -delete

echo "==> Done. Please review 'git diff' and run tests."