# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['equi7grid_lite']

package_data = \
{'': ['*'], 'equi7grid_lite': ['data/*']}

install_requires = \
['geopandas>=0.13.2',
 'nptyping>=2.5.0',
 'numpy>=1.25.2',
 'pandas>=2.0.3',
 'shapely>=2.0.4']

setup_kwargs = {
    'name': 'equi7grid-lite',
    'version': '0.4.0',
    'description': 'A user-friendly Python interface to interact with the Equi7Grid grid system',
    'long_description': '# equi7grid-lite\n\n<p align="center">\n  <a href="https://ipl-uv.github.io"><img src="docs/logo.jpeg" alt="header" width="50%"></a>\n</p>\n\n<p align="center">\n    <em>\n    No one will drive us from the paradise which EquiGrid created for us\n    </em>\n</p>\n\n<p align="center">\n<a href=\'https://pypi.python.org/pypi/equi7grid-lite\'>\n<img src=\'https://img.shields.io/pypi/v/equi7grid-lite.svg\' alt=\'PyPI\' />\n</a>\n<a href=\'https://colab.research.google.com/drive/1SBjl4GVgCFUpVch2Prju5oiXN8WyzZTi?usp=sharing\'>\n<img src=\'https://colab.research.google.com/assets/colab-badge.svg\' alt=\'COLAB\' />\n</a>\n<a href="https://opensource.org/licenses/MIT" target="_blank">\n<img src="https://img.shields.io/badge/License-MIT-blue.svg" alt="License">\n</a>\n<a href="https://github.com/psf/black" target="_blank">\n<img src="https://img.shields.io/badge/code%20style-black-000000.svg" alt="Black">\n</a>\n<a href="https://pycqa.github.io/isort/" target="_blank">\n<img src="https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336" alt="isort">\n</a>\n</p>\n\nThe `equi7grid-lite` package implements a user-friendly Python interface to interact with the [**Equi7Grid**](https://github.com/TUW-GEO/Equi7Grid) grid system. \n\n`equi7grid-lite` is an **unofficial Python implementation of [**Equi7Grid**](https://github.com/TUW-GEO/Equi7Grid)**. With this package, users can convert geographic coordinates to Equi7Grid tiles and vice versa. This implementation differs from the official version in tree key ways:\n\n- *Quad-Tree Grid Splitting*: Users are required to split the grid in a Quad-Tree fashion, meaning each grid level is divided into four tiles. For example, transitioning from level 1 to level 0 involves splitting each tile into four regular smaller tiles.\n\n- *Revised Grid ID Encoding*: The grid ID is always encoded in meters, and the reference to the tile system (e.g., "T1", "T3", "T6") is removed. Instead, tiles are dynamically defined by the `min_grid_size` parameter. Here is a comparison between the original Equi7Grid and equi7grid-lite:\n\n    - \'EU500M_E036N006T6\' -> \'EU2560_E4521N3011\'\n\n    Where \'EU\' and \'AF\' are the zones, \'2560\' is the `min_grid_size`, \'E4521\' is the position in the *x* tile axis, and \'N3011\' is the position in the *y* tile axis.\n\n- *Upper Bound Level*: The maximum grid level is determined as the nearest lower distance to 2_500_000 meters. This threshold serves as a limit to create the Quad-Tree grid structure.\n\n\n<p align="center">\n  <img src="docs/equi7grid_tiling.gif" alt="equi7grid-lite" width="100%"/>\n</p>\n\nPlease refer to the [Equi7Grid repository](https://github.com/TUW-GEO/Equi7Grid) for **more information of the official implementation**.\n\n## Installation\n\nThe `equi7grid-lite` package is available on PyPI and can be installed using `pip`:\n\n```python\npip install equi7grid-lite\n```\n\n## Usage\n\nThe `equi7grid-lite` package provides a single class, `Equi7Grid`, which can be used to convert between geographic coordinates and Equi7Grid tiles.\n\n```python\nfrom equi7grid_lite import Equi7Grid\n\ngrid_system = Equi7Grid(min_grid_size=2560)\n# Equi7Grid(min_grid_size=2560)\n# ----------------\n# levels: 0, 1, ... , 7, 8\n# zones: AN, NA, OC, SA, AF, EU, AS\n# min_grid_size: 2560 meters\n# max_grid_size: 1310720 meters\n```\n\nTo convert between geographic coordinates and Equi7Grid tiles, use the `lonlat2grid` and `grid2lonlat` methods.\n\n```python\nlon, lat = -79.5, -5.49\ngrid_system.lonlat2grid(lon=lon, lat=lat)\n# id\tx\ty\tzone\tlevel\tgeometry\n# 0\tSA2560_E2008N2524\t5140480.0\t6461440.0\tSA\tZ1\tPOLYGON ((5145600.000 6461440.000, 5145600.000...\n\ngrid_system.grid2lonlat(grid_id="SA2560_E2008N2524")\n#      lon       lat        x        y\n# 0 -79.543717 -5.517556  5140480  6461440\n```\n\nFor users who want to align coordinates to the Equi7Grid Quad-Tree structure, the `align2grid` method is available.\n\n```python\ngrid_system.align2grid(lon=lon, lat=lat, level=5)\n#         lon       lat\n#0 -80.116158 -6.105519\n```\n\nThe `Equi7Grid` class also provides a method for creating a grid of Equi7Grid tiles that cover a given bounding box.\n\n```python\nimport geopandas as gpd\nfrom equi7grid_lite import Equi7Grid\n\n# Define a POLYGON geometry\nworld_filepath = gpd.datasets.get_path(\'naturalearth_lowres\')\nworld = gpd.read_file(world_filepath)\ncountry = world[world.name == "Peru"].geometry.values[0]\n\n# Create a grid of Equi7Grid tiles that cover the bounding box of the POLYGON geometry\ngrid = grid_system.create_grid(\n    level=4,\n    zone="SA",\n    mask=country, # Only include tiles that intersect the polygon\n    coverland=True # Only include tiles with landmasses    \n)\n\n# Export the grid to a GeoDataFrame\ngrid.to_file("grid.shp")\n```\n\nBy running `create_grid` with different levels, you can obtain its corresponding Equi7Grid Quad-Tree grid structure for any region.\n\n![grid](docs/equi7grid_demo.gif)\n\nObtain the metadata of each Equi7Grid zone:\n\n```python\nfrom equi7grid_lite import Equi7Grid\n\n# Zones: SA, EU, AF, AS, NA, AU\nEqui7Grid.SA\n```\n\nEach zone has the following attributes:\n\n- *id*: The zone ID code.\n- *crs*: The WKT representation of the CRS.\n- *geometry_geo*: The geometry of the zone in EPSG:4326.\n- *geometry_equi7grid*: The geometry of the zone in the Equi7Grid CRS.\n- *bbox_geo*: The bounding box of the zone in EPSG:4326.\n- *bbox_equi7grid*: The bounding box of the zone in the Equi7Grid CRS.\n- *landmasses_equi7grid*: The landmasses of the zone in the Equi7Grid CRS.\n- *origin*: The central meridian and the latitude of origin.\n\n## License\n\nThis package is released under the MIT License. For more information, see the [LICENSE](LICENSE) file.\n\n## Contributing\n\nContributions are welcome! For bug reports or feature requests, please open an issue on GitHub. For contributions, please submit a pull request with a detailed description of the changes.\n\n## Citation\n\nThis is a simple adaptation of the Equi7Grid paper and code. If you use this package in your research, please consider citing the original Equi7Grid package and paper.\n\n**Package:**\n\n```\n@software{bernhard_bm_2023_8252376,\n  author       = {Bernhard BM and\n                  Sebastian Hahn and\n                  actions-user and\n                  cnavacch and\n                  Manuel Schmitzer and\n                  shochsto and\n                  Senmao Cao},\n  title        = {TUW-GEO/Equi7Grid: v0.2.4},\n  month        = aug,\n  year         = 2023,\n  publisher    = {Zenodo},\n  version      = {v0.2.4},\n  doi          = {10.5281/zenodo.8252376},\n  url          = {https://doi.org/10.5281/zenodo.8252376}\n}\n```\n\n**Paper:**\n\n```\n@article{BAUERMARSCHALLINGER201484,\ntitle = {Optimisation of global grids for high-resolution remote sensing data},\njournal = {Computers & Geosciences},\nvolume = {72},\npages = {84-93},\nyear = {2014},\nissn = {0098-3004},\ndoi = {https://doi.org/10.1016/j.cageo.2014.07.005},\nurl = {https://www.sciencedirect.com/science/article/pii/S0098300414001629},\nauthor = {Bernhard Bauer-Marschallinger and Daniel Sabel and Wolfgang Wagner},\nkeywords = {Remote sensing, High resolution, Big data, Global grid, Projection, Sampling, Equi7 Grid}\n}\n```\n',
    'author': 'Cesar Aybar',
    'author_email': 'cesar.aybar@uv.es',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/csaybar/equi7grid-lite',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
