import subprocess
import json
import PIL

from typing import List, Dict, Any, Optional, Union


from jinja2 import Environment, FileSystemLoader, Undefined

from sqlalchemy import create_engine

from npcpy.npc_sysenv import (
    render_markdown,
    lookup_provider,
    request_user_input, 
    get_system_message
)
from npcpy.gen.response import get_litellm_response
from npcpy.gen.image_gen import generate_image, edit_image
from npcpy.gen.video_gen import generate_video_diffusers

def gen_image(
    prompt: str,
    model: str = None,
    provider: str = None,
    npc: Any = None,
    height: int = 1024,
    width: int = 1024,
    input_images: List[Union[str, bytes, PIL.Image.Image]] = None,
):
    """This function generates an image using the specified provider and model.
    Args:
        prompt (str): The prompt for generating the image.
    Keyword Args:
        model (str): The model to use for generating the image.
        provider (str): The provider to use for generating the image.
        filename (str): The filename to save the image to.
        npc (Any): The NPC object.
    Returns:
        str: The filename of the saved image.
    """
    if model is not None and provider is not None:
        pass
    elif model is not None and provider is None:
        provider = lookup_provider(model)
    elif npc is not None:
        if npc.provider is not None:
            provider = npc.provider
        if npc.model is not None:
            model = npc.model
        if npc.api_url is not None:
            api_url = npc.api_url

    image = generate_image(
        prompt=prompt,
        model=model,
        provider=provider,
        height=height,
        width=width, 
        attachments=input_images,
        
    )
    return image



def gen_video(
    prompt,
    model: str = None,
    provider: str = None,
    npc: Any = None,
    device: str = "cpu",
    output_path="",
    num_inference_steps=10,
    num_frames=25,
    height=256,
    width=256,
    messages: list = None,
):
    """
    Function Description:
        This function generates a video using the Stable Diffusion API.
    Args:
        prompt (str): The prompt for generating the video.
        model_id (str): The Hugging Face model ID to use for Stable Diffusion.
        device (str): The device to run the model on ('cpu' or 'cuda').
    Returns:
        PIL.Image: The generated image.
    """
    output_path = generate_video_diffusers(
        prompt,
        model,
        npc=npc,
        device=device,
        output_path=output_path,
        num_inference_steps=num_inference_steps,
        num_frames=num_frames,
        height=height,
        width=width,
    )
    if provider == "diffusers":
        return {"output": "output path at " + output_path, "messages": messages}





def get_llm_response(
    prompt: str,
    model: str=None,
    provider: str = None,
    images: List[str] = None,
    npc: Any = None,
    team: Any = None,
    messages: List[Dict[str, str]] = None,
    api_url: str = None,
    api_key: str = None,
    context=None,    
    stream: bool = False,
    attachments: List[str] = None,
    **kwargs,
):
    """This function generates a response using the specified provider and model.
    Args:
        prompt (str): The prompt for generating the response.
    Keyword Args:
        provider (str): The provider to use for generating the response.
        model (str): The model to use for generating the response.
        images (List[Dict[str, str]]): The list of images.
        npc (Any): The NPC object.
        messages (List[Dict[str, str]]): The list of messages.
        api_url (str): The URL of the API endpoint.
        attachments (List[str]): List of file paths to include as attachments
    Returns:
        Any: The response generated by the specified provider and model.
    """
    # Determine provider and model from NPC if needed
    if model is not None and provider is not None:
        pass
    elif provider is None and model is not None:
        provider = lookup_provider(model)
    elif npc is not None:
        if npc.provider is not None:
            provider = npc.provider
        if npc.model is not None:
            model = npc.model
        if npc.api_url is not None:
            api_url = npc.api_url
    else:
        provider = "ollama"
        if images is not None or attachments is not None:
            model = "llava:7b"
        else:
            model = "llama3.2"
            
    system_message = get_system_message(npc) if npc else "You are a helpful assistant."

    if context is not None:
        context_str = f'User Provided Context: {context}'
    else:
        context_str = ''

    if messages is None or len(messages) == 0:
        messages = [{"role": "system", "content": system_message}]
        if prompt:
            messages.append({"role": "user", "content": prompt+context_str})
    elif prompt and messages[-1]["role"] == "user":

        if isinstance(messages[-1]["content"], str):
            messages[-1]["content"] += "\n" + prompt+context_str
    elif prompt:
        messages.append({"role": "user", "content": prompt + context_str})

    response = get_litellm_response(
        prompt + context_str,
        messages=messages,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        images=images,
        attachments=attachments,
        stream=stream,
        **kwargs,
    )
    return response




def execute_llm_command(
    command: str,
    model: Optional[str] = None,
    provider: Optional[str] = None,
    api_url: str = None,
    api_key: str = None,
    npc: Optional[Any] = None,
    messages: Optional[List[Dict[str, str]]] = None,
    stream=False,
    context=None,
) -> str:
    """This function executes an LLM command.
    Args:
        command (str): The command to execute.

    Keyword Args:
        model (Optional[str]): The model to use for executing the command.
        provider (Optional[str]): The provider to use for executing the command.
        npc (Optional[Any]): The NPC object.
        messages (Optional[List[Dict[str, str]]): The list of messages.
    Returns:
        str: The result of the LLM command.
    """
    if messages is None:
        messages = []
    max_attempts = 5
    attempt = 0
    subcommands = []

    # Create context from retrieved documents
    context = ""
    while attempt < max_attempts:
        prompt = f"""
        A user submitted this query: {command}.
        You need to generate a bash command that will accomplish the user's intent.
        Respond ONLY with the bash command that should be executed. 
        Do not include markdown formatting
        """
        response = get_llm_response(
            prompt,
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            messages=messages,
            npc=npc,
            context=context,
        )

        bash_command = response.get("response", {})
 
        print(f"LLM suggests the following bash command: {bash_command}")
        subcommands.append(bash_command)

        try:
            print(f"Running command: {bash_command}")
            result = subprocess.run(
                bash_command, shell=True, text=True, capture_output=True, check=True
            )
            print(f"Command executed with output: {result.stdout}")

            prompt = f"""
                Here was the output of the result for the {command} inquiry
                which ran this bash command {bash_command}:

                {result.stdout}

                Provide a simple response to the user that explains to them
                what you did and how it accomplishes what they asked for.
                """

            messages.append({"role": "user", "content": prompt})
            # print(messages, stream)
            response = get_llm_response(
                prompt,
                model=model,
                provider=provider,
                api_url=api_url,
                api_key=api_key,
                npc=npc,
                messages=messages,
                context=context,
                stream =stream
            )

            return response
        except subprocess.CalledProcessError as e:
            print(f"Command failed with error:")
            print(e.stderr)

            error_prompt = f"""
            The command '{bash_command}' failed with the following error:
            {e.stderr}
            Please suggest a fix or an alternative command.
            Respond with a JSON object containing the key "bash_command" with the suggested command.
            Do not include any additional markdown formatting.

            """

            fix_suggestion = get_llm_response(
                error_prompt,
                model=model,
                provider=provider,
                npc=npc,
                api_url=api_url,
                api_key=api_key,
                format="json",
                messages=messages,
                context=context,
            )

            fix_suggestion_response = fix_suggestion.get("response", {})

            try:
                if isinstance(fix_suggestion_response, str):
                    fix_suggestion_response = json.loads(fix_suggestion_response)

                if (
                    isinstance(fix_suggestion_response, dict)
                    and "bash_command" in fix_suggestion_response
                ):
                    print(
                        f"LLM suggests fix: {fix_suggestion_response['bash_command']}"
                    )
                    command = fix_suggestion_response["bash_command"]
                else:
                    raise ValueError(
                        "Invalid response format from LLM for fix suggestion"
                    )
            except (json.JSONDecodeError, ValueError) as e:
                print(f"Error parsing LLM fix suggestion: {e}")

        attempt += 1

    return {
        "messages": messages,
        "output": "Max attempts reached. Unable to execute the command successfully.",
    }

def handle_jinx_call(
    command: str,
    jinx_name: str,
    model: str = None,
    provider: str = None,
    api_url: str = None,
    api_key: str = None,
    messages: List[Dict[str, str]] = None,
    npc: Any = None,
    stream=False,
    shell=False,
    n_attempts=3,
    attempt=0,
    context=None,
) -> Union[str, Dict[str, Any]]:
    """This function handles a jinx call.
    Args:
        command (str): The command.
        jinx_name (str): The jinx name.
    Keyword Args:
        model (str): The model to use for handling the jinx call.
        provider (str): The provider to use for handling the jinx call.
        messages (List[Dict[str, str]]): The list of messages.
        npc (Any): The NPC object.
    Returns:
        Union[str, Dict[str, Any]]: The result of handling
        the jinx call.

    """
    if npc is None:
        return f"No jinxs are available. "
    else:
        if jinx_name not in npc.jinxs_dict:
            print("not available")
            print(f"jinx '{jinx_name}' not found in NPC's jinxs_dict.")
            return f"jinx '{jinx_name}' not found."
        elif jinx_name in npc.jinxs_dict:
            jinx = npc.jinxs_dict[jinx_name]
        render_markdown(f"jinx found: {jinx.jinx_name}")
        jinja_env = Environment(loader=FileSystemLoader("."), undefined=Undefined)
        example_format = {}
        for inp in jinx.inputs:
            if isinstance(inp, str):
                example_format[inp] = f"<value for {inp}>"
            elif isinstance(inp, dict):
                key = list(inp.keys())[0]
                example_format[key] = f"<value for {key}>"
        
        json_format_str = json.dumps(example_format, indent=4)
        

        prompt = f"""
        The user wants to use the jinx '{jinx_name}' with the following request:
        '{command}'
        Here is the jinx file:
        ```
        {jinx.to_dict()}
        ```

        Please determine the required inputs for the jinx as a JSON object.
        
        They must be exactly as they are named in the jinx.
        For example, if the jinx has three inputs, you should respond with a list of three values that will pass for those args.
        
        If the jinx requires a file path, you must include an absolute path to the file including an extension.
        If the jinx requires code to be generated, you must generate it exactly according to the instructions.
        Your inputs must satisfy the jinx's requirements.

        Return only the JSON object without any markdown formatting.

        The format of the JSON object is:
        {{
        {json_format_str}
        }}
        """

        if npc and hasattr(npc, "shared_context"):
            if npc.shared_context.get("dataframes"):
                context_info = "\nAvailable dataframes:\n"
                for df_name in npc.shared_context["dataframes"].keys():
                    context_info += f"- {df_name}\n"
                prompt += f"""Here is contextual info that may affect your choice: {context_info}
                """
        response = get_llm_response(
            prompt,
            format="json",
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            npc=npc,
            context=context
        )

        try:
            response_text = response.get("response", "{}")
            if isinstance(response_text, str):
                response_text = (
                    response_text.replace("```json", "").replace("```", "").strip()
                )

            # Parse the cleaned response
            if isinstance(response_text, dict):
                input_values = response_text
            else:
                input_values = json.loads(response_text)
            # print(f"Extracted inputs: {input_values}")
        except json.JSONDecodeError as e:
            print(f"Error decoding input values: {e}. Raw response: {response}")
            return f"Error extracting inputs for jinx '{jinx_name}'"
        # Input validation (example):
        required_inputs = jinx.inputs
        missing_inputs = []
        for inp in required_inputs:
            if not isinstance(inp, dict):
                # dicts contain the keywords so its fine if theyre missing from the inputs.
                if inp not in input_values or input_values[inp] == "":
                    missing_inputs.append(inp)
        if len(missing_inputs) > 0:
            # print(f"Missing required inputs for jinx '{jinx_name}': {missing_inputs}")
            if attempt < n_attempts:
                print(f"attempt {attempt+1} to generate inputs failed, trying again")
                print("missing inputs", missing_inputs)
                print("llm response", response)
                print("input values", input_values)
                return handle_jinx_call(
                    command +' . In the previous attempt, the inputs were: ' + str(input_values) + ' and the missing inputs were: ' + str(missing_inputs) +' . Please ensure to not make this mistake again.',
                    jinx_name,
                    model=model,
                    provider=provider,
                    messages=messages,
                    npc=npc,
                    api_url=api_url,
                    api_key=api_key,
                    stream=stream,
                    attempt=attempt + 1,
                    n_attempts=n_attempts,
                    context=context
                )
            return {
                "output": f"Missing inputs for jinx '{jinx_name}': {missing_inputs}",
                "messages": messages,
            }

        
        render_markdown( "\n".join(['\n - ' + str(key) + ': ' +str(val) for key, val in input_values.items()]))

        try:
            jinx_output = jinx.execute(
                input_values,
                jinja_env,
                npc=npc,

                messages=messages,
            )
            if 'llm_response' in jinx_output and 'messages' in jinx_output:
                if len(jinx_output['llm_response'])>0:                
                    messages = jinx_output['messages']
        except Exception as e:
            print(f"An error occurred while executing the jinx: {e}")
            print(f"trying again, attempt {attempt+1}")
            print('command', command)
            if attempt < n_attempts:
                jinx_output = handle_jinx_call(
                    command,
                    jinx_name,
                    model=model,
                    provider=provider,
                    messages=messages,
                    npc=npc,
                    api_url=api_url,
                    api_key=api_key,
                    stream=stream,
                    attempt=attempt + 1,
                    n_attempts=n_attempts,
                    context=f""" \n \n \n "jinx failed: {e}  \n \n \n here was the previous attempt: {input_values}""",
                )
            else:
                if shell:
                    user_input = input(
                        "the jinx execution has failed after three tries, can you add more context to help or would you like to run again?"
                    )
                    return handle_jinx_call(
                        command + " " + user_input,
                        jinx_name,
                        model=model,
                        provider=provider,
                        messages=messages,
                        npc=npc,
                        api_url=api_url,
                        api_key=api_key,
                        stream=stream,
                        attempt=attempt + 1,
                        n_attempts=n_attempts,
                        context=context,
                    )
                else:
                    return {
                        "output": f"Jinx execution failed after {n_attempts} attempts: {e}",
                        "messages": messages,
                    }

        # process the jinx call
        #print(messages)
        if not stream and messages[-1]['role'] != 'assistant':
            # if the jinx has already added a message to the output from a final prompt we dont want to double that
            
            render_markdown(f""" ## jinx OUTPUT FROM CALLING {jinx_name} \n \n output:{jinx_output['output']}""" )

            
            response = get_llm_response(f"""
                The user had the following request: {command}. 
                Here were the jinx outputs from calling {jinx_name}: {jinx_output}
                
                Given the jinx outputs and the user request, please format a simple answer that 
                provides the answer without requiring the user to carry out any further steps.
                """,
                model=model,
                provider=provider,
                api_url=api_url,
                api_key=api_key,
                npc=npc,
                messages=messages,
                context=context, 
                stream=stream,
            )
            messages = response['messages']
            response = response.get("response", {})
            return {'messages':messages, 'output':response}
        return {'messages': messages, 'output': jinx_output['output']}


def handle_request_input(
    context: str,
    model: str ,
    provider: str 
):
    """
    Analyze text and decide what to request from the user
    """
    prompt = f"""
    Analyze the text:
    {context}
    and determine what additional input is needed.
    Return a JSON object with:
    {{
        "input_needed": boolean,
        "request_reason": string explaining why input is needed,
        "request_prompt": string to show user if input needed
    }}

    Do not include any additional markdown formatting or leading ```json tags. Your response
    must be a valid JSON object.
    """

    response = get_llm_response(
        prompt,
        model=model,
        provider=provider,
        messages=[],
        format="json",
    )

    result = response.get("response", {})
    if isinstance(result, str):
        result = json.loads(result)

    user_input = request_user_input(
        {"reason": result["request_reason"], "prompt": result["request_prompt"]},
    )
    return user_input


def check_llm_command(
    command: str,
    model: str = None, 
    provider: str  = None, 
    api_url: str = None,
    api_key: str = None,
    npc: Any = None,
    team: Any = None,
    messages: List[Dict[str, str]] = None,
    jinxs: List[Dict[str, str]] = None,
    tools = None,
    tool_map: Dict[str, str] = None,
    images: list = None,
    stream=False,
    context=None,    
    shell = False,
):
    """This function checks an LLM command.
    Args:
        command (str): The command to check.
    Keyword Args:
        model (str): The model to use for checking the command.
        provider (str): The provider to use for checking the command.
        npc (Any): The NPC object.
        messages (List[Dict[str, str]]): The message history.
        stream (bool): Whether to stream the response.
    Returns:
        Any: The result of checking the LLM command or a generator if stream=True.
    """
    if messages is None:
        messages = []
    
    prompt = f"""
    A user submitted this query: {command}
    """
    
    if tools:
        result = get_llm_response(
            prompt,
            model=model,
            provider=provider,
            api_url=api_url,
            api_key=api_key,
            npc=npc,
            messages=messages,
            tools=tools,
            tool_map=tool_map,
            context=None,
            stream=stream,
        )
        return {
            'messages': result.get('messages', messages),
            'output': result.get('response', '')
        }
            
    prompt += f"""
    Determine the nature of the user's request:

    1. Should a jinx be invoked to fulfill the request? A jinx is a jinja-template execution script.

    2. Is it a general question that requires an informative answer or a highly specific question that
        requires information on the web?

    """

        
    jinx_component = ''
    if npc is not None:
        if npc.shared_context:
            prompt += f"""
            Relevant shared context for the npc:
            {npc.shared_context}
            """

        if npc.jinxs_dict is None :
            jinx_component += "No NPC jinxs available. Do not invoke jinxs."
        else:
            jinx_component += "Available jinxs: \n"
            jinxs_set = {}
            if npc.jinxs_dict is not None:
                for jinx_name, jinx in npc.jinxs_dict.items():
                    if jinx_name not in jinxs_set:
                        jinxs_set[jinx_name] = jinx.description
            for jinx_name, jinx_description in jinxs_set.items():

                jinx_component += f"""
                            {jinx_name} : {jinx_description} \n

                            """
        #import pdb 
        #pdb.set_trace()
    if jinxs is not None:
        for jinx in jinxs:
            jinx_component += f"""
            Here is a jinx that may be relevant to the user's request:
            {jinx}
            """        

    if team is None:
        prompt += "No NPCs available for alternative answers."
    else:
        # team.npcs is a dict , need to check if it is empty
        #if team.npcs is None or len(team.npcs) == 0:
        #    prompt += "No NPCs available for alternative answers."
        #else:
        #    prompt += f"""
        #    Available NPCs for alternative answers:#
        #
        #        {team.npcs}
        #    """
        if team.context:
            prompt += f"""
            Relevant shared context for the team:
            {team.context}
            """
        if team.preferences is not None:
            prompt += f"""
            Relevant team preferences:
            {team.preferences}
            """

    action_space = ["invoke_jinx",
                     "answer_question", 
                     ]
    prompt += f"""
    These were the most recent 5 messages in the conversation which should help you respond appropriately:

    {messages[-5:] if len(messages) > 5 else messages}


    In considering how to answer this, consider:

    - Whether a jinx should be used.
    Use jinxs when it is obvious that the answer needs to be as up-to-date as possible. For example,
        a question about where mount everest is does not necessarily need to be answered by a jinx call or an agent pass.

    If a user asks to explain the plot of the aeneid, this can be answered without a jinx call or agent pass.
    
    If a user were to ask for the current weather in tokyo or the current price of bitcoin or who the mayor of a city is,
        then a jinx call is appropriate.

    If the user wants you to read a file, it must use a jinx to read the file.

    If the user asks you to edit a file, you must use a jinx to edit the file.

    If the user asks you to take a screenshot, you must to use a jinx to take the screenshot if available
    
    If a user asks you to search or to take a screenshot or to open a program or to write a program most likely it is
    appropriate to use a jinx. 

    Here are the jinxs that are available to you:
    {jinx_component}

    Respond with a JSON object containing:
    - "action": one of {action_space}
    - "jinx_name": : if action is "invoke_jinx": the name of the jinx to use.
    - "explanation": a brief explanation of why you chose this action.

    Remember, use a jinx if appropriate.

    Return only the JSON object. Do not include any additional text.

    The format of the JSON object is:
    {{
        "action": "invoke_jinx" | "answer_question"  ,
        "jinx_name": "<jinx_name(s)_if and only if the action is invoke_jinx>",
        "explanation": "<your_explanation>",

    }}

    In your explanation, do not needlessly reference the user's files or provided context. Simply provide the explanation for your choice in as few words as possible.

    Remember, the action must be one of "invoke_jinx" or "answer_question". Do not use a jinx's name for the action.     
    
    Remember, do not include ANY ADDITIONAL MARKDOWN FORMATTING.
    There should be no leading ```json.
    
        """
    if context:
        prompt += f"""
        Additional relevant context from user:

        {context}

        """
    
    action_response = get_llm_response(
        prompt,
        model=model,
        provider=provider,
        api_url=api_url,
        api_key=api_key,
        npc=npc,
        format="json",
        messages=[],
        context=None,
    )

    if "Error" in action_response:
        print(f"LLM Error: {action_response['error']}")
        return action_response["error"]

    response_content = action_response.get("response", {})

    if isinstance(response_content, str):
        try:
            response_content_parsed = json.loads(response_content)
        except json.JSONDecodeError as e:
            print(
                f"Invalid JSON received from LLM: {e}. Response was: {response_content}"
            )
            return f"Error: Invalid JSON from LLM: {response_content}"
    else:
        response_content_parsed = response_content

    #print(prompt)
    action = response_content_parsed.get("action").strip()
    explanation = response_content_parsed.get("explanation").strip()
    jinx_name = response_content_parsed.get('jinx_name', '')
    #print(response_content_parsed)
    jinx_name_print = '\n Jinx: ' + str(jinx_name) if jinx_name else ''

    render_markdown(f"- Action chosen: {action + jinx_name_print}\n")
    render_markdown(f"- Explanation given: {explanation}\n")

    command_with_action_response = f"""
    Here is the user question: {command}
    The action chosen by the agent is: {action}
    The explanation provided by the agent is: {explanation}
    The available jinxs were: {jinx_component}
    The jinx name chosen by the agent is: {jinx_name} (if invoke_jinx was used, otherwise this will be empty)

    Do not needlessly reference the user's files or provided context. Simply provide the answer to the user's question. Avoid
    appearing zany or unnecessarily forthcoming about the fact that you have received such information. You know it
    and the user knows it. there is no need to constantly mention the facts that are aware to both.
    
    """

    if action == "invoke_jinx":
        if npc and npc.jinxs_dict and jinx_name in npc.jinxs_dict:
            if stream and not shell:
                # Create wrapper generator for streaming case
                def decision_wrapped_gen():
                    # First yield decision
                    yield {'role': 'decision', 'content': f"- Action chosen: {action + jinx_name}\n- Explanation given: {explanation}\n"}
                    # Then execute jinx and yield from its result
                    result = handle_jinx_call(
                        command, 
                        jinx_name,
                        model=model,
                        provider=provider,
                        api_url=api_url, 
                        api_key=api_key,
                        messages=messages, 
                        npc=npc,
                        stream=stream
                    )
                    yield from result['output']
                return {'messages': messages, 'output': decision_wrapped_gen()}
            elif stream and shell:
                result  = handle_jinx_call(
                    command,
                    jinx_name, 
                    model=model, 
                    provider=provider, 
                    api_url=api_url, 
                    api_key=api_key, 
                    messages=messages, 
                    npc = npc ,
                    stream =stream
                )
                return {'messages': result.get('messages', messages), 'output': result.get('output', '')}
            else:
                # Non-streaming case
                result = handle_jinx_call(
                    command, jinx_name,
                    model=model, provider=provider,
                    api_url=api_url, api_key=api_key,
                    messages=messages, npc=npc,
                    stream=stream
                )
                return {'messages': result.get('messages', messages), 'output': result.get('output', '')}
        else:
            return {"messages": messages, "output": f"jinx '{jinx_name}' not found"}
    
    elif action == "answer_question":
        if stream and not shell:
            def decision_wrapped_gen():
                yield {'role': 'decision',
                        'content': f"- Action chosen: {action + jinx_name}\n- Explanation given: {explanation}\n"}
                result = get_llm_response(
                    command_with_action_response,
                    model=model,
                    provider=provider,
                    api_url=api_url, 
                    api_key=api_key,
                    messages=messages, 
                    npc=npc,
                    stream=stream, 
                    images=images,
                )
                yield from result['response']
            return {'messages': messages, 'output': decision_wrapped_gen()}
        elif stream and shell:
            result = get_llm_response(
                command_with_action_response,
                model=model, 
                provider=provider,
                api_url=api_url, 
                api_key=api_key,
                messages=messages,
                npc=npc,
                stream=stream, 
                images=images,
            )


            return {'messages': result.get('messages', messages), 'output': result.get('response', '')}


        else:
            result = get_llm_response(
                command_with_action_response,
                model=model, 
                provider=provider,
                api_url=api_url, 
                api_key=api_key,
                messages=messages, 
                npc=npc,
                stream=stream, 
                images=images,
            )
            return {'messages': result.get('messages', messages), 'output': result.get('response', '')}


    elif action == "request_input":
        explanation = response_content_parsed.get("explanation")
        request_input = handle_request_input(
            f"Explanation from check_llm_command:  {explanation} \n for the user input command: {command}",
            model=model,
            provider=provider,
        )
        
        messages.extend([
            {
                "role": "assistant",
                "content": f"""It's clear that extra input is required.
                             Could you please provide it? Here is the reason:
                             {explanation},
                             and the prompt: {command}"""
            },
            {
                "role": "user",
                "content": command + " \n \n \n extra context: " + request_input
            }
        ])
        
        if stream and not shell:
            def decision_wrapped_gen():
                yield {'role': 'decision', 'content': f"- Action chosen: {action + jinx_name}\n- Explanation given: {explanation}\n"}
                result = check_llm_command(
                    command_with_action_response + " \n \n \n extra context: " + request_input,
                    model=model, provider=provider,
                    api_url=api_url, api_key=api_key,
                    npc=npc, messages=messages,
                    stream=stream, 
                    shell = shell
                )
                yield from result['output']
            return {'messages': messages, 'output': decision_wrapped_gen()}
        elif stream and shell:
            return check_llm_command(
                    command_with_action_response + " \n \n \n extra context: " + request_input,
                    model=model, provider=provider,
                    api_url=api_url, api_key=api_key,
                    npc=npc, messages=messages,
                    stream=stream, shell = shell
                )

        else:
            return check_llm_command(
                command_with_action_response + " \n \n \n extra context: " + request_input,
                model=model, provider=provider,
                api_url=api_url, api_key=api_key,
                npc=npc, messages=messages,
                stream = stream,
                shell = shell
            )
    else:
        print("Error: Invalid action in LLM response")
        return {"messages": messages, "output": "Error: Invalid action in LLM response"}